/****************************************************************************
 **
 ** Copyright (C) 2001-2005 Tilo Riemer <riemer@lincvs.org> and
 **                         Frank Hemer <frank@hemer.org>
 **
 **
 **----------------------------------------------------------------------------
 **
 **----------------------------------------------------------------------------
 **
 ** LinCVS is available under two different licenses:
 **
 ** If LinCVS is linked against the GPLed version of Qt 
 ** LinCVS is released under the terms of GPL also.
 **
 ** If LinCVS is linked against a nonGPLed version of Qt 
 ** LinCVS is released under the terms of the 
 ** LinCVS License for non-Unix platforms (LLNU)
 **
 **
 ** LinCVS License for non-Unix platforms (LLNU):
 **
 ** Redistribution and use in binary form, without modification, 
 ** are permitted provided that the following conditions are met:
 **
 ** 1. Redistributions in binary form must reproduce the above copyright
 **    notice, this list of conditions and the following disclaimer in the
 **    documentation and/or other materials provided with the distribution.
 ** 2. It is not permitted to distribute the binary package under a name
 **    different than LinCVS.
 ** 3. The name of the authors may not be used to endorse or promote
 **    products derived from this software without specific prior written
 **    permission.
 ** 4. The source code is the creative property of the authors.
 **    Extensions and development under the terms of the Gnu Public License
 **    are limited to the Unix platform. Any distribution or compilation of 
 **    the source code against libraries licensed other than gpl requires 
 **    the written permission of the authors.
 **
 **
 ** THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR 
 ** IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 ** WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 ** ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY 
 ** DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
 ** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE 
 ** GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 ** INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 ** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
 ** NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
 ** SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 **
 **
 **
 ** LinCVS License for Unix platforms:
 **
 ** This program is free software; you can redistribute it and/or modify 
 ** it under the terms of the GNU General Public License as published by 
 ** the Free Software Foundation; either version 2 of the License, or 
 ** (at your  option) any later version.  This program is distributed in 
 ** the hope that it will be useful, but WITHOUT ANY WARRANTY; without 
 ** even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 ** PARTICULAR PURPOSE.
 **
 ** See the GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program; if not, write to the Free Software Foundation,
 ** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 **
 *****************************************************************************/

#include "History.h"

#include <qapplication.h>
#include <assert.h>

#include "globals.h"
#include "HistoryDialogImpl.h"

History * History::historyFile(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, false);
}

History * History::historyDir(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench)
{
   return setup(whatsThisIconSet, parent, workBench, true);
}

History * History::setup(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench, bool isDir) {

   DirBase * dir = workBench->selectedItem();
   if (!dir) return NULL;

   if (dir->getType() == DirBase::Cvs) {

      History * p = new History(whatsThisIconSet, parent, workBench, dir, isDir);
      p->acceptCvs();
      return p;

   } else {

      qDebug("History: Method not implemented");
      return NULL;
   }
}

History::History(const QIconSet &whatsThisIconSet, QWidget* parent, CvsDirListView * workBench, DirBase * dir, bool isDir)
   :  m_whatsThisIconSet(whatsThisIconSet),
      m_parent(parent),
      m_workBench(workBench),
      m_dir(dir),
      m_dlg(NULL),
      m_isDir(isDir),
      m_name(""),
      m_rev("")
{
   connect(this,SIGNAL(deleteObject(QObject *)),parent,SLOT(slot_deleteObject(QObject *)));
}

History::~History() {
   if (m_dlg) delete m_dlg;
   m_dlg = NULL;
}

void History::acceptCvs() {

   assert(m_dir);

   QString files = "";
   QString command = CvsOptions::cmprStr() + " history ";
   if (CvsOptions::g_historyPeriod != -1) {
      command +=
	 "-D " +
	 masqWs(QDateTime::currentDateTime().addDays(-CvsOptions::g_historyPeriod).toString("yyyy-MM-dd hh:mm")) +
	 " ";
   }
   command += "-e -a";

   if (m_isDir) {
      m_name = m_dir->repository();
      command += " -p " + masqWs(m_name);
   } else {
      QStringList fileList = m_dir->getSelectedFiles();
      if (fileList.isEmpty()) {
         reject();
         return;
      }
      m_name = masqWs(fileList.join("\" \""));
      if (fileList.count() == 1) {
         QString tmp = fileList[0];
         m_rev = m_dir->getRev(&tmp);
      }
      masqQuoteMarks(&fileList);
      files = masqWs(fileList.join("\" \""));
   }

   QString dir = m_dir->fullName();
   QString topModule = m_dir->topControlledDir()->relativeName();
   
   callInteractive( topModule, dir, command,
	 files, CVS_HISTORY_CMD,
	 ExtApps::g_cvsRsh.path,  //additional options of cvsRsh not supported yet
	 NOROOT);
}

void History::reject() {
   emit deleteObject(this);
}

void History::cvsCallStarted() {
   QApplication::setOverrideCursor(Qt::waitCursor);
}

void History::cvsCallFinished() {
   QApplication::restoreOverrideCursor();
}

void History::afterCall(int cmd,CvsBuffer * buf,bool failed) {
   cvsCallFinished();
   if (failed) {
      reject();
      return;
   }
   assert(m_dir);
   switch( cmd) {
      case CVS_HISTORY_CMD: {
	 HistoryDialogImpl * dlg = new HistoryDialogImpl( m_whatsThisIconSet,
	       m_parent,
	       LookAndFeel::g_nonModalF | WDestructiveClose);
	 connect( dlg, SIGNAL(editFile(QString)), m_parent, SLOT(viewFileWithPath(QString)));
	 dlg->parseHistory(buf, m_dir->repository(), (QDir*)m_dir->topControlledDir(),m_name,m_rev);
	 dlg->show();
	 break;
      }
      default: {
	 qDebug("History::afterCall: Unknown cmd");
      }
   }
   reject();
}
