package com.limegroup.gnutella.update;

import com.limegroup.gnutella.gui.GUIMediator;
import com.limegroup.gnutella.update.gui.UpdateCallback;
import com.limegroup.gnutella.Assert;
import java.io.*;
import java.net.*;

/**
 * This class handles the downloading of files from urls.
 */
//2345678|012345678|012345678|012345678|012345678|012345678|012345678|012345678|
final class FileDownloader {

	/**
	 * flag for whether or not the downlaod should be stopped
	 * this is volatile because multiple threads will likely
	 * modify it and should not use their own copies.
	 */
	private volatile boolean _stopDownload = false;

	/**
	 * set the flag for stopping the download to true.
	 */
	synchronized void stopDownload() {
		_stopDownload = true;
	}

	/**
	 * downloads the file specified in the url parameter to the 
	 * path specified in the path parameter.
	 *
	 * @param urlString the url to download the file from.
	 * @param newFilePath the full path for the newly downloaded file
	 *  on the local system.
	 * @exception throws a FileDownloadException if something goes wrong.
	 */
	void downloadFile(String urlString, String newFilePath) 
		throws FileDownloadException {
		Assert.that((urlString != null) &&
					(!urlString.equals("")), "Invalid Download URL");
		Assert.that((newFilePath != null) &&
					(!newFilePath.equals("")), "Invalid Download Path");
		
		// create a file object so that we can delete any old
		// partially downloaded file that may still be present
		File newFile = new File(newFilePath);
		newFile.delete();
		
		URL url;
		int updateSize;
		URLConnection urlConn = null;
		InputStream is        = null;
		FileOutputStream fos  = null;
		try {
			url = new URL(urlString);
			urlConn = url.openConnection();
			updateSize = urlConn.getContentLength();
			
			// if the update size is less than zero, the file was 
			// not found.  display an error and exit.
			if(updateSize == -1) {
				// we get the string resource here because the error message method
				// we are using takes a hard coded value
				String msg = 
				    GUIMediator.getStringResource("UPDATE_ERROR_ACCESSING_FILE");
				throw new FileDownloadException(msg);
			}

			// tell the gui component what the size of the download
			// will be so it can calculate the percent complete
			UpdateCallback.setDownloadSize(updateSize);

			is = urlConn.getInputStream();
			
			fos = new FileOutputStream(newFilePath, false);				

			// we have established the connection with the server, so 
			// change the status in the progress window.
			UpdateCallback.setDownloadFile(urlString);

			// variable for the total number of bytes read
			int amountRead = 0;
			
			// variable for the number of new bytes read on this iteration.
			int newBytes = -1;

			// a buffer to read bytes into.
			byte[] buf = new byte[1024];
			while(true) {
				if(_stopDownload) {
					// delete the partially downloaded file so that
					// it does not get in the way of future updates.
					File partialFile = new File(newFilePath);
					partialFile.delete();
					return;
				}
				
				// update the percentage read in the progress window
				UpdateCallback.updateProgressWindow(amountRead);

				// break out of the loop if we've downloaded everything
				if(amountRead >= updateSize) break;

				// read in the next set of bytes into the buffer
				newBytes = is.read(buf);

				// break out of the loop if there are no new bytes
				if(newBytes == -1) break;
				
				// write the new bytes to the file
				fos.write(buf, 0, newBytes);

				// increase the amount read by the new number of
				// bytes read
				amountRead += newBytes;
			}		   
		} catch(IOException ioe) { 
			// we get the string resource here because the error message method
			// we are using takes a hard coded value
			String msg = 
			    GUIMediator.getStringResource("UPDATE_ERROR_COMMUNICATING_WITH_SERVER");
			throw new FileDownloadException(msg);
		} finally {
			// close all of the streams if they exist.
			try {
				if(is != null) is.close();

				// close the FileOutputStream
				if(fos != null) {
					fos.flush();
					fos.close();	
				}
			} catch(IOException ioee) {
			}
			// close the urlConnection
			if(urlConn instanceof HttpURLConnection) 
				((HttpURLConnection)urlConn).disconnect();		
		}
	}
}





