package com.limegroup.gnutella.update;

import java.net.URLEncoder;
import com.limegroup.gnutella.util.URLDecoder;
import java.util.StringTokenizer;
//import java.io.IOException;

/**
 * Handles the client-side representation of an 
 * <tt>AbstractRemoteUpdateInfo</tt> object, which reconstructs the data
 * for the update created on the server side.  This parses that 
 * information and creates the appropriate <tt>Updator</tt>
 * object based on the data.
 */
//2345678|012345678|012345678|012345678|012345678|012345678|012345678|012345678|
public final class ClientRemoteUpdateInfo extends AbstractRemoteUpdateInfo {

    /** 
	 * Updator object that will perform the final update.  This is initially
	 * set the be a <tt>NoUpdateUpdator</tt> by default.
	 */
    private Updator _updator = new NoUpdateUpdator();
   
    /**
     * Constructor taking no arguments for use in constructing the
     * <tt>AbstractRemoteUpdateInfo</tt> object on the client side.
     */
    public ClientRemoteUpdateInfo() {}
    
	/**
	 * Adds one key/value pair of update information.<p> 
     * 
     * This method is used on the client side to reconstruct the 
     * <tt>RemoteUpdateInfo</tt> object originally created on the server.
	 *
	 * @param infoLine string containing one name-value pair
	 *                 from the server
     * @throws IOException if the data supplied in the <tt>infoLine</tt>
     *  argument could not be successfully parsed for some reason
	 */
	final void addRemoteInfo(final String infoLine) {
        StringTokenizer st = new StringTokenizer(infoLine, "&");
        while(st.hasMoreTokens()) {
            handleKeyValuePair(st.nextToken());
        }
    }
    
	/**
	 * Returns the <tt>Updator</tt> object that this class has created.
	 *
	 * @return the <tt>Updator</tt> instance that will perform
	 *         the type of update specified by the servlet
	 */
	final Updator getUpdator() {
		return _updator;
	}

	/**
	 * Parses the passed in String and sets the appropriate values 
	 * for the <tt>Updator</tt> object.<p> 
     * 
     * This method is used on the client side to reconstruct the 
     * <tt>RemoteUpdateInfo</tt> object originally created on the server.
	 *
	 * @param line a string representing one key-value pair of
	 *             information for the update
	 */
    private final void handleKeyValuePair(final String line) {
		if(line.startsWith(DIRECTIVE)) {
			setUpdator(line);
		} else if(line.startsWith(MESSAGE_TO_DISPLAY)) {
            String value = getUpdateValue(line);
			// set the message to display to the user
			_updator.setMessageToDisplay(value);
		} else if(line.startsWith(UPDATE_URL)) {
            String value = getUpdateValue(line);
			// set the url to use for the update
			_updator.setUpdateURL(value);
		} else if(line.startsWith(UPDATE_CLASSPATH)) {
            String value = getUpdateValue(line);
			// set the classpath that the local machine should use
			_updator.setClassPath(value);
		} else if(line.startsWith(UPDATE_MAIN_CLASS)) {
            String value = getUpdateValue(line);
			// set the main class that the local machine should load.
			_updator.setMainClass(value);
		} else {
			// something random -- don't worry about it
        }
	}

	/**
	 * Returns the value associated with the key/value pair contained
	 * in the <tt>line</tt> argument.<p> 
     *
     * This method is used on the client side to reconstruct the 
     * <tt>RemoteUpdateInfo</tt> object originally created on the server. 
	 *
	 * @param line the line of text containing a key/value pair
	 */
	private final String getUpdateValue(String line) {
		// set the index to be after the last "=" value
		int index = line.lastIndexOf("=")+1;
		line = URLDecoder.decode(line.substring(index));
		
		// this will get rid of any trailing spaces or
		// carriage returns.
		return line.trim();
	}	

	/**
	 * Determines the type of update we should perform based
	 * on the information received from the server and constructs
	 * the appropriate update object.<p>  
     *
     * This method is used on the client side to reconstruct the 
     * <tt>RemoteUpdateInfo</tt> object originally created on the server. 
	 *
	 * @param str the <tt>String</tt> specifying the type of update
	 *            to perform
	 */
	private final void setUpdator(String str) {
		int index = str.lastIndexOf("=")+1;
		String directiveValue = URLDecoder.decode(str.substring(index));
		if(directiveValue.startsWith(NO_UPDATE_STRING)) {
			// create an updator object that does nothing
			_updator = new NoUpdateUpdator();
		} else if(directiveValue.startsWith(UPDATE_JAR_STRING)) {
			// create an updator for updating the jar
			_updator = new MultipleFileUpdator();
		} else if(directiveValue.startsWith(OPEN_WEB_PAGE_STRING)) {
			// create an updator for opening a web page
			_updator = new WebPageUpdator();
		} else if(directiveValue.startsWith(DISPLAY_MESSAGE_STRING)) {
			// create an updator for displaying a message
			_updator = new DisplayMessageUpdator();
		} 
	}

	// overrides Object.toString
	public String toString() {
		return "ClientRemoteUpdateInfo::Updator: "+_updator.toString();
	}
}
