package com.limegroup.gnutella.gui.tables;

import com.limegroup.gnutella.gui.*;
import javax.swing.border.*;
import javax.swing.*;
import java.awt.*;
import javax.swing.table.*;
import java.net.URL;

/**
 * Renders the column in the search window that displays an icon for
 * whether or not the host returning the result is chattable.
 */
//2345678|012345678|012345678|012345678|012345678|012345678|012345678|012345678|
public final class ChatRenderer extends JLabel 
	implements TableCellRenderer, ThemeObserver {

    private static Border _noFocusBorder = new EmptyBorder(1, 1, 1, 1); 

    // We need a place to store the color the JLabel should be returned 
    // to after its foreground and background colors have been set 
    // to the selection background color. 
    // These ivars will be made protected when their names are finalized. 
    private Color _unselectedForeground; 
    private Color _unselectedBackground; 

	/**
	 * Constant <tt>Icon</tt> for chatability.
	 */
	private static Icon _chatIcon = 
		GUIMediator.getThemeImage("chat.gif");

	
	/**
	 * The constructor sets this <tt>JLabel</tt> to be opaque and sets the
	 * border.
	 */
	public ChatRenderer() {
		setOpaque(true);
        setBorder(_noFocusBorder);
		setHorizontalAlignment(SwingConstants.CENTER);
	}

	// inherit doc comment
	public void updateTheme() {
		_chatIcon = GUIMediator.getThemeImage("chat.gif");
	}

    /**
     * Overrides <tt>JComponent.setForeground</tt> to assign
     * the unselected-foreground color to the specified color.
     * 
     * @param c set the foreground color to this value
     */
    public void setForeground(Color c) {
        super.setForeground(c); 
        _unselectedForeground = c; 
    }
    
    /**
     * Overrides <tt>JComponent.setForeground</tt> to assign
     * the unselected-background color to the specified color.
     *
     * @param c set the background color to this value
     */
    public void setBackground(Color c) {
        super.setBackground(c); 
        _unselectedBackground = c; 
    }

    /**
     * Notification from the <tt>UIManager</tt> that the look and feel
     * [L&F] has changed.
     * Replaces the current UI object with the latest version from the 
     * <tt>UIManager</tt>.
     *
     * @see JComponent#updateUI
     */
    public void updateUI() {
        super.updateUI(); 
		setForeground(null);
		setBackground(null);
    }

	/**
	 * Returns the <tt>Component</tt> that displays the stars based
	 * on the number of stars in the <tt>QualityHolder</tt> object.
	 */
	public Component getTableCellRendererComponent
		(JTable table,Object value,boolean isSelected,
		 boolean hasFocus,int row,int column) {
		
		if(((Boolean)value).booleanValue()) {
			setIcon(_chatIcon);
		}
		else {
			setIcon(null);
		}


		if (isSelected) {
			super.setForeground(table.getSelectionForeground());
			super.setBackground(table.getSelectionBackground());
		}
		else {
			super.setForeground((_unselectedForeground != null) ? 
								_unselectedForeground 
								: table.getForeground());
			super.setBackground((_unselectedBackground != null) ? 
								_unselectedBackground 
								: table.getBackground());
		}
		
		
		if (hasFocus) {
			super.setBorder( UIManager.getBorder("Table.focusCellHighlightBorder") );
	    if (table.isCellEditable(row, column)) {
	        super.setForeground( UIManager.getColor("Table.focusCellForeground") );
	        super.setBackground( UIManager.getColor("Table.focusCellBackground") );
	    }
		} else {
			super.setBorder(_noFocusBorder);
		}				
		
		return this;	
	}
	

		
    /*
     * The following methods are overridden as a performance measure to 
     * to prune code-paths are often called in the case of renders
     * but which we know are unnecessary.  Great care should be taken
     * when writing your own renderer to weigh the benefits and 
     * drawbacks of overriding methods like these.
     */

    /**
     * Overridden for performance reasons.
     * See the <a href="#override">Implementation Note</a> 
     * for more information.
     */
    public boolean isOpaque() { 
		Color back = getBackground();
	Component p = getParent(); 
	if (p != null) { 
	    p = p.getParent(); 
	}
	JComponent jp = (JComponent)p;
	// p should now be the JTable. 
	boolean colorMatch = (back != null) && (p != null) && 
	    back.equals(p.getBackground()) && 
			jp.isOpaque();
	return !colorMatch && super.isOpaque(); 
    }
		
    /**
     * Overridden for performance reasons.
     * See the <a href="#override">Implementation Note</a> 
     * for more information.
     */
    public void validate() {}

    /**
     * Overridden for performance reasons.
     * See the <a href="#override">Implementation Note</a> 
     * for more information.
     */
    public void revalidate() {}

    /**
     * Overridden for performance reasons.
     * See the <a href="#override">Implementation Note</a> 
     * for more information.
     */
    public void repaint(long tm, int x, int y, int width, int height) {}

    /**
     * Overridden for performance reasons.
     * See the <a href="#override">Implementation Note</a> 
     * for more information.
     */
    public void repaint(Rectangle r) { }

    /**
     * Overridden for performance reasons.
     * See the <a href="#override">Implementation Note</a> 
     * for more information.
     */
    protected void firePropertyChange(String propertyName, Object oldValue, Object newValue) {	
	// Strings get interned...
	if (propertyName=="text") {
	    super.firePropertyChange(propertyName, oldValue, newValue);
	}
	}

    /**
     * Overridden for performance reasons.
     * See the <a href="#override">Implementation Note</a> 
     * for more information.
     */
    public void firePropertyChange(String propertyName, boolean oldValue, boolean newValue) { }	

}
