package com.limegroup.gnutella.gui.search;

import java.util.Properties;
import com.sun.java.util.collections.*;
import java.io.*;
import com.limegroup.gnutella.xml.*;
import com.limegroup.gnutella.*;
import com.limegroup.gnutella.gui.*;
import com.limegroup.gnutella.util.*;

/**
 * The class that handles all the display details.
 * <p>
 * Has a singleton pattern,
 * <p>
 * When instantiated, loads up the display properties for each schema, and 
 * stores the URIs against the properties in a HashMap.
 * <p>
 * This class also has the responsibility of remembering which columns are 
 * visible in the search window and the width of each column
 * @author Sumeet Thadani
 */
public class DisplayManager {
    
    private static DisplayManager instance;
    /**field rawName -> width
     * the rawName is the canonicalized field name if the field belongs to a
     * schema, otherwise we have assigned a "special rawName" for ordinary
     * fields. See method getKeySting for non schema fields' rawNames
     */
    private Properties props;
    /** The name of the file where we want to store display.props */
    private final String DISP_FILE = "Display.props";

    private HashMap/* String -> Properties*/ schemaURIToDispProps;

    ///////////////////IMPORTANT NOTE//////////////////////////////////
    //If you change these you must change static methods in this class
    //a .getKeyString() 
    //b. getDefualtWidth()
    ///////////////////////////////////////////////////////////////////
    static final int QUALITY_COL=0;
    static final int GROUP_COUNT = 1;
    static final int NAME_COL=2;
    static final int EXTENSION_COL=3;
    static final int SIZE_COL=4;
    static final int SPEED_COL=5;
	static final int CHAT_COL=6;
    static final int LOCATION_COL=7;
    static final int VENDOR_COL=8;

    /**private constructor
     * <p>
     * loads up all the display properties for the schemas and populates the 
     * schemaURIToDispProps map
     * <p>
     * Next it tries to load up Display.props which contains the list
     * of columns the user has chosen to see, with prefererd widths
     */
    private DisplayManager() {
        schemaURIToDispProps = new HashMap();
        LimeXMLSchemaRepository rep = LimeXMLSchemaRepository.instance();
        String[] schemas = rep.getAvailableSchemaURIs();
        for(int i=0; i<schemas.length; i++) {
            Properties properties = new Properties();
            loadDisplayProperties(schemas[i], properties);
            schemaURIToDispProps.put(schemas[i],properties);
        }
        props = new Properties();
        FileInputStream fis = null;
        try {
            fis = new FileInputStream(new File(CommonUtils.getUserSettingsDir(),DISP_FILE));
            props.load(fis);
			if(props.size() == 0) //bad Display.props w/o any excpetions?
				loadDefaults();
        } catch(IOException e){
            loadDefaults();
        } finally {
            if(fis != null) {
                try {
                    fis.close();
                } catch(IOException e) {
                    // nothing to do
                }
            }
        }
    }

    //singleton accessor
    public static DisplayManager instance(){
        if(instance==null) {
            instance =  new DisplayManager();
            return instance;
        }
        else
            return instance;
    }
    
    public void resetPropsFile() {
        new File(CommonUtils.getUserSettingsDir(),DISP_FILE).delete();
        loadDefaults();       
    }

    /**
     *@return for give URI returns Properties, which was loaded up at startup
     * If Properties was not found, returns a empty properties
     */
    public Properties getProperties(String schemaURI){
        Properties retProps = (Properties)schemaURIToDispProps.get(schemaURI);
        if(retProps == null)
            return new Properties();
        return retProps;
    }

    /**
     * For given schemaURI and rawFieldName returns the Display name. Either
     * by looking up the properties or, by proecessing the rawName
     */
    public String getDisplayName(String rawName, String schemaURI){
        Properties properties = getProperties(schemaURI);
        String dispName = properties.getProperty(rawName,"null");
        if(dispName.equals("null")) {//no property found...use other method
            dispName = LimeXMLUtils.processColName(rawName);
            return dispName;
        }
        dispName = stripPreferredSize(dispName);
        return dispName;
    }

    /**
     * called when a column is added. Puts that in the props, and commits it
     */
    public void addColumn(String rawName, int prefWidth){
        props.put(rawName, ""+prefWidth);
        commit();
    }
    
    public void setValue(String rawName, int prefWidth){
        props.put(rawName,""+prefWidth);
    }
    
    /**
     * called when column is removed. Puts it in the props, and commits it
     */ 
    public void removeColumn(String rawName){
        props.remove(rawName);
        commit();
    }

    /**
     * returns the preferred width of a column, if the column is not found in
     * the props, we return a width of -1
     * @param key if normal column the key has a special value stored in 
     * ResultPanel, otherwise, its the raw name of the column
     */
    public String getValue(String key) {
        return props.getProperty(key,"-1");//defualt column width 0
    }

    /**
     * Writes the props out to disk
     */
    public void commit(){
        FileOutputStream fos = null;
        try{
            fos = 
                new FileOutputStream(new File(CommonUtils.getUserSettingsDir(),DISP_FILE));
            props.save(fos,"");
        } catch(IOException e) {
            // this shouldn't happen -- we want to know if it does
            GUIMediator.showInternalError(e);
        } finally {
            if(fos != null) {
                try {
                    fos.close();
                } catch(IOException e) {
                    // nothing to do
                }
            }
        }
    }

    /////////////////////Non-schema fields specific methods///////////////

    /**
     * @return The "raw name" for columns that do not belong to any schema
     * this is based on the value of the column index
     */
    public static String getKeyString(int val) {
        switch(val){
        case QUALITY_COL: return "QUALITY";
        case GROUP_COUNT: return "COUNT";
        case NAME_COL: return "NAME";
        case EXTENSION_COL: return "TYPE";
        case SIZE_COL: return "SIZE";
        case SPEED_COL: return "SPEED";
        case CHAT_COL: return "CHAT";
        case LOCATION_COL: return "LOCATION";
        case VENDOR_COL: return "VENDOR";
        default:
            Assert.that(false, "Cannot get column value greater than 8");
        }
        return "";
    }
   
    /**
     * @return the preferred default widths for non-schema columns. 
     * The equivalent for schema columns are stored in their property files
     * (separated by a ^)
     */
    public static int getDefaultWidth(int val){
        switch(val){
        case QUALITY_COL: return 40;  //for quality
        case GROUP_COUNT: return 5;  //for count
        case NAME_COL: return 230; //for Name
        case EXTENSION_COL: return 30;  //for type
        case SIZE_COL: return 60;  //for size
        case SPEED_COL: return 30;  //for speed
        case CHAT_COL: return 10;  //for chat
        case LOCATION_COL: return 20;  //for location
        case VENDOR_COL: return 40;
        default:
            Assert.that(false, "Cannot have greater column value than 8");
        }
        return 0;
    }
    
    
    /////////////////////////private methods///////////////////////////
    
    //Utility method to that returns the DisplayName of a field without the
    //preferred with (which is usually ^ separated)
    public static String stripPreferredSize(String rawName){
        int i = rawName.indexOf("^");
        if(i < 0)
            return rawName;
        return rawName.substring(0,i);
    }
    
    //helper method to load properties.
    private void loadDisplayProperties(String uri, Properties props){
        String dispStr = LimeXMLSchema.getDisplayString(uri);
        LimeXMLProperties p = LimeXMLProperties.instance();        
        String dir = p.getXMLDisplayPropsDir();
        File f = new File(dir, dispStr+".props");
        FileInputStream fis = null;
        try{
            fis = new FileInputStream(f);
            props.load(fis);
        } catch(IOException ignored){
        } finally {
            if(fis != null) {
                try {
                    fis.close();
                } catch(IOException e) {
                    // nothing to do
                }
            }            
        }
    }


    //returns defualt preferres column sizes for normal columns
    private void loadDefaults(){
        props.put("QUALITY", "40");
        props.put("COUNT","3");
        props.put("NAME","230");
        props.put("TYPE","30");
        props.put("SIZE","60");
        props.put("SPEED","30");
        props.put("CHAT","10");
        props.put("audios__audio__bitrate__","20");
        //props.put("audios__audio__genre__","30");
        commit();
    }

}//end of class
