package com.limegroup.gnutella.gui.options.panes;

import com.limegroup.gnutella.gui.*;
import com.limegroup.gnutella.util.CommonUtils;
import com.limegroup.gnutella.settings.*;
import java.awt.event.*;
import java.awt.*;
import javax.swing.*;
import javax.swing.filechooser.*;
import java.io.File;
import java.io.IOException;

/**
 * This class defines the panel in the options window that allows the user
 * to change the theme used for display.
 */
//2345678|012345678|012345678|012345678|012345678|012345678|012345678|012345678|
public final class ThemePaneItem extends AbstractPaneItem {

    /**
     * Constant for the key of the locale-specific <code>String</code> for the
     * label on the component that allows to user to change the setting for
     * this <tt>PaneItem</tt>.
     */
    private final String OPTION_LABEL = "OPTIONS_THEME_BOX_LABEL";

    /**
     * Handle to the <tt>JTextField</tt> that displays the theme file.
     */
    private JTextField _themeField;

    /**
     * The constructor constructs all of the elements of this
     * <tt>AbstractPaneItem</tt>.  This includes the row of buttons that
     * allow the user to select the theme file.
     *
     * @param key the key for this <tt>AbstractPaneItem</tt> that the
     *            superclass uses to generate strings
     */
    public ThemePaneItem(final String key) {
        super(key);

        _themeField = new SizedTextField();
        LabeledComponent comp =
            new LabeledComponent(OPTION_LABEL, _themeField,
                                 LabeledComponent.TOP_LEFT);

        String[] labelKeys = {
            "GENERAL_BROWSE_BUTTON_LABEL",
            "GENERAL_DEFAULT_BUTTON_LABEL"
        };
        String[] toolTipKeys = {
            "OPTIONS_THEME_BROWSE_BUTTON_TIP",
            "OPTIONS_THEME_DEFAULT_BUTTON_TIP"
        };
        ActionListener[] listeners = {
            new SelectThemeFileListener(),
            new DefaultListener()
        };
        ButtonRow br = new ButtonRow(labelKeys, toolTipKeys, listeners,
                                     ButtonRow.X_AXIS, ButtonRow.LEFT_GLUE);

        add(comp.getComponent());
        add(getVerticalSeparator());
        add(br);
    }

    /**
     * This listener responds to the selection of the default theme
     * file and sets the current theme file to be the default.
     */
    private class DefaultListener implements ActionListener {
        public void actionPerformed(ActionEvent e) {
            _themeField.setText(
                ThemeSettings.THEME_DEFAULT.getValue().getAbsolutePath());
        }
    }

    /**
     * This listener displays a <tt>JFileChooser</tt> to the user,
     * allowing the user to select the theme file.
     */
    private class SelectThemeFileListener implements ActionListener {
        public void actionPerformed(ActionEvent e) {
            File file = null;
            String key = "OPTIONS_THEME_FILE_CHOOSER_TITLE";
            if (!CommonUtils.isMacOSX() || !ThemeSettings.isDefaultTheme()) {
                file = FileChooserHandler.getInputFile(
                    MEDIATOR.getMainOptionsComponent(),
                    key,
                    new File(CommonUtils.getUserSettingsDir(), "themes"),
                    new ThemeFileFilter());
                        
            } else {
                  FileDialog dialog =
                      new FileDialog(GUIMediator.getAppFrame(),
                                     GUIMediator.getStringResource(key),
                                     FileDialog.LOAD);
                  dialog.setVisible(true);
                  String dirStr  = dialog.getDirectory();
                  String fileStr = dialog.getFile();
                  if ((dirStr == null) || (fileStr == null)) return;
                  file = new File(dirStr, fileStr);
            }
            if (file == null) return;              
            if (!file.isFile()) return;
            String newFile = file.getAbsolutePath();
            _themeField.setText(newFile);
        }
    }

    /**
     * <tt>FileFilter</tt> class for only displaying theme file types in
     * the directory chooser.
     */
    private static class ThemeFileFilter extends FileFilter {
        public boolean accept(File f) {
            if (f.isDirectory()) return true;
            String name = f.getName();
            
            // don't allow anything that isn't a theme file
            if(!name.endsWith(ThemeSettings.EXTENSION))
                return false;
                
            // only allow the osx theme if we're on osx.
            if(!CommonUtils.isMacOSX() && 
              name.equals(ThemeSettings.DEFAULT_OSX_THEME_NAME))
                return false;
                
            // only allow the windows theme if we're on windows.
            if(!CommonUtils.isWindows() &&
              name.equals(ThemeSettings.WINDOWS_LAF_THEME_NAME))
                return false;
                
            // everything's okay -- allow it.                
            return true;
        }
        
        public String getDescription() {
            return GUIMediator.getStringResource("THEME_FILE_DESCRIPTION");
        }
    }

    /**
     * Defines the abstract method in <tt>AbstractPaneItem</tt>.<p>
     *
     * Sets the options for the fields in this <tt>PaneItem</tt> when the 
     * window is shown.
     */
    public void initOptions() {
        _themeField.setText(
            ThemeSettings.THEME_FILE.getValue().getAbsolutePath());
    }

    /**
     * Defines the abstract method in <tt>AbstractPaneItem</tt>.<p>
     *
     * Applies the options currently set in this window, displaying an
     * error message to the user if a setting could not be applied.
     *
     * @throws IOException if the options could not be applied for some reason
     */
    public void applyOptions() throws IOException {
        final String theme = _themeField.getText();

        File themeFile = new File(theme);
        if (!themeFile.isFile()) {
            GUIMediator.showError("ERROR_INVALID_THEME_FILE");
            throw new IOException("theme file is not a valid file");
        }

        if (!ThemeSettings.THEME_FILE.getValue().equals(themeFile)) {
            ThemeSettings.THEME_FILE.setValue(themeFile);
            GUIMediator.showMessage("THEME_CHANGED_MESSAGE",
                    QuestionsHandler.THEME_CHANGED);
            //ThemeFileHandler.reload();
            //GUIMediator.instance().setCurrentTheme(themeFile);
        }
    }
}
