package com.limegroup.gnutella.gui.options.panes;

import com.limegroup.gnutella.gui.*;
import com.limegroup.gnutella.settings.*;
import com.limegroup.gnutella.util.CommonUtils;
import java.awt.event.*;
import java.awt.*;
import javax.swing.*;
import java.io.*;

/**
 * This class defines the panel in the options window that allows the user
 * to change the directory for saving files.
 */
//2345678|012345678|012345678|012345678|012345678|012345678|012345678|012345678|
public final class SaveDirPaneItem extends AbstractPaneItem {

	/**
	 * Constant for the key of the locale-specific <code>String</code> for the 
	 * label on the component that allows to user to change the setting for
	 * this <tt>PaneItem</tt>.
	 */
	private final String OPTION_LABEL = "OPTIONS_SAVE_DIR_BOX_LABEL";

	/**
	 * Handle to the <tt>JTextField</tt> that displays the save directory.
	 */
	private JTextField _saveField;

	/**
	 * String for storing the initial save directory.
	 */
    private String _saveDirectory;


	/**
	 * The constructor constructs all of the elements of this 
	 * <tt>AbstractPaneItem</tt>.  This includes the row of buttons that
	 * allow the user to select the save directory.
	 *
	 * @param key the key for this <tt>AbstractPaneItem</tt> that the
	 *            superclass uses to generate strings
	 */
	public SaveDirPaneItem(final String key) {
		super(key);
		_saveField = new SizedTextField();
		LabeledComponent comp = 
		    new LabeledComponent(OPTION_LABEL, _saveField, 
								 LabeledComponent.TOP_LEFT);

		String[] labelKeys = {
			"OPTIONS_SAVE_DIR_BROWSE_BUTTON_LABEL",
			"OPTIONS_SAVE_DIR_DEFAULT_BUTTON_LABEL"
		};

		String[] toolTipKeys = {
			"OPTIONS_SAVE_DIR_BROWSE_BUTTON_TIP",
			"OPTIONS_SAVE_DIR_DEFAULT_BUTTON_TIP"
		};
        
		ActionListener[] listeners = {
			new SelectSaveDirectoryListener(),
			new DefaultListener()
		};

		ButtonRow br = new ButtonRow(labelKeys, toolTipKeys, listeners, 
									 ButtonRow.X_AXIS, ButtonRow.LEFT_GLUE);

		add(comp.getComponent());
		add(getVerticalSeparator());		
		add(br);				
	}

	/**
	 * This listener responds to the selection of the default savee 
	 * directory and sets the current save directory to be the default.
	 */
    private class DefaultListener implements ActionListener {
        public void actionPerformed(ActionEvent e) {
            _saveField.setText(SharingSettings.DEFAULT_SAVE_DIR.getAbsolutePath());
        }
    }

	/**
	 * This listener displays a <tt>JFileChooser</tt> to the user,
	 * allowing the user to select the save directory.
	 */
    private class SelectSaveDirectoryListener implements ActionListener {
        public void actionPerformed(ActionEvent e) {
			File dir = 
				FileChooserHandler.getInputDirectory(MEDIATOR.getMainOptionsComponent());
            if(dir == null) return;  			
  			if(!dir.isDirectory()) return;
  			String newDir = "";
		    try {
			    newDir = dir.getCanonicalPath();
			    if(!newDir.equals(_saveDirectory)) {
				    MEDIATOR.setSaveDirectoryChanged(true);
				    _saveField.setText(newDir);
			    }
			} catch(IOException ioe) {
  			}
        }
    }

	/**
	 * Defines the abstract method in <tt>AbstractPaneItem</tt>.<p>
	 *
	 * Sets the options for the fields in this <tt>PaneItem</tt> when the 
	 * window is shown.
	 */
	public void initOptions() {
		try {
            File file = SharingSettings.getSaveDirectory();
            
            if (file == null) {
                throw (new FileNotFoundException());
            }
            
			_saveDirectory = file.getCanonicalPath();
		} catch(FileNotFoundException fnfe) {
			// simply use the empty string if we could not get the save
			// directory.
			_saveDirectory = "";
		} catch(IOException ioe) {
			_saveDirectory = "";
		}
        _saveField.setText(_saveDirectory);
	}

	/**
	 * Defines the abstract method in <tt>AbstractPaneItem</tt>.<p>
	 *
	 * Applies the options currently set in this window, displaying an
	 * error message to the user if a setting could not be applied.
	 *
	 * @throws IOException if the options could not be applied for some reason
	 */
	public void applyOptions() throws IOException {
        final String save = _saveField.getText();
        if(!save.equals(_saveDirectory)) {
			MEDIATOR.setSaveDirectoryChanged(true);
            try {
				File saveDir = new File(save);
                
                if(!saveDir.isDirectory()) {
                    if(!saveDir.mkdirs()) throw new IOException();
                }
        
                //SharingSettings.DIRECTORY_FOR_SAVING_FILES.setValue(saveDir);
                SharingSettings.setSaveDirectory(saveDir); /* updates Incomplete directory etc... */
                
                _saveDirectory = save;
            }
            catch(IOException ioe) {
                GUIMediator.showError("ERROR_INVALID_SAVE_DIRECTORY");
                _saveField.setText(_saveDirectory);
				throw new IOException();
            }

            catch(NullPointerException npe) {
                GUIMediator.showError("ERROR_INVALID_SAVE_DIRECTORY");
                _saveField.setText(_saveDirectory);
				throw new IOException();
			}
        }

	}
}
