package com.limegroup.gnutella.gui.options.panes;

import com.limegroup.gnutella.gui.*;
import com.limegroup.gnutella.util.*;
import com.limegroup.gnutella.settings.*;
import java.awt.event.*;
import java.awt.*;
import javax.swing.*;
import java.io.*;
import java.net.*;

/**
 * This class defines the panel in the options window that allows the user
 * to force their ip address to the specified value.
 */
//2345678|012345678|012345678|012345678|012345678|012345678|012345678|012345678|
public final class ForceIPPaneItem extends AbstractPaneItem {
	/**
	 * Constant for the key of the locale-specific <tt>String</tt> 
	 * for the label on the checkbox.
	 */
	private final String CHECK_BOX_LABEL_KEY = 
		"OPTIONS_FORCE_IP_CHECK_BOX_LABEL";

	/**
	 * Constant for the key of the locale-specific <tt>String</tt> 
	 * for the label on the port text field.
	 */
	private final String PORT_LABEL_KEY = 
		"OPTIONS_FORCE_IP_PORT_LABEL";


	/**
	 * Constant handle to the check box that enables or disables this 
	 * feature.
	 */
	private final JCheckBox CHECK_BOX = new JCheckBox();

	/**
	 * Constant <tt>WholeNumberField</tt> instance that holds the port 
	 * to force to.
	 */
	private final WholeNumberField PORT_FIELD = new SizedWholeNumberField();


	/**
	 * The stored value to allow rolling back changes.
	 */
	private volatile int _forcedPort;

	/**
	 * The stored value to allow rolling back changes.
	 */
	private volatile boolean _forceLocalIP;

	/**
	 * The constructor constructs all of the elements of this 
	 * <tt>AbstractPaneItem</tt>.
	 *
	 * @param key the key for this <tt>AbstractPaneItem</tt> that the
	 *            superclass uses to generate locale-specific keys
	 */
	public ForceIPPaneItem(final String key) {
		super(key);

		CHECK_BOX.addItemListener(new LocalIPBoxListener());

		// create the check box panel with a glue forcing it to the left
		LabeledComponent comp0 = new LabeledComponent(CHECK_BOX_LABEL_KEY, 
													  CHECK_BOX, 
													  LabeledComponent.RIGHT_GLUE);
		
		LabeledComponent comp1 = new LabeledComponent(PORT_LABEL_KEY, 
													  PORT_FIELD);

		BoxPanel bp0 = new BoxPanel(BoxPanel.X_AXIS);
		bp0.add(comp0.getComponent());
		bp0.add(Box.createHorizontalGlue());

		BoxPanel bp1 = new BoxPanel(BoxPanel.X_AXIS);
		bp1.add(comp1.getComponent());

		add(comp0.getComponent());
		add(getVerticalSeparator());
		add(bp1);
	}

    /** 
	 * Listener class that responds to the checking and the 
	 * unchecking of the check box specifying whether or not to 
	 * use a local ip configuration.  It makes the other fields 
	 * editable or not editable depending on the state of the
	 * check box.
	 */
    private class LocalIPBoxListener implements ItemListener {
        public void itemStateChanged(ItemEvent e) {
			PORT_FIELD.setEditable(CHECK_BOX.isSelected());
        }
    }

	/**
	 * Defines the abstract method in <tt>AbstractPaneItem</tt>.<p>
	 *
	 * Sets the options for the fields in this <tt>PaneItem</tt> when the 
	 * window is shown.
	 */
	public void initOptions() {
        _forcedPort   = ConnectionSettings.FORCED_PORT.getValue();
        _forceLocalIP = ConnectionSettings.FORCE_IP_ADDRESS.getValue();

		PORT_FIELD.setValue(_forcedPort);
		CHECK_BOX.setSelected(_forceLocalIP);
	}

	/**
	 * Defines the abstract method in <tt>AbstractPaneItem</tt>.<p>
	 *
	 * Applies the options currently set in this window, displaying an
	 * error message to the user if a setting could not be applied.
	 *
	 * @throws IOException if the options could not be applied for some reason
	 */
	public void applyOptions() throws IOException {
		handleForcedIP();
	}

    /** 
	 * Helper method that attempts to apply the changes
	 * to the force ip address options.
	 */
    private void handleForcedIP() throws IOException {
        ConnectionSettings.FORCE_IP_ADDRESS.setValue(CHECK_BOX.isSelected());
        final int forcedPort = PORT_FIELD.getValue();

        if(!NetworkUtils.isValidPort(forcedPort)) {
            GUIMediator.showError("ERROR_FORCE_IP_PORT_RANGE");
            PORT_FIELD.setValue(_forcedPort);
            ConnectionSettings.FORCE_IP_ADDRESS.setValue(false);
            throw new IOException("bad port: "+forcedPort);            
        }
        
        ConnectionSettings.FORCED_PORT.setValue(forcedPort);
        _forcedPort = forcedPort;
    }
}
