package com.limegroup.gnutella.gui.init;

import com.limegroup.gnutella.util.CommonUtils;
import com.limegroup.gnutella.gui.GUIMediator;
import com.limegroup.gnutella.settings.ApplicationSettings;
import com.limegroup.gnutella.settings.SettingsHandler;
import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.util.Stack;

/**
 * This class manages the setup wizard.  It constructs all of the primary
 * classes and acts as the mediator between the various objects in the
 * setup windows.
 */
//2345678|012345678|012345678|012345678|012345678|012345678|012345678|012345678|

public class SetupManager {	

	/**
	 * the dialog window that holds all other gui elements for the setup.
	 */
	private JDialog _dialog;

	/**
	 * the class that handles the buttons for the setup panels.
	 */
	private SetupButtons _setupButtons;

	/** 
	 * the holder for the setup windows 
	 */
	private SetupWindowHolder _setupWindowHolder;

	/**
	 * holder for the current setup window.
	 */
	private SetupWindow _currentWindow;

	/**
	 * The array of scanned directory path names.
	 */
	private String[] _scannedPaths;

	/**
	 * Flag for whether or not a scan has already been completed.
	 */
	private boolean _scanned = false;

	/**
	 * The constructor creates the main <tt>JDialog</tt> instance and
	 * creates all of the setup window classes, buttons, etc.
	 */
	public SetupManager() {
		_dialog = new JDialog();
		_dialog.setModal(true);
	   
		// JDialog sizing seems to work differently with some Unix
		// systems, so we'll just make it resizable.
		if(!CommonUtils.isUnix())_dialog.setResizable(false);
		
        _dialog.addWindowListener(new WindowAdapter() {
            public void windowClosing(WindowEvent e) {
                cancelSetup();
            }
        });

		// set the layout of the content pane
		Container container = _dialog.getContentPane();
		BoxLayout containerLayout = new BoxLayout(container, BoxLayout.Y_AXIS);
        container.setLayout(containerLayout);

		// create the main panel
		JPanel setupPanel = new JPanel();
		BoxLayout layout = new BoxLayout(setupPanel, BoxLayout.Y_AXIS);
		setupPanel.setLayout(layout);


		// compare against a little bit less than the screen size,
		// as the screen size includes the taskbar
        Dimension d = new Dimension(SetupWindow.SETUP_WIDTH, 
									SetupWindow.SETUP_HEIGHT);

        Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
        _dialog.setLocation((screenSize.width-d.width)/2,
							(screenSize.height-d.height)/2);


		/********** code for the windows goes here ************/

		// create all of the windows needed for the setup
		SetupWindow window0 = new WelcomeWindow(this);
		SetupWindow window1 = new SaveWindow(this);
		SetupWindow window2 = new SpeedWindow(this);
		ScanWindow  window3 = new ScanWindow(this);
		SetupWindow window4 = new WaitWindow(this);
		SetupWindow window5 = new ScanConfirmWindow(this);
		SetupWindow window6 = new FinishWindow(this);

		// set the next windows
		window0.setNext(window1);    
		window1.setNext(window2); 
		window2.setNext(window3); 

		// handle the scan window slightly differently, as its
		// next window is dynamically determined
		window3.setYesWindow(window4);
		window3.setNoWindow(window6);

		// finish setting the next windows
		window4.setNext(window5);  
		window5.setNext(window6); 
		window6.setNext(window6);

		// set the previous windows
		window0.setPrevious(window0);    
		window1.setPrevious(window0); 
		window2.setPrevious(window1);    
		window3.setPrevious(window2);
		window4.setPrevious(window3);  
		window5.setPrevious(window3);
		window6.setPrevious(window5);
		
		// create the setup window holder class.
		_setupWindowHolder = new SetupWindowHolder();
		_setupWindowHolder.add(window0);
		_setupWindowHolder.add(window1);
		_setupWindowHolder.add(window2);
		_setupWindowHolder.add(window3);
		_setupWindowHolder.add(window4);
		_setupWindowHolder.add(window5);
		_setupWindowHolder.add(window6);
			
		show(window0);

		/********** end code for the windows  ************/

		// create the setup buttons panel
		_setupButtons = new SetupButtons(this);
		setupPanel.add(_setupWindowHolder);
		setupPanel.add(Box.createVerticalStrut(17));
		setupPanel.add(_setupButtons);

		// add the panel and make it visible		
		container.add(setupPanel);

		_dialog.pack();
		_dialog.setVisible(true);

	}
   
	/**
	 * Displays the next window in the setup sequence.
	 */
	public void next() {
		SetupWindow newWindow = _currentWindow.getNext();
		try {			
			_currentWindow.applySettings();
			newWindow.handleWindowOpeningEvent();
			show(newWindow);
		} catch(ApplySettingsException ase) {
			// there was a problem applying the settings from
			// the current window, so display the error message 
			// to the user.
			GUIMediator.showError(ase.getMessage());			
		}
	}

	/**
	 * Displays the previous window in the setup sequence.
	 */
	public void previous() {
		SetupWindow newWindow = _currentWindow.getPrevious();
		newWindow.handleWindowOpeningEvent();
		show(newWindow);
	}

	/**
	 * Sets the array of scanned path names.
	 */
	public void scan() {
		if(!_scanned) {
			_scannedPaths = scanDrive();
			_scanned = true;
		}
	}

	/**
	 * Returns the array of potential directory paths to share
	 * based on the hard drive scan.
	 *
	 * @return the array of scanned directory paths containing 
	 *         potential files to share
	 */
	public String[] getScannedPaths() {
		return _scannedPaths;
	}

	
	/**
	 * Cancels the setup.
	 */
	public void cancelSetup() {
		_dialog.dispose();
		System.exit(0);
	}

	/**
	 * Completes the setup.
	 */
	public void finishSetup() {		
		_dialog.dispose();
		ApplicationSettings.INSTALLED.setValue(true);
		SettingsHandler.save();
	}

	/**
	 * Mediator method for changing the buttons in the setup button class
	 * to show the finish button.
	 */
	public void goToFinishButtons() {
		_setupButtons.goToFinishButtons();	
	}

	/**
	 * Mediator method for changing the buttons in the setup button class
	 * to show the cancel button.
	 */
	public void goToCancelButtons() {
		_setupButtons.goToCancelButtons();	
	}

	/**
	 * Mediator method for changing the buttons in the setup button class
	 * to show its standard buttons
	 */
	public void goToStandardButtons() {
		_setupButtons.goToStandardButtons();	
	}

	/**
	 * Show the specified window
	 */
	private void show(SetupWindow window) {
		_setupWindowHolder.show(window.getKey());
		_dialog.setTitle(window.getName());
		_currentWindow = window;
	}

	/**
	 * Scans the user's hard drive for media files, and returns,
	 * at most, the top 5 directories as an array of Strings.
	 *
	 * @return the array of pathname <tt>String</tt>s of the directories
	 *         containing media files
	 */   
	private String[] scanDrive() {
		FileScan fs = new FileScan();
		String[] filters = {
			"Recycle","Incomplete","LimeWire","Microsoft", "bin", 
			"system","WINNT"
		};
		fs.setFilters(filters);

		// get the root directory of the current directory.
		// scan that directory for files.
		File root = getRoot(CommonUtils.getCurrentDirectory());
		try {
			fs.scan( root.getCanonicalPath() );
		} catch(IOException ioe) {
			
		}
		return fs.getListAsArray();
	}

	/**
	 * Recursively travels up the path pf the file until it finds
	 * the root directory.  
	 *
	 * @return the <tt>File</tt> instance denoting the abstract pathname
	 *         of the root directory
	 */
	private File getRoot(File f) {
		String parent;
		parent = f.getParent();
		if (parent == null)
			return f;
		else 
			return getRoot( new File(parent) );
	}

}






