package com.limegroup.gnutella.gui;

import com.limegroup.gnutella.util.CommonUtils;
import com.apple.mrj.*;
import java.io.*;

/**
 * This class handles Macintosh specific events, registering this class as
 * the handler for those events with MRJApplicationUtils.  The handled
 * events include the selection of the "About" option in the Mac file menu,
 * the selection of the "Quit" option from the Mac file menu, and the
 * dropping of a file on LimeWire on the Mac, which LimeWire would be
 * expected to handle in some way.
 */
final class MacEventHandler 
	implements MRJAboutHandler, MRJQuitHandler, MRJOpenDocumentHandler {

	/**
	 * The constructor registers this class as the event handler for all
	 * of the handled events.
	 */
	MacEventHandler() {
		MRJApplicationUtils.registerAboutHandler(this);
		MRJApplicationUtils.registerQuitHandler(this);
		MRJApplicationUtils.registerOpenDocumentHandler(this);
	}

	/**
	 * This responds to the selection of the about option by displaying the
	 * about window to the user.  On OSX, this runs in a new thread to handle
     * the possibility that event processing can become blocked if launched
     * in the calling thread.
	 */
	public void handleAbout() {
        if(CommonUtils.isMacClassic() || CommonUtils.isJaguarOrAbove()) {
			GUIMediator.showAboutWindow();
        } else {
            new Thread("Mac about menu thread") {
                public void run() {
                    GUIMediator.showAboutWindow();
                }
            }.start();
        }
	}

	/**
	 * This method responds to a quit event by closing the application in
	 * the whichever method the user has configured (closing after completed
	 * file transfers by default).  On OSX, this runs in a new thread to handle
     * the possibility that event processing can become blocked if launched
     * in the calling thread.
	 */
	public void handleQuit() {
        if(CommonUtils.isMacClassic() || CommonUtils.isJaguarOrAbove()) {
            GUIMediator.close();
        } else {
            new Thread("Mac quit thread") {
                public void run() {
                    GUIMediator.close();
                }
            }.start();
        }
	}

	/**
	 * This method handles a request to open the specified file.
	 */
	public void handleOpenFile(File file) {
        if(CommonUtils.isMacClassic() || CommonUtils.isJaguarOrAbove()) {
            PackagedMediaFileLauncher.launchFile(file.toString(), false);  
        } else {
            new HandleOpenFile(file).start();
        }
	}

    class HandleOpenFile extends Thread {
        File requestedFile;

        public HandleOpenFile(File file) {
            super("handleOpenFile");
            requestedFile = file;
        }
        public void run() {
            PackagedMediaFileLauncher.launchFile(
              requestedFile.toString(), false);  
        }
    }
}
