package com.limegroup.gnutella.gui;

import javax.swing.*;
import javax.swing.table.*;
import java.awt.*;
import java.awt.event.MouseEvent;

import com.limegroup.gnutella.gui.tables.JSortTable;
import com.limegroup.gnutella.settings.ThemeFileHandler;
import com.limegroup.gnutella.settings.UISettings;
import com.limegroup.gnutella.gui.search.*;
import com.limegroup.gnutella.util.CommonUtils;
import com.limegroup.gnutella.util.DataUtils;

/**
 * Extends JTreeTable. Basially adds a class variable - a String[],
 * which is set by one of the methods we added.
 * <p> 
 * Overrides the createToolTip method of JComponent.
 * @author Sumeet Thadani
 */

public final class JMultilineToolTipTreeTable extends JTreeTable implements JSortTable {

    private String[] tips;
    
    private Object _sortedColumn;
    private Object _pressedColumn;
    private boolean _isAscending;

    private static final boolean IS_JAVA_13 =
        CommonUtils.isJava13OrLater() && !CommonUtils.isJava14OrLater();    
    
    public JMultilineToolTipTreeTable(TreeTableModel treeTableModel) {
        super(treeTableModel);
        tips = DataUtils.EMPTY_STRING_ARRAY;
    }
    
    /**
     * Sets the internal tooltip text for use with the next
     * createToolTip.
     */
    public String getToolTipText(MouseEvent e) {
        Point p = e.getPoint();
        int row = rowAtPoint(p);
        if ( row > -1 ) {
            TableLine theLine = (TableLine)nodeForRow(row);
            if(theLine == null) {
                tips = DataUtils.EMPTY_STRING_ARRAY;
                return null;
            }
            tips = theLine.getMetaText();
            return "" + row;
        }
        tips = DataUtils.EMPTY_STRING_ARRAY;
        return null;
    }


    /**
     *@return The JToolTip returned is actually a JMultilineToolTip
     */
    public JToolTip createToolTip() {
        JMultilineToolTip ret = JMultilineToolTip.instance();
        ret.setToolTipArray( tips );
        tips = DataUtils.EMPTY_STRING_ARRAY;
        return ret;
    }
    
    public void sortColumn(Object id, boolean asc) {
        _sortedColumn = id;
        _isAscending = asc;
    }
    
    // returns the VIEW index of the sorted column.
    public int getSortedColumnIndex() {
        if ( _sortedColumn == null ) return -1;
        // if the column is hidden it'll throw the exception.
        try {
            return convertColumnIndexToView( getColumn(_sortedColumn).getModelIndex());
        } catch (IllegalArgumentException e) {
            return -1;
        }    
    }
    
    public int getPressedColumnIndex() {
        if ( _pressedColumn == null ) return -1;
        // if the column is hidden it'll throw the exception.
        try {
            return convertColumnIndexToView( getColumn(_pressedColumn).getModelIndex());
        } catch (IllegalArgumentException e) {
            return -1;
        }    
    }
    
    public void setPressedColumnId(Object id) {
        _pressedColumn = id;
    }
    
    // returns whether or not the sorted column is ascending
    public boolean isSortedColumnAscending() {
        return _isAscending;
    }
    
    /**
     * This overrides JTable.prepareRenderer so that we can stripe the
     * rows as needed.
     */
    public Component prepareRenderer(TableCellRenderer renderer,
      int row, int column) {
        Object value = getValueAt(row, column);
    	boolean isSelected = isCellSelected(row, column);
    	boolean rowIsAnchor = (selectionModel.getAnchorSelectionIndex()==row);
    	boolean colIsAnchor =
    	  (columnModel.getSelectionModel().getAnchorSelectionIndex()==column);
    	boolean hasFocus = (rowIsAnchor && colIsAnchor) && hasFocus();
    
    	Component r = renderer.getTableCellRendererComponent(this, value,
    	                                              isSelected, hasFocus,
    	                                              row, column);
    	                                              
        Color  odd = ThemeFileHandler.TABLE_ALTERNATE_COLOR.getValue();
        Color even = ThemeFileHandler.TABLE_BACKGROUND_COLOR.getValue();
	                                              
        if ( isSelected ) {
            // do nothing if selected.
        } else if (hasFocus && isCellEditable(row, column)) {
            // do nothing if we're focused & editting.
        } else if (even.equals(odd)) {
            // do nothing if backgrounds are the same.
        } else if (!UISettings.ROW_STRIPE_ENABLED.getValue()) {
            // if the renderer's background isn't already the normal one,
            // change it.  (needed for real-time changing of the option)
            if( r != null && !r.equals(even) )
                r.setBackground(even);
        } else {
            TableLine line = (TableLine)nodeForRow(row);
            int realRow = 0;
            // if it is a top-level result, use its row.
            if( line.parentLine == null ) {
                realRow = line.childName;
            } else { // otherwise, use its parents row.
                realRow = line.parentLine.childName;
            }
            
            if ( realRow % 2 != 0 ) {
                r.setBackground(odd);
            } else {
                r.setBackground(even);
            }
        }
        
        // This is necessary to work around the 'optimization' of
        // DefaultTableCellRenderer in 1.3
        // See: http://developer.java.sun.com/developer/bugParade/bugs/4382860.html
        if( IS_JAVA_13 ) {
        	Color back = r.getBackground();
            boolean colorMatch = (back != null) && ( back.equals(even) );
            ((JComponent)r).setOpaque(!colorMatch);
        }        
        
        return r;
    }  

}
