/*********************************************************************
 *
 *      Copyright (C) 2002-2003 Nathan Fiedler
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Id: EvaluatorTest.java 1095 2003-12-05 08:20:57Z nfiedler $
 *
 ********************************************************************/

package com.bluemarsh.jswat.expr;

import junit.extensions.*;
import junit.framework.*;

/**
 * Tests the Evaluator class.
 */
public class EvaluatorTest extends TestCase {

    public EvaluatorTest(String name) {
        super(name);
    }

    public static Test suite() {
        return new TestSuite(EvaluatorTest.class);
    }

    public static void main(String[] args) {
        junit.textui.TestRunner.run(suite());
    }

    public void test_evaluator() {
        TestData[] testDatum = new TestData[] {
            // base cases
            new TestData(null, null, "Null expression should be null"),
            new TestData("", null, "Empty string should be null"),
            new TestData("null", null, "'null' should be null"),
            new TestData("false", Boolean.FALSE),
            new TestData("true", Boolean.TRUE),
            new TestData("\"abc\"", new String("abc")),
            new TestData("128", new Integer(128)),
            new TestData("182L", new Long(182)),
            new TestData("1.28", new Float(1.28)),
            new TestData("1.82F", new Float(1.82)),
            new TestData("1.28D", new Double(1.28D)),

            // binary: +, -
            new TestData("1 + 1", new Integer(2)),
            new TestData("\"abc\" + 123", new String("abc123")),
            new TestData("1 + true", new String("1true")),
            new TestData("1 + null", new String("1null")),
            new TestData("2 - 1", new Integer(1)),
            new TestData("1 - 2", new Integer(-1)),
            new TestData("1 - 2L", new Long(-1)),
            new TestData("1L - 2", new Long(-1)),
            new TestData("1.0D - 2", new Double(-1.0)),
            new TestData("1.0 + 2.0", new Float(3.0)),

            // unary: -, +, !, ~
            new TestData("-1", new Integer(-1)),
            new TestData("-1L", new Long(-1)),
            new TestData("+1", new Integer(1)),
            new TestData("+1L", new Long(1)),
            new TestData("!true", Boolean.FALSE),
            new TestData("!false", Boolean.TRUE),
            new TestData("~128", new Integer(-129)),
            new TestData("2 - -1", new Integer(3)),
            new TestData("1 - +2", new Integer(-1)),
            new TestData("~'a'", new Integer(-98)),

            // mult, div, mod: *, /, %
            new TestData("1 * 2", new Integer(2)),
            new TestData("6 / 2", new Integer(3)),
            new TestData("1 / 0"),
            new TestData("0.0 / 0.0", new Float(Float.NaN)),
            new TestData("1 / null"),
            new TestData("1 / 2.0", new Float(0.5)),
            new TestData("2 / 8.0", new Float(0.25)),
            new TestData("1.0 / 0", new Float(Float.POSITIVE_INFINITY)),
            new TestData("-1.0 / 0", new Float(Float.NEGATIVE_INFINITY)),
            new TestData("11 % 2", new Integer(1)),
            new TestData("1 + 2 * 3", new Integer(7)),

            // equality: ==, !=
            new TestData("true == true", Boolean.TRUE),
            new TestData("true == false", Boolean.FALSE),
            new TestData("'A' == 'a'", Boolean.FALSE),
            new TestData("'a' == 'a'", Boolean.TRUE),
            new TestData("true != true", Boolean.FALSE),
            new TestData("false != false", Boolean.FALSE),
            new TestData("true != false", Boolean.TRUE),
            new TestData("true == null", Boolean.FALSE),
            // For some reason doesn't this work, but i'm not concerned.
            //new TestData("0.0 / 0.0 != 0.0 / 0.0", Boolean.TRUE),
            new TestData("0.0 / 0.0 != 0.0", Boolean.TRUE),

            // bitwise: &, |, ^
            new TestData("255 & 32", new Integer(32)),
            new TestData("64 | 32", new Integer(96)),
            new TestData("255 ^ 32", new Integer(223)),
            new TestData("2.0 ^ 32"),
            new TestData("255 ^ 3.2"),
            new TestData("null ^ 32"),
            new TestData("'a' | 32", new Integer(97)),
            new TestData("'a' ^ 32", new Integer(65)),

            // shift: <<, >>, >>>
            new TestData("32 >> 4", new Integer(2)),
            new TestData("32L >> 4", new Long(2)),
            new TestData("0xF1234567 >>> 4", new Long(252851286)),
            new TestData("2 << 4", new Integer(32)),

            // comparison: <, >, <=, <=
            new TestData("32 > 4", Boolean.TRUE),
            new TestData("4 > 32", Boolean.FALSE),
            new TestData("32 < 4", Boolean.FALSE),
            new TestData("4 < 32", Boolean.TRUE),
            new TestData("4 <= 32", Boolean.TRUE),
            new TestData("32 <= 32", Boolean.TRUE),
            new TestData("32 >= 4", Boolean.TRUE),
            new TestData("32 >= 32", Boolean.TRUE),
            new TestData("4 >= 32", Boolean.FALSE),
            new TestData("32 <= 4", Boolean.FALSE),
            new TestData("0.0 / 0.0 < 0.0", Boolean.FALSE),
            new TestData("0.0 / 0.0 > 0.0", Boolean.FALSE),
            new TestData("0.0 >= 0.0 / 0.0", Boolean.FALSE),
            new TestData("0.0 <= 0.0 / 0.0", Boolean.FALSE),
            new TestData("(0.0/0.0 < 0.0) == !(0.0/0.0 >= 0.0)",
                         Boolean.FALSE),

            // logical: &&, ||
            new TestData("true || false", Boolean.TRUE),
            new TestData("false || false", Boolean.FALSE),
            new TestData("null || false"),
            new TestData("true && true", Boolean.TRUE),
            new TestData("true && false", Boolean.FALSE),

            // grouping: ()
            new TestData("(1 + 2) * 3", new Integer(9)),
            new TestData("(1) + (2) * (3)", new Integer(7)),
            new TestData("((1 + (2 * 3)))", new Integer(7)),
            new TestData("(((1)))", new Integer(1)),

            // typecast ()
            new TestData("(boolean) true", Boolean.TRUE),
            new TestData("(Boolean) false", Boolean.FALSE),
            new TestData("(String) \"string\"", new String("string")),
            new TestData("(Number) 123456", new Integer(123456)),
            new TestData("(String) 123456", new String("123456")),
            new TestData("(long) 123456", new Long(123456)),
            new TestData("(Long) 123456", new Long(123456)),
            new TestData("(byte) 12", new Byte((byte) 12)),
            new TestData("(Byte) 12", new Byte((byte) 12)),
            new TestData("(short) 12", new Short((short) 12)),
            new TestData("(Short) 12", new Short((short) 12)),
            new TestData("(float) 0.1", new Float(0.1)),
            new TestData("(Float) 0.1", new Float(0.1)),
            new TestData("(double) 0.2", new Double(0.20000000298023224)),
            new TestData("(Double) 0.2", new Double(0.20000000298023224)),
            new TestData("(Object) null", null),
            new TestData("(boolean) 123"),
            new TestData("(void) 123"),
            new TestData("(Class) 123"),
            new TestData("(List) 123"),
            new TestData("(short) true"),
        };

        for (int ii = 0; ii < testDatum.length; ii++) {
            TestData data = testDatum[ii];
            Evaluator eval = new Evaluator(data.expr);
            Object result = null;
            try {
                result = eval.evaluate(null, 0);
                if (data.fail) {
                    // was expected to fail
                    StringBuffer buf = new StringBuffer();
                    buf.append(data.expr);
                    buf.append(" should have failed, but got ");
                    buf.append(result);
                    if (result != null) {
                        buf.append(" (");
                        buf.append(result.getClass().getName());
                        buf.append(')');
                    }
                    fail(buf.toString());
                }
            } catch (EvaluationException ee) {
                if (!data.fail) {
                    // was not expected to fail
                    fail(ee.toString());
                }
            }

            boolean equals;
            if (result == null && data.result == null) {
                equals = true;
            } else if (result == null || data.result == null) {
                equals = false;
            } else {
                equals = result.equals(data.result);
            }

            if (!equals) {
                if (data.message == null) {
                    StringBuffer buf = new StringBuffer();
                    buf.append(data.expr);
                    buf.append(" should have been ");
                    buf.append(data.result);
                    if (data.result != null) {
                        buf.append(" (");
                        buf.append(data.result.getClass().getName());
                        buf.append(") ");
                    }
                    buf.append(" but got ");
                    buf.append(result);
                    if (result != null) {
                        buf.append(" (");
                        buf.append(result.getClass().getName());
                        buf.append(')');
                    }
                    fail(buf.toString());

                } else {
                    StringBuffer buf = new StringBuffer();
                    buf.append(data.message);
                    buf.append(": expected ");
                    buf.append(data.result);
                    if (data.result != null) {
                        buf.append(" (");
                        buf.append(data.result.getClass().getName());
                        buf.append(')');
                    }
                    buf.append(": got ");
                    buf.append(result);
                    if (result != null) {
                        buf.append(" (");
                        buf.append(result.getClass().getName());
                        buf.append(')');
                    }
                    fail(buf.toString());
                }
            }
        }
    }

    /**
     * Structure to hold test parameters.
     */
    protected class TestData {
        /** Expression to be evaluated. */
        public String expr;
        /** The expected result as the expect type. */
        public Object result;
        /** The error message to display if the expression does not
         * evaluate to be equal to the reslt. */
        public String message;
        /** True if the expression is expected to cause an exception. */
        public boolean fail;

        public TestData(String expr) {
            this.expr = expr;
            fail = true;
        }

        public TestData(String expr, Object result) {
            this.expr = expr;
            this.result = result;
        }

        public TestData(String expr, Object result, String message) {
            this.expr = expr;
            this.result = result;
            this.message = message;
        }
    }
}
