/*********************************************************************
 *
 *      Copyright (C) 2002-2003 Nathan Fiedler
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Id: MethodNode.java 1093 2003-12-05 03:27:21Z nfiedler $
 *
 ********************************************************************/

package com.bluemarsh.jswat.expr;

import com.bluemarsh.jswat.breakpoint.AmbiguousMethodException;
import com.bluemarsh.jswat.parser.java.node.Token;
import com.bluemarsh.jswat.util.Classes;
import com.bluemarsh.jswat.util.Variables;
import com.bluemarsh.jswat.util.Strings;
import com.sun.jdi.ClassNotLoadedException;
import com.sun.jdi.IncompatibleThreadStateException;
import com.sun.jdi.InvalidTypeException;
import com.sun.jdi.Location;
import com.sun.jdi.Method;
import com.sun.jdi.ObjectReference;
import com.sun.jdi.ReferenceType;
import com.sun.jdi.StackFrame;
import com.sun.jdi.ThreadReference;
import com.sun.jdi.Value;
import com.sun.jdi.VirtualMachine;
import java.util.ArrayList;
import java.util.List;

/**
 * Class MethodNode represents a method invocation.
 *
 * @author  Nathan Fiedler
 */
class MethodNode extends OperatorNode {
    /** The method that was invoked. */
    private Method theMethod;

    /**
     * Constructs a MethodNode associated with the given token.
     *
     * @param  node  lexical token.
     */
    public MethodNode(Token node) {
        super(node);
    } // MethodNode

    /**
     * Returns the value of this node.
     *
     * @param  context  evaluation context.
     * @return  value.
     * @throws  EvaluationException
     *          if an error occurred during evaluation.
     */
    protected Object eval(EvaluationContext context)
        throws EvaluationException {

        // Get the preliminaries out of the way first.
        ThreadReference thread = context.getThread();
        if (thread == null) {
            throw new EvaluationException(
                Bundle.getString("error.method.thread.set"));
        }
        int frameIdx = context.getFrame();
        VirtualMachine vm = thread.virtualMachine();

        // First argument had better be the method name.
        IdentifierNode idn = (IdentifierNode) getChild(0);
        String methodRef = idn.getName();

        // Get the object or class referenced by the first part of the
        // method identifier. If no first part given, infer from the
        // current location in the debuggee.
        ObjectReference objref = null;
        ReferenceType reftype = null;
        int lastDot = methodRef.lastIndexOf('.');
        if (lastDot == -1) {
            // First part was not given, default to class containing
            // current location.
            try {
                if (thread.frameCount() == 0) {
                    throw new EvaluationException(
                        Bundle.getString("error.method.thread.stack"));
                }
                StackFrame frame = thread.frame(frameIdx);
                objref = frame.thisObject();

                if (objref == null) {
                    Location location = context.getLocation();
                    if (location == null) {
                        throw new EvaluationException(
                            Bundle.getString("error.method.location"));
                    }
                    reftype = location.declaringType();
                    if (reftype == null) {
                        throw new EvaluationException(
                            Bundle.getString("error.method.location"));
                    }
                } else {
                    reftype = objref.referenceType();
                }
            } catch (IncompatibleThreadStateException itse) {
                throw new EvaluationException(
                    Bundle.getString("error.method.thread.state"));
            }

        } else {
            // Is the first part a variable or a class name?
            String varOrClass = methodRef.substring(0, lastDot);
            try {
                Value variable = Variables.getValue(varOrClass, thread,
                                                    frameIdx);
                objref = (ObjectReference) variable;
                reftype = objref.referenceType();
            } catch (Exception e) {
                // Maybe it's not a variable reference.
            }

            if (reftype == null) {
                // See if the argument is a class name.
                List classes = Classes.findClassesByPattern(
                    vm, varOrClass);
                if (classes == null || classes.size() == 0) {
                    throw new EvaluationException(
                        Bundle.getString("error.method.name")
                        + ' ' + varOrClass);
                }
                reftype = (ReferenceType) classes.get(0);
            }
        }

        // Get the list of method arguments and their types.
        int count = childCount();
        // The first child is the method name, the rest are arguments.
        List argumentValues = new ArrayList(count - 1);
        List argumentTypes = new ArrayList(count - 1);
        for (int ii = 1; ii < count; ii++) {
            Node n = getChild(ii);
            Object o = n.evaluate(context);
            argumentValues.add(o);
            String t = n.getType(context);
            if (t == null) {
                // For method arguments, 'null' is a special type.
                argumentTypes.add("<null>");
            } else {
                argumentTypes.add(t);
            }
        }

        // Locate method in the resolved class.
        String methodName = methodRef.substring(lastDot + 1);
        try {
            theMethod = Classes.findMethod(
                reftype, methodName, argumentTypes, true);
        } catch (AmbiguousMethodException ame) {
            throw new EvaluationException(
                Bundle.getString("error.method.ambiguous")
                + ' ' + methodName + '('
                + Strings.listToString(argumentTypes) + ')');
        } catch (ClassNotLoadedException cnle) {
            throw new EvaluationException(
                Bundle.getString("error.method.class")
                + ' ' + cnle.className());
        } catch (InvalidTypeException ite) {
            throw new EvaluationException(
                Bundle.getString("error.method.argument")
                + ' ' + ite.getMessage());
        } catch (NoSuchMethodException nsme) {
            throw new EvaluationException(
                Bundle.getString("error.method.method")
                + ' ' + methodName + '('
                + Strings.listToString(argumentTypes) + ')');
        }

        // Convert the arguments to JDI objects.
        mirrorArguments(vm, argumentValues);

        // Invoke the method and return the results.
        try {
            return Classes.invokeMethod(objref, reftype, thread,
                                        theMethod, argumentValues);
        } catch (Exception e) {
            throw new EvaluationException(e);
        }
    } // eval

    /**
     * Translate the given list of arguments into Value instances of the
     * appropriate type. The list is modified in-place.
     *
     * @param  vm    debuggee virtual machine.
     * @param  args  list of arguments to mirror.
     */
    protected static void mirrorArguments(VirtualMachine vm, List args) {
        for (int ii = 0; ii < args.size(); ii++) {
            Object o = args.get(ii);
            if (o instanceof String) {
                args.set(ii, vm.mirrorOf((String) o));
            } else if (o instanceof Boolean) {
                args.set(ii, vm.mirrorOf(((Boolean) o).booleanValue()));
            } else if (o instanceof Character) {
                args.set(ii, vm.mirrorOf(((Character) o).charValue()));
            } else if (o instanceof Double) {
                args.set(ii, vm.mirrorOf(((Double) o).doubleValue()));
            } else if (o instanceof Float) {
                args.set(ii, vm.mirrorOf(((Float) o).floatValue()));
            } else if (o instanceof Integer) {
                args.set(ii, vm.mirrorOf(((Integer) o).intValue()));
            } else if (o instanceof Long) {
                args.set(ii, vm.mirrorOf(((Long) o).longValue()));
            } else if (o instanceof Short) {
                args.set(ii, vm.mirrorOf(((Short) o).shortValue()));
            } else if (o instanceof Byte) {
                args.set(ii, vm.mirrorOf(((Byte) o).byteValue()));
            }
            // Else it's a null or Value.
        }
    } // mirrorArguments

    /**
     * Returns this operator's precedence value. The lower the value
     * the higher the precedence. The values are equivalent to those
     * described in the Java Language Reference book (2nd ed.), p 106.
     *
     * @return  precedence value.
     */
    public int precedence() {
        return 1;
    } // precedence

    /**
     * Returns the signature of the type this node represents. If the
     * type is void, or otherwise unrecognizable, an exception is
     * thrown.
     *
     * @param  context  evaluation context.
     * @return  type signature, or null if value is null.
     * @throws  EvaluationException
     *          if an error occurred during evaluation.
     */
    protected String type(EvaluationContext context)
        throws EvaluationException {

        // Force the evaluation to happen, if it hasn't already, so we
        // get the method reference.
        evaluate(context);
        try {
            return theMethod.returnType().signature();
        } catch (ClassNotLoadedException cnle) {
            throw new EvaluationException(
                Bundle.getString("error.method.class")
                + ' ' + cnle.className());
        }
    } // type
} // MethodNode
