/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 2000-2007 Sun Microsystems, Inc. All rights reserved. 
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License ("CDDL") (collectively, the "License").  You may
 * not use this file except in compliance with the License.  You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or mq/legal/LICENSE.txt.  See the License for the specific language
 * governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at mq/legal/LICENSE.txt.  Sun designates
 * this particular file as subject to the "Classpath" exception as provided by
 * Sun in the GPL Version 2 section of the License file that accompanied this
 * code.  If applicable, add the following below the License Header, with the
 * fields enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or  to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright holder. 
 */

/*
 * @(#)StoreManager.java	1.25 06/29/07
 */ 

package com.sun.messaging.jmq.jmsserver.persist;

import com.sun.messaging.jmq.util.log.Logger;
import com.sun.messaging.jmq.jmsserver.Globals;
import com.sun.messaging.jmq.jmsserver.BrokerStateHandler;
import com.sun.messaging.jmq.jmsserver.persist.jdbc.JDBCStore;
import com.sun.messaging.jmq.jmsserver.persist.jdbc.DBConnectionPool;
import com.sun.messaging.jmq.jmsserver.config.*;
import com.sun.messaging.jmq.jmsserver.resources.*;
import com.sun.messaging.jmq.jmsserver.util.BrokerException;

/**
 * This class contains static methods to obtain a singleton Store instance
 * for storing and retrieving data needed by the broker.
 */

public class StoreManager {

    static private final String PERSIST_PROP = Globals.IMQ + ".persist.";
    static private final String CLASS_PROP = ".class";
    static private final String STORE_TYPE_PROP =
					Globals.IMQ + ".persist.store";

    static private final String DEFAULT_STORE_TYPE = Store.FILE_STORE_TYPE;

    static private final String DEFAULT_FILESTORE_CLASS =
	"com.sun.messaging.jmq.jmsserver.persist.file.FileStore";

    static private final String DEFAULT_JDBCSTORE_CLASS =
	"com.sun.messaging.jmq.jmsserver.persist.jdbc.JDBCStore";

    // Singleton Store instance
    static private Store store = null;

    /**
     * Return a singleton instance of a Store object.
     * <p>
     * The type of store to use is defined in the property:<br>
     * jmq.persist.store=<type>
     * <p>
     * The class to use for the specified type is defined in:<br>
     * jmq.persist.<type>.class=<classname>
     * <p>
     * If the type property is not defined, the default file based
     * store will be instantiated and returned.
     * If 'jdbc' type is defined, and no class is defined, the default
     * jdbc based store will be instantiated and returned.
     * <p>
     * If the type property is defined but we fail to instantiate the
     * correspoinding class, a BrokerException will be thrown
     * @return	a Store
     * @exception BrokerException	if it fails to instantiate a Store
     *					instance
     */
    public static synchronized Store getStore() throws BrokerException {
        Logger logger = Globals.getLogger();
        BrokerResources br = Globals.getBrokerResources();

	if (store == null) {
            // Can't open the store if we are shutting down
            if (BrokerStateHandler.shuttingDown) {
                throw new BrokerException(
                   Globals.getBrokerResources().getKString(
                       BrokerResources.X_SHUTTING_DOWN_BROKER),
                   BrokerResources.X_SHUTTING_DOWN_BROKER);
            }

	    BrokerConfig config = Globals.getConfig();

	    String type = config.getProperty(STORE_TYPE_PROP,
					DEFAULT_STORE_TYPE);
	    if (Store.getDEBUG()) {
		logger.log(logger.DEBUG, STORE_TYPE_PROP + "=" + type);
	    }

	    String classname = config.getProperty(
				PERSIST_PROP + type + CLASS_PROP);
	    
	    if (classname == null || classname.equals("")) {
		if (type.equals(Store.FILE_STORE_TYPE))
		    classname = DEFAULT_FILESTORE_CLASS;
		else if (type.equals(Store.JDBC_STORE_TYPE))
		    classname = DEFAULT_JDBCSTORE_CLASS;
		else
		    classname = null;
	    }

	    if (classname == null) {
		throw new BrokerException(
                    br.getString(br.E_BAD_STORE_TYPE, type));
	    } else {
		if (Store.getDEBUG()) {
		    logger.log(logger.DEBUG,
			PERSIST_PROP + type + CLASS_PROP + "=" + classname);
		}
	    }

	    try {
		store = (Store)Class.forName(classname).newInstance();

                // Initialize transaction logging class if enabled.
                // If return status is set to true, the store need to be
                // reopen because the store has been reconstructed from
                // the txn log files; probably due to system crash or the
                // broker didn't shutdown cleanly.
                if (store.initTxnLogger()) {
                    store.close();
                    store = (Store)Class.forName(classname).newInstance();
                }
	    } catch (Exception e) {
                if (e instanceof BrokerException) {
                    throw (BrokerException)e;
                } else {
                    throw new BrokerException(
                        br.getString(br.E_OPEN_STORE_FAILED), e);
                }
	    }
	}

	return store;
    }

    /**
     * Release the singleton instance of a Store object.
     * The store will be closed and any resources used by it released.
     * <p>
     * The next time <code>getStore()</code> is called, a new instance will
     * be instantiaed.
     * @param	cleanup	if true, the store will be cleaned up, i.e.
     *			redundant data removed.
     */
    public static synchronized void releaseStore(boolean cleanup) {

	if (store != null) {
	    // this check is for tonga test so that the store
	    // is not closed twice
	    if (!store.closed()) {
		store.close(cleanup);
	    }

	    store = null;
	}
    }

    public static Store getStoreForTonga() throws BrokerException {

        // For tonga test, get a store without a lock otherwise we'll
        // an error because the store is already locked by the broker!
        BrokerConfig config = Globals.getConfig();
        String type = config.getProperty(STORE_TYPE_PROP, DEFAULT_STORE_TYPE);
        if (type.equals(Store.JDBC_STORE_TYPE)) {
            config.put(JDBCStore.LOCK_STORE_PROP, "false");
            // Limit the # of conn in the pool to 2
            config.put(DBConnectionPool.NUM_CONN_PROP, "2");
        }
        return getStore();
    }
}
