/*
 * @(#)EmbeddedBrokerRunner.java	1.10 06/02/09
 *
 * Copyright 2005 Sun Microsystems, Inc. All Rights Reserved.
 * SUN PROPRIETARY/CONFIDENTIAL.
 * Use is subject to license terms.
 *
 */

package com.sun.messaging.jms.ra;

import java.util.logging.Logger;
import java.util.Vector;
import java.util.Properties;
import java.util.Enumeration;
import java.util.StringTokenizer;

import com.sun.messaging.jmq.jmsserver.BrokerProcess;
import com.sun.messaging.jmq.jmsserver.BrokerNotification;

/**
 * Runs an embedded broker instance through
 * the Broker* interfaces exposed for in-process
 * broker lifecycle control
 * 
 * @author George Tharakan
 */

public class EmbeddedBrokerRunner implements BrokerNotification {

    /* Properties to be used by the embedded broker */
    private Properties brokerProps = null;

    /* The BrokerProcess instance */
    private BrokerProcess bp = null;

    /* The main broker thread */
    private Thread bt = null;

    /* Loggers */
    private static transient final String _className = "com.sun.messaging.jms.ra.EmbeddedBrokerRunner";
    protected static transient Logger _loggerL;
    protected static transient final String _lgrNameLifecycle = "javax.resourceadapter.mqjmsra.lifecycle";
    protected static transient final String _lgrMIDPrefix = "MQJMSRA_EB";
    protected static transient final String _lgrMID_EET = _lgrMIDPrefix + "1001: ";
    protected static transient final String _lgrMID_INF = _lgrMIDPrefix + "1101: ";
    protected static transient final String _lgrMID_WRN = _lgrMIDPrefix + "2001: ";
    protected static transient final String _lgrMID_ERR = _lgrMIDPrefix + "3001: ";
    protected static transient final String _lgrMID_EXC = _lgrMIDPrefix + "4001: ";

    static {
        _loggerL = Logger.getLogger(_lgrNameLifecycle);
    }

    public EmbeddedBrokerRunner(
            String brokerInstanceName, String brokerBindAddress, int brokerPort,
            String brokerHomeDir, String brokerLibDir, String brokerVarDir, String brokerJavaDir,
            String brokerExtraArgs,
            boolean useJNDIRMIServiceURL, int rmiRegistryPort, boolean startRMIRegistry,
            boolean useSSLJMXConnector, int brokerStartTimeout,
            String adminUsername, String adminPassword, String adminPassFile, Properties sysProps)
    {
        Object params[] = new Object[16];
        params[0] = brokerInstanceName;
        params[1] = brokerBindAddress;
        params[2] = Integer.toString(brokerPort);
        params[3] = brokerHomeDir;
        params[4] = brokerLibDir;
        params[5] = brokerVarDir;
        params[6] = brokerJavaDir;
        params[7] = brokerExtraArgs;
        params[8] = new Boolean(useJNDIRMIServiceURL);
        params[9] = new Integer(rmiRegistryPort);
        params[10] = new Boolean(startRMIRegistry);
        params[11] = new Boolean(useSSLJMXConnector);
        params[12] = new Integer(brokerStartTimeout);
        params[13] = adminUsername;
        params[14] = adminPassFile;
        params[15] = sysProps;

        _loggerL.entering(_className, "constructor()", params);

        Vector v = new Vector();

        //Add extra args first; explicit config will override args 
        if (brokerExtraArgs != null && !("".equals(brokerExtraArgs)) ) {
            StringTokenizer st = new StringTokenizer(brokerExtraArgs, " ");
            while (st.hasMoreTokens()) {
                String t = st.nextToken();
                v.add(t);
            }
        }

        //Add explicit config
        //
        v.add("-port");
        v.add(Integer.toString(brokerPort));

        if (brokerInstanceName != null) {
            v.add("-name");
            v.add(brokerInstanceName);
        }
 
        if (brokerHomeDir != null) {
            v.add("-imqhome");
            v.add(brokerHomeDir);
        }
 
        if (brokerVarDir != null) {
            v.add("-varhome");
            v.add(brokerVarDir);
        }

        if (brokerLibDir != null && !("".equals(brokerLibDir))) {
            v.add("-libhome");
            v.add(brokerLibDir);
        }

        if (useJNDIRMIServiceURL == true) {
            if (startRMIRegistry == true) {
                v.add("-startRmiRegistry");
            } else {
                v.add("-useRmiRegistry");
            }
            v.add("-rmiRegistryPort");
            v.add(Integer.toString(rmiRegistryPort));
        }


        ////XXX: use useSSLJMXConnector

        String[] brokerArgs = (String []) v.toArray(new String[0]);

        bp = BrokerProcess.getBrokerProcess(); 
 
        brokerProps = bp.convertArgs(brokerArgs);
 
        if (brokerBindAddress != null && !("localhost".equals(brokerBindAddress)) ) {
            brokerProps.setProperty("imq.hostname", brokerBindAddress);
        }

        String key;
        String val;
        //System.out.println("SJSMQRA_EB:DebugENM_BEGIN>>>>>>>>>>>>>>>>");
        for (Enumeration e = sysProps.keys() ; e.hasMoreElements() ;) {
            key = (String)e.nextElement();
            //System.out.println("SJSMQRA_EB:DebugENM_Key="+key);
            if (key != null) {
                val = (String)sysProps.getProperty(key);
                //System.out.println("SJSMQRA_EB:DebugENM_Val="+val);
                brokerProps.setProperty(key, val);
            } else {
                //System.out.println("SJSMQRA_EB:DebugENM_Val=null");
            }
        }

        //System.out.println("SJSMQRA_EB:DebugDump_BEGIN>>>>>>>>>>>>>");
        //System.out.println("SJSMQRA_EB:brokerArgs="+brokerArgs);
        //System.out.println("SJSMQRA_EB:EmbeddedBroker:brokerProps="+brokerProps.toString());
        //System.out.println("SJSMQRA_EB:EmbeddedBroker:SystemProps="+System.getProperties().toString());
        //System.out.println("SJSMQRA_EB:DebugDump___END>>>>>>>>>>>>>");

        //bt = new Thread(this, "SJSMQRA_EmbeddedBrokerRunnerThread");
        //bt.setDaemon(true);
    }

    public synchronized void
    init()
    {
        _loggerL.entering(_className, "init()");
        int code = -1;
        if (bp != null) {
            try {
                code = bp.init(true, brokerProps, (BrokerNotification)this);
            } catch (IllegalArgumentException iae) {
                _loggerL.severe(_lgrMID_EXC+"run:Broker Init Failure:IllegalArgumentException Message="+iae.getMessage());
            }
            if (code != 0) {
                throw new RuntimeException("EMBEDDED Broker init failure:code = "+code);
            } else {
                _loggerL.info(_lgrMID_INF+"run:EMBEDDED broker initialized with code ="+code);
            }
        }
    }

    public synchronized void
    start()
    {
        _loggerL.entering(_className, "start()");
        System.err.println("EB-start:brokerProps="+brokerProps.toString());
        int code = -1;
        if (bp != null) {
            try {
                code = bp.start(true, brokerProps, (BrokerNotification)this);
            } catch (IllegalArgumentException iae) {
                _loggerL.severe(_lgrMID_EXC+"run:Broker Start Failure:IllegalArgumentException Message="+iae.getMessage());
            }
            if (code != 0) {
                throw new RuntimeException("EMBEDDED Broker start failure:code = "+code);
            } else {
                _loggerL.info(_lgrMID_INF+"run:EMBEDDED broker started with code ="+code);
            }
        }
    }

    protected synchronized void
    stop()
    {
        _loggerL.entering(_className, "stop()");

        //Unkown whether the VM is about to be shutdown
        //So, indicate that the VM is *not* about to be shutdown
        //    thereby making a .start possible within the same VM
        try {
            bp.stop(true);
        } catch (Exception bse) {
            _loggerL.severe(_lgrMID_EXC+"stop:Exception on EMBEDDED broker shutdown:msg="+bse.getMessage());
            bse.printStackTrace();
        }
    }

    //com.sun.messaging.jmq.jmsserver.BrokerNotification interface methods
    /**
    * Broker would like to exit for some reason.
    * @param reason why the broker would like to exit, one of
    *              <UL><LI>REASON_SHUTDOWN</LI>
    *                  <LI>REASON_RESTART</LI>
    *                  <LI>REASON_FATAL</LI>
    *                  <LI>REASON_ERROR</LI>
    *                  <LI>REASON_EXCEPTION</LI>
    *                  <LI>REASON_STOP</LI></UL>
    * @param info string which indicates when the exit was requested
    * @param thr optional Throwable
    * @return true if the broker should clean up and exit and false
    *         if the request to exit was denied (if the request to
    *         shutdown after a fatal error is denied, the broker may not
    *         respond correctly).
    */
    public boolean
    exitRequested(int reason, String info, Throwable thr)
    {
        Object params[] = new Object[3];
        params[0] = new Integer(reason);
        params[1] = info;
        params[2] = thr;

        _loggerL.entering(_className, "exitRequested()", params);
        _loggerL.severe(_lgrMID_INF+"exitRequested:Reason="+reason+":"+info+":"+(thr == null ? "null" : thr.getMessage()));
        return true;
    }

    /**
    * Notification that the broker has shutdown.
    */
    public void
    brokerShutdown()
    {
        _loggerL.entering(_className, "brokerShutdown()");
    }

    /**
    * Notification that the broker has started and is responding to jms requests.
    */
    public void
    brokerStarted()
    {
        _loggerL.entering(_className, "brokerStarted()");
    }
}
