/*
 * @(#)SysLogHandler.java	1.2 02/03/06
 *
 * Copyright 2000 Sun Microsystems, Inc. All Rights Reserved
 *
 */

package com.sun.messaging.jmq.util.log;

import java.io.File;
import java.io.IOException;
import java.util.Properties;

import com.sun.messaging.jmq.resources.SharedResources;
import com.sun.messaging.jmq.util.StringUtil;

/**
 * A LogHandler that logs to Unix syslog
 *
 * @version	1.2
 *
 */
public class SysLogHandler extends LogHandler {

    /*
     * Apply to all instances of a SysLogHandler. This is due to
     * the way syslog works. It is designed for a process to have
     * one connection to syslogd.
     */
    static String ident = "SysLogHandler";
    static int     facility = SysLog.LOG_DAEMON;
    static int     logopt = (SysLog.LOG_PID | SysLog.LOG_CONS);

    static boolean open = false;

    public SysLogHandler() {
    }

    /**
     * Configure SysLogHandler with the values contained in
     * the passed Properties object. Note that all SysLogHandlers
     * will be affected, not just this instance. The handler's properties are
     * prefixed with the specified prefix.
     * <P>
     * An example of valid properties are:
     * <PRE>
     * imq.log.syslog.facility=LOG_DAEMON
     * imq.log.syslog.logpid=true
     * imq.log.syslog.logconsole=false
     * imq.log.syslog.identity=imqbrokerd_imqbroker
     * </PRE>
     * In this case prefix would be "jmq.log.file"
     *
     * @param props	Properties to get configuration information from
     * @param prefix	String that this handler's properties are prefixed with
     *
     * @throws IllegalArgumentException if one or more property values are
     *                                  invalid. All valid properties will
     *					still be set.
     * @throws UnsatisfiedLinkError     if native code can't be loaded
     */
    public void configure(Properties props, String prefix)
	throws IllegalArgumentException, UnsatisfiedLinkError {

	String value = null;
	String property = null;
	String error_msg = null;

	prefix = prefix + ".";

        synchronized (getClass()) {

	property = prefix + "facility";
	if ((value = props.getProperty(property)) != null) {
	    try {
		//facility = parseFacility(value);
		facility = SysLog.LOG_DAEMON;
            } catch (Exception e) {
	        //error_msg = rb.getString(rb.W_BAD_NFORMAT, property, value);
	        error_msg = "Bad syslog facility " + property + "=" + value;
            }
	}

	property = prefix + "logpid";
	if ((value = props.getProperty(property)) != null) {
            if (value.equalsIgnoreCase("true")) {
                logopt = logopt | SysLog.LOG_PID;
            }
	}

	property = prefix + "logconsole";
	if ((value = props.getProperty(property)) != null) {
            if (value.equalsIgnoreCase("true")) {
                logopt = logopt | SysLog.LOG_CONS;
            }
	}

	property = prefix + "identity";
	if ((value = props.getProperty(property)) != null) {
            ident = StringUtil.expandVariables(value, props);
	}

	property = prefix + "output"; 
	if ((value = props.getProperty(property)) != null) {
	    try {
	        setLevels(value);
	    } catch (IllegalArgumentException e) {
	        error_msg = (error_msg != null ? error_msg + "\n" : "") +
			property + ": " + e.getMessage();
	    }
        } 

        if (open) {
            this.close();
        }
        // Causes prop changes to take effect and forces link error
        this.open();

        }
    }

    /**
     * Publish string to log
     *
     * @param level	Log level to use
     * @param message	Message to write to log file
     *
     */
    public void publish(int level, String message) throws IOException {

        int priority = SysLog.LOG_INFO;

        /* 
         * Force messages are informational messages sent to all
         * log handlers. We typically don't want this type of information
         * appearing in syslog
         */
        if (level == Logger.FORCE) {
            return;
        }

        /* Syslog adds its own timestamp, strip ours */
        if (message.indexOf("[") == 0) {
            message = message.substring(message.indexOf("]") + 1);
        }

        // Map level to a syslog level
        switch (level) {
            case Logger.ERROR:   priority = SysLog.LOG_ERR; break;
            case Logger.WARNING: priority = SysLog.LOG_WARNING; break;
            case Logger.INFO:    priority = SysLog.LOG_INFO; break;
            case Logger.DEBUG:   
            case Logger.DEBUGMED:
            case Logger.DEBUGHIGH: priority = SysLog.LOG_DEBUG; break;
        }

        SysLog.syslog(priority, message);
    }

    /**
     * Open handler
     */
    public void open() {
       
        synchronized(getClass()) {
            if (!open) {
                SysLog.openlog(ident, logopt, facility);
                open = true;
            }
        }
    }

    /**
     * Close handler
     */
    public void close() {
        synchronized(getClass()) {
            if (open) {
                SysLog.closelog();
                open = false;
            }
        }
    }

    /**
     * Return a string description of this FileHandler. The descirption
     * is the class name followed by the path of the file we are logging to.
     */
    public String toString() {
	return this.getClass().getName() + ":" + ident;
    }
}
