/*
 * @(#)LogHandler.java	1.8 03/06/22
 *
 * Copyright 2000 Sun Microsystems, Inc. All Rights Reserved
 *
 */

package com.sun.messaging.jmq.util.log;

import java.io.IOException;
import java.util.StringTokenizer;
import java.util.Properties;

import com.sun.messaging.jmq.resources.SharedResources;

/**
 * Abstract class defining interface for a LogHandler. A LogHandler is
 * used by a Logger to publish log message to a logging device. A 
 * LogHandler could be implemented as a set of rolling files, or a 
 * simple output stream.
 *
 * @version	1.8
 *
 */
public abstract class LogHandler {

    /**
     * The levels of messages this handler wants to accept.
     * Log levels are described in the Logger class.
     */
    public int levels = Logger.INFO | Logger.WARNING | Logger.ERROR |
                        Logger.FORCE;

    /* Our parent Logger */
    protected Logger logger = null;

    protected static SharedResources rb = SharedResources.getResources();

    protected String name = null;

    /**
     * Convenience routine to have handler accept messages of all levels
     */
    public void acceptAllLevels() {
	levels = Logger.FORCE    |
                 Logger.ERROR	 |
    		 Logger.WARNING  |
    		 Logger.INFO     |
    		 Logger.DEBUG    |
    		 Logger.DEBUGMED |
    		 Logger.DEBUGHIGH;
    }

    /**
     * Set the log levels this handler will handle based on a String 
     * description. This is useful for setting the levels from a
     * property string.
     * 
     * @param levelList		A | seperated list of log levels this
     *				handler will accept. Valid values are
     *				"ALL", "NONE" or a list of one or more of
     *				ERROR, WARNING, INFO, DEBUG, DEBUGMED,
     *				and DEBUGHIGH
     */
    protected void setLevels(String levelList) throws IllegalArgumentException {

        String s;
        levels = 0;

        // All handles should accept forced messages (unless NONE)
        levels = Logger.FORCE;

        // Parse string and initialize levels bitmask.
	StringTokenizer token = new StringTokenizer(levelList, "|", false);
        while (token.hasMoreElements()) {
	    s = token.nextToken();
	    if (s.equals("ALL")) {
		acceptAllLevels();
		break;
            } else if (s.equals("NONE")) {
		levels = 0;
		break;
	    }

	    levels |= Logger.levelStrToInt(s);
        }

    }

    /**
     * Perform basic initialization of the LogHandler
     *
     * @param parent	Logger parent of this LogHandler
     */
    public void init(Logger parent) {
	this.logger = parent;
        acceptAllLevels();
    }

    /**
     * Set the name of this handler
     *
     * @param name  Handler name
     */
    public void setName(String name) {
        this.name = name;
    }

    /**
     * Get the name of this handler
     */
    public String getName() {
        return this.name;
    }

    abstract public void configure(Properties props, String prefix)
	throws IllegalArgumentException;
       

    /**
     * Publish string to log
     *
     * @param level	The log level
     * @param message	The message to publish to loggin device.
     */
    abstract public void publish(int level, String message) throws IOException;

    /**
     * Open handler
     */
    abstract public void open() throws IOException;


    /**
     * Close handler
     */
    abstract public void close();
}
