/*
 * @(#)JMQXid.java	1.5 05/11/30
 *
 * Copyright 2000-2001 Sun Microsystems, Inc. All Rights Reserved.
 *
 * This software is the proprietary information of Sun Microsystems, Inc.
 * Use is subject to license terms.
 *
 */
package com.sun.messaging.jmq.util;

import java.io.*;
import javax.transaction.xa.Xid;

/**
 * JMQ version of the RI's Xid Implementation. We extend XidImpl
 * to add methods for marshalling and unmarshaling an Xid to I/O Streams.
 *
 * @see javax.transaction.xa.Xid
 */
public class JMQXid extends XidImpl implements Serializable {

    // compatibility w/ 3.0.1, 3.5, 3.6
    static final long serialVersionUID = -5632229224716804510L;

    public JMQXid() {
        super();
    }

    public JMQXid(Xid foreignXid) {
        super(foreignXid);
    }

    /**
     * Write the Xid to the specified DataOutputStream. This is an
     * alternative to serialization that is faster, more compact and
     * language independent.
     *
     * The data written is guaranteed to be a fixed size. In particular
     * It will be of size
     *
     *      4 + 2 + Xid.MAXGTRIDSIZE + 2 + Xid.MAXBQUALSIZE 
     *
     * Which in practice will be 4 + 2 + 2 + 64 + 64 = 136 bytes
     *
     * If the globalTransactionId or the branchQualifierId is less than
     * MAX*SIZE bytes, then it will be padded with trailing 0's.
     * 
     * The format of the written data will be:
     *
     *<PRE>
     *    0                   1                   2                   3
     *   |0 1 2 3 4 5 6 7|8 9 0 1 2 3 4 5|6 7 8 9 0 1 2 3|4 5 6 7 8 9 0 1|
     *   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
     *   |                     format Id                                 |
     *   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
     *   | globalTransactionId Length    |   branchQualifier Length      |
     *   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
     *   |                                                               |
     *   +                                                               +
     *   |                     globalTransactionId                       |
     *   +                    MAXGTRIDSIZE bytes                         +
     *                           .  .  .
     *   +                                                               +
     *   |                                                               |
     *   +                                                               +
     *   |                                                               |
     *   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
     *   |                                                               |
     *   +                                                               +
     *   |                     branchQualifier                           |
     *   +                    MAXBQUALSIZE bytes                         +
     *                           .  .  .
     *   +                                                               +
     *   |                                                               |
     *   +                                                               +
     *   |                                                               |
     *   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
     *   </PRE>
     *
     */
    public void write(DataOutput out)
        throws IOException {

        out.writeInt(formatId);
        out.writeShort(gtLength);
        out.writeShort(bqLength);

        // These are fixed size arrays. 
        out.write(globalTxnId, 0, MAXGTRIDSIZE);
        out.write(branchQualifier, 0, MAXBQUALSIZE);
    }

    /**
     * Read the Xid from the input stream
     */
    public static JMQXid read(DataInput in)
        throws IOException {

        JMQXid xid = new JMQXid();
        xid.formatId = in.readInt();
        xid.gtLength = in.readShort();
        xid.bqLength = in.readShort();

        // These are fixed size arrays
        in.readFully(xid.globalTxnId, 0, MAXGTRIDSIZE);
        in.readFully(xid.branchQualifier, 0, MAXBQUALSIZE);

        return xid;
    }

    /**
     * Size in bytes that the object will be when marshalled.
     */
    public static int size() {
        return 8 + MAXGTRIDSIZE + MAXBQUALSIZE;
    }
}
