/*
 * @(#)TransactionDAOImpl.java	1.20 09/28/05
 *
 * Copyright 2004 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms.
 *
 */

package com.sun.messaging.jmq.jmsserver.persist.jdbc;

import com.sun.messaging.jmq.util.log.Logger;
import com.sun.messaging.jmq.util.JMQXid;
import com.sun.messaging.jmq.jmsserver.util.*;
import com.sun.messaging.jmq.jmsserver.data.TransactionState;
import com.sun.messaging.jmq.jmsserver.data.TransactionUID;
import com.sun.messaging.jmq.jmsserver.resources.*;
import com.sun.messaging.jmq.jmsserver.Globals;
import com.sun.messaging.jmq.io.Status;

import java.util.*;
import java.sql.*;
import java.io.ByteArrayInputStream;
import java.io.IOException;

/**
 * This class implement a generic TransactionDAO.
 *
 * @version	1.20
 */
class TransactionDAOImpl extends BaseDAOImpl implements TransactionDAO {

    protected String tableName;

    // SQLs
    protected String insertSQL;
    protected String updateTxnStateSQL;
    protected String updateAccessedTimeSQL;
    protected String updateBrokerSQL;
    protected String deleteSQL;
    protected String deleteNotInStateSQL;
    protected String selectTxnStateSQL;
    protected String selectAccessedTimeSQL;
    protected String selectBrokerSQL;
    protected String selectCountSQL;
    protected String selectAllStatesSQL;
    protected String selectUsageInfoSQL;
    protected String selectTakeoverSQL;
    protected String takeoverSQL;

    /**
     * Constructor
     * @throws com.sun.messaging.jmq.jmsserver.util.BrokerException
     */
    TransactionDAOImpl() throws BrokerException {

        // Initialize all SQLs
        DBManager dbMgr = DBManager.getDBManager();

        tableName = dbMgr.getTableName( TABLE_NAME_PREFIX );

        insertSQL = new StringBuffer(128)
            .append( "INSERT INTO " ).append( tableName )
            .append( " ( " )
            .append( ID_COLUMN ).append( ", " )
            .append( STATE_COLUMN ).append( ", " )
            .append( TXN_STATE_COLUMN ).append( ", " )
            .append( AUTO_ROLLBACK_COLUMN ).append( ", " )
            .append( XID_COLUMN ).append( ", " )
            .append( BROKER_ID_COLUMN ).append( ", " )
            .append( EXPIRED_TS_COLUMN ).append( ", " )
            .append( ACCESSED_TS_COLUMN )
            .append( ") VALUES ( ?, ?, ?, ?, ?, ?, ?, ? )" )
            .toString();

        updateTxnStateSQL = new StringBuffer(128)
            .append( "UPDATE " ).append( tableName )
            .append( " SET " )
            .append( STATE_COLUMN ).append( " = ?, " )
            .append( TXN_STATE_COLUMN ).append( " = ?" )
            .append( " WHERE " )
            .append( ID_COLUMN ).append( " = ?" )
            .toString();

        updateAccessedTimeSQL = new StringBuffer(128)
            .append( "UPDATE " ).append( tableName )
            .append( " SET " )
            .append( ACCESSED_TS_COLUMN ).append( " = ?" )
            .append( " WHERE " )
            .append( ID_COLUMN ).append( " = ?" )
            .toString();

        updateBrokerSQL = new StringBuffer(128)
            .append( "UPDATE " ).append( tableName )
            .append( " SET " )
            .append( BROKER_ID_COLUMN ).append( " = ?" )
            .append( " WHERE " )
            .append( ID_COLUMN ).append( " = ?" )
            .toString();

        deleteSQL = new StringBuffer(128)
            .append( "DELETE FROM " ).append( tableName )
            .append( " WHERE " )
            .append( ID_COLUMN ).append( " = ?" )
            .toString();

        deleteNotInStateSQL = new StringBuffer(128)
            .append( "DELETE FROM " ).append( tableName )
            .append( " WHERE " )
            .append( STATE_COLUMN ).append( " <> ?" )
            .toString();

        selectCountSQL = new StringBuffer(128)
            .append( "SELECT COUNT(*)" )
            .append( " FROM " ).append( tableName )
            .toString();

        selectTxnStateSQL = new StringBuffer(128)
            .append( "SELECT " )
            .append( TXN_STATE_COLUMN )
            .append( " FROM " ).append( tableName )
            .append( " WHERE " )
            .append( ID_COLUMN ).append( " = ?" )
            .toString();

        selectAccessedTimeSQL = new StringBuffer(128)
            .append( "SELECT " )
            .append( ACCESSED_TS_COLUMN )
            .append( " FROM " ).append( tableName )
            .append( " WHERE " )
            .append( ID_COLUMN ).append( " = ?" )
            .toString();

        selectBrokerSQL = new StringBuffer(128)
            .append( "SELECT " )
            .append( BROKER_ID_COLUMN )
            .append( " FROM " ).append( tableName )
            .append( " WHERE " )
            .append( ID_COLUMN ).append( " = ?" )
            .toString();

        selectAllStatesSQL = new StringBuffer(128)
            .append( "SELECT " )
            .append( ID_COLUMN ).append( ", " )
            .append( STATE_COLUMN ).append( ", " )
            .append( TXN_STATE_COLUMN )
            .append( " FROM " ).append( tableName )
            .append( " WHERE " )
            .append( BROKER_ID_COLUMN ).append( " = ?" )
            .append( " AND " )
            .append( STATE_COLUMN ).append( " <> -1" )
            .toString();

        selectUsageInfoSQL = new StringBuffer(128)
            .append( "SELECT MAX(mcount), MAX(scount) FROM (" )
            .append(    "SELECT COUNT(*) mcount, 0 scount FROM " )
            .append(    dbMgr.getTableName( MessageDAO.TABLE_NAME_PREFIX ) )
            .append(    " WHERE " )
            .append(    MessageDAO.TRANSACTION_ID_COLUMN ).append( " = ?" )
            .append( " UNION " )
            .append(    "SELECT 0 mcount, COUNT(*) scount FROM " )
            .append(    dbMgr.getTableName( ConsumerStateDAO.TABLE_NAME_PREFIX ) )
            .append(    " WHERE " )
            .append(    ConsumerStateDAO.TRANSACTION_ID_COLUMN ).append( " = ?" )
            .append( ")" )
            .toString();

        selectTakeoverSQL = new StringBuffer(128)
            .append( "SELECT " )
            .append( ID_COLUMN )
            .append( " FROM " ).append( tableName )
            .append( " WHERE " )
            .append( BROKER_ID_COLUMN ).append( " = ?" )
            .toString();

        takeoverSQL = new StringBuffer(128)
            .append( "UPDATE " ).append( tableName )
            .append( " SET " )
            .append( BROKER_ID_COLUMN ).append( " = ?" )
            .append( " WHERE " )
            .append( BROKER_ID_COLUMN ).append( " = ?" )
            .toString();
    }

    /**
     * Get the prefix name of the table.
     * @return table name
     */
    public final String getTableNamePrefix() {
        return TABLE_NAME_PREFIX;
    }

    /**
     * Get the name of the table.
     * @return table name
     */
    public String getTableName() {
        return tableName;
    }

    /**
     * Insert a new entry.
     * @param conn database connection
     * @param txnUID the transaction ID
     * @param txnState the TransactionState
     * @throws com.sun.messaging.jmq.jmsserver.util.BrokerException
     */
    public void insert( Connection conn, TransactionUID txnUID,
        TransactionState txnState ) throws BrokerException {

        long id = txnUID.longValue();

        boolean myConn = false;
        PreparedStatement pstmt = null;
        try {
            // Get a connection
            DBManager dbMgr = DBManager.getDBManager();
            if ( conn == null ) {
                conn = dbMgr.getConnection( true );
                myConn = true;
            }

            if ( hasTransaction( conn, id) ) {
                throw new BrokerException(
                    br.getKString( BrokerResources.E_TRANSACTIONID_EXISTS_IN_STORE,
                    txnUID.toString() ) );
            }

            try {
                pstmt = conn.prepareStatement( insertSQL );
                pstmt.setLong( 1, id );
                pstmt.setInt( 2, txnState.getState() );
                ByteArrayInputStream bais = Util.setObject( pstmt, 3, txnState );
                pstmt.setInt( 4, txnState.getType().intValue() );

                JMQXid jmqXid = txnState.getXid();
                if ( jmqXid != null ) {
                    pstmt.setString( 5, jmqXid.toString() );
                } else {
                    pstmt.setNull( 5, Types.VARCHAR );
                }

                pstmt.setString( 6, dbMgr.getBrokerID() );
                pstmt.setLong( 7, txnState.getExpirationTime() );
                pstmt.setLong( 8, txnState.getLastAccessTime() );
                pstmt.executeUpdate();
                bais.close();
            } catch ( Exception e ) {
                try {
                    if ( (conn != null) && !conn.getAutoCommit() ) {
                        conn.rollback();
                    }
                } catch ( SQLException rbe ) {
                    logger.log( Logger.ERROR, BrokerResources.X_DB_ROLLBACK_FAILED, rbe );
                }

                Exception ex;
                if ( e instanceof BrokerException ) {
                    throw (BrokerException)e;
                } else if ( e instanceof IOException ) {
                    ex = DBManager.wrapIOException("[" + insertSQL + "]", (IOException)e);
                } else if ( e instanceof SQLException ) {
                    ex = DBManager.wrapSQLException("[" + insertSQL + "]", (SQLException)e);
                } else {
                    ex = e;
                }

                throw new BrokerException(
                    br.getKString( BrokerResources.X_PERSIST_TRANSACTION_FAILED,
                    txnUID ), ex );
            }
        } finally {
            if ( myConn ) {
                Util.close( null, pstmt, conn );
            } else {
                Util.close( null, pstmt, null );
            }
        }
    }

    /**
     * Update the TransactionState for the specified transaction.
     * @param conn database connection
     * @param txnUID the transaction ID
     * @param txnState the new state
     * @throws com.sun.messaging.jmq.jmsserver.util.BrokerException if
     *   transaction does not exists in the store
     */
    public void updateTransactionState( Connection conn, TransactionUID txnUID,
        TransactionState txnState ) throws BrokerException {

        boolean updated = false;
        boolean myConn = false;
        PreparedStatement pstmt = null;
        try {
            // Get a connection
            if ( conn == null ) {
                conn = DBManager.getDBManager().getConnection( true );
                myConn = true;
            }

            pstmt = conn.prepareStatement( updateTxnStateSQL );
            pstmt.setInt( 1, txnState.getState() );
            ByteArrayInputStream bais = Util.setObject( pstmt, 2, txnState );
            pstmt.setLong( 3, txnUID.longValue() );
            if ( pstmt.executeUpdate() == 1 ) {
                updated = true;
            }
            bais.close();
        } catch ( Exception e ) {
            try {
                if ( (conn != null) && !conn.getAutoCommit() ) {
                    conn.rollback();
                }
            } catch ( SQLException rbe ) {
                logger.log( Logger.ERROR, BrokerResources.X_DB_ROLLBACK_FAILED, rbe );
            }

            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof IOException ) {
                ex = DBManager.wrapIOException("[" + updateTxnStateSQL + "]", (IOException)e);
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + updateTxnStateSQL + "]", (SQLException)e);
            } else {
                ex = e;
            }
            throw new BrokerException(
                br.getKString( BrokerResources.X_UPDATE_TXNSTATE_FAILED,
                txnUID ), ex );
        } finally {
            if ( myConn ) {
                Util.close( null, pstmt, conn );
            } else {
                Util.close( null, pstmt, null );
            }
        }

        if ( !updated ) {
            throw new BrokerException(
                br.getKString( BrokerResources.E_TRANSACTIONID_NOT_FOUND_IN_STORE,
                txnUID ), Status.NOT_FOUND );
        }
    }

    /**
     * Update the transaction accessed time.
     * @param conn database connection
     * @param txnUID the transaction ID
     * @param accessedTime the new timestamp
     * @throws com.sun.messaging.jmq.jmsserver.util.BrokerException if
     *   transaction does not exists in the store
     */
    public void updateAccessedTime( Connection conn, TransactionUID txnUID,
        long accessedTime ) throws BrokerException {

        boolean myConn = false;
        PreparedStatement pstmt = null;
        try {
            // Get a connection
            if ( conn == null ) {
                conn = DBManager.getDBManager().getConnection( true );
                myConn = true;
            }

            pstmt = conn.prepareStatement( updateAccessedTimeSQL );
            pstmt.setLong( 1, accessedTime );
            pstmt.setLong( 2, txnUID.longValue() );
            if ( pstmt.executeUpdate() == 0 ) {
                throw new BrokerException(
                    br.getKString( BrokerResources.E_TRANSACTIONID_NOT_FOUND_IN_STORE,
                    txnUID ), Status.NOT_FOUND );
            }
        } catch ( Exception e ) {
            try {
                if ( (conn != null) && !conn.getAutoCommit() ) {
                    conn.rollback();
                }
            } catch ( SQLException rbe ) {
                logger.log( Logger.ERROR, BrokerResources.X_DB_ROLLBACK_FAILED, rbe );
            }

            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + updateAccessedTimeSQL + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.X_PERSIST_TRANSACTION_FAILED,
                txnUID ), ex );
        } finally {
            if ( myConn ) {
                Util.close( null, pstmt, conn );
            } else {
                Util.close( null, pstmt, null );
            }
        }
    }

    /**
     * Update the broker that owns this transaction.
     * @param conn database connection
     * @param txnUID the transaction ID
     * @param brokerID the broker ID
     * @throws com.sun.messaging.jmq.jmsserver.util.BrokerException if
     *   transaction does not exists in the store
     */
    public void updateBroker( Connection conn, TransactionUID txnUID,
        String brokerID ) throws BrokerException {

        boolean updated = false;
        boolean myConn = false;
        PreparedStatement pstmt = null;
        try {
            // Get a connection
            if ( conn == null ) {
                conn = DBManager.getDBManager().getConnection( true );
                myConn = true;
            }

            pstmt = conn.prepareStatement( updateBrokerSQL );
            pstmt.setString( 1, brokerID );
            pstmt.setLong( 2, txnUID.longValue() );
            if ( pstmt.executeUpdate() > 0 ) {
                updated = true;
            }
        } catch ( Exception e ) {
            try {
                if ( (conn != null) && !conn.getAutoCommit() ) {
                    conn.rollback();
                }
            } catch ( SQLException rbe ) {
                logger.log( Logger.ERROR, BrokerResources.X_DB_ROLLBACK_FAILED, rbe );
            }

            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + updateBrokerSQL + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.X_PERSIST_TRANSACTION_FAILED,
                txnUID ), ex );
        } finally {
            if ( myConn ) {
                Util.close( null, pstmt, conn );
            } else {
                Util.close( null, pstmt, null );
            }
        }

        if ( !updated ) {
            throw new BrokerException(
                br.getKString( BrokerResources.E_TRANSACTIONID_NOT_FOUND_IN_STORE,
                txnUID ), Status.NOT_FOUND );
        }
    }

    /**
     * Delete an existing entry.
     * @param conn database connection
     * @param txnUID the transaction ID
     * @throws com.sun.messaging.jmq.jmsserver.util.BrokerException
     */
    public void delete( Connection conn, TransactionUID txnUID )
        throws BrokerException {

        boolean updated = false;
        boolean myConn = false;
        PreparedStatement pstmt = null;
        try {
            // Get a connection
            DBManager dbMgr = DBManager.getDBManager();
            if ( conn == null ) {
                conn = dbMgr.getConnection( false );
                myConn = true;
            }

            pstmt = conn.prepareStatement( deleteSQL );
            pstmt.setLong( 1, txnUID.longValue() );
            if ( pstmt.executeUpdate() > 0 ) {
                updated = true;

                // For HA, there is an edge case where msgs is being redelivered
                // after the takeover eventhough the txn has been committed.
                // What happen is the broker has committed the txn but didn't
                // has a chance to ack or deleted the msgs and it crashed.
                // So to be on the safe side, we'll delete all consumer states
                // for this txn when it is committed, i.e. translate to txn is
                // being removed from data store.
                dbMgr.getDAOFactory().getConsumerStateDAO().deleteByTransaction( conn, txnUID );
            }

            // Commit all changes
            if ( myConn ) {
                conn.commit();
            }            
        } catch ( Exception e ) {
            try {
                if ( (conn != null) && !conn.getAutoCommit() ) {
                    conn.rollback();
                }
            } catch ( SQLException rbe ) {
                logger.log( Logger.ERROR, BrokerResources.X_DB_ROLLBACK_FAILED, rbe );
            }

            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + deleteSQL + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.X_REMOVE_TRANSACTION_FAILED,
                txnUID ), ex );
        } finally {
            if ( myConn ) {
                Util.close( null, pstmt, conn );
            } else {
                Util.close( null, pstmt, null );
            }
        }

        if ( !updated ) {
            throw new BrokerException(
                br.getKString( BrokerResources.E_TRANSACTIONID_NOT_FOUND_IN_STORE,
                txnUID ), Status.NOT_FOUND );
        }
    }

    /**
     * Delete all entries.
     * @param conn database connection
     * @throws com.sun.messaging.jmq.jmsserver.util.BrokerException
     */
    public void deleteAll( Connection conn )
        throws BrokerException {

        String whereClause = null;
        if ( Globals.getHAEnabled() ) {
            // Only delete transactions that belong to the running broker,
            // construct the where clause for the delete statement:
            //   DELETE FROM mqtxn40cmycluster WHERE broker_id = 'mybroker'
            whereClause = new StringBuffer(128)
                .append( BROKER_ID_COLUMN ).append( " = '")
                .append( DBManager.getDBManager().getBrokerID() ).append( "'" )
                .toString();
        }

        deleteAll( conn, whereClause, null, 0 );
    }

    /**
     * Take over the transactions.
     * @param conn database connection
     * @param brokerID the current or local broker ID
     * @param targetBrokerID the broker ID of the store being taken over
     * @return a List of all transactions the target broker owns
     * @throws BrokerException
     */
    public List takeover( Connection conn, String brokerID, String targetBrokerID )
        throws BrokerException {

        List list = new ArrayList(100);

        String sql = null;
        PreparedStatement pstmt = null;
        try {
            // First retrieve all transactions for the target broker
            sql = selectTakeoverSQL;
            pstmt = conn.prepareStatement( sql );
            pstmt.setString( 1, targetBrokerID );
            ResultSet rs = pstmt.executeQuery();
            while ( rs.next() ) {
                list.add( new TransactionUID( rs.getLong( 1 ) ) );
            }

            Util.close( rs, pstmt, null );

            // Now takeover those transactions
            sql = takeoverSQL;
            pstmt = conn.prepareStatement( sql );
            pstmt.setString( 1, brokerID );
            pstmt.setString( 2, targetBrokerID );
            int count = pstmt.executeUpdate();

            // Verify that we are able to takeover all transactions
            if ( count != list.size() ) {
                // This shouldn't occur but just being safe
                String[] args = { targetBrokerID,
                                  String.valueOf( list.size() ),
                                  String.valueOf( count ) };
                throw new BrokerException(
                    br.getKString( BrokerResources.E_TAKEOVER_TXN_FAILED, args ) );
            }
        } catch ( Exception e ) {
            try {
                if ( (conn != null) && !conn.getAutoCommit() ) {
                    conn.rollback();
                }
            } catch ( SQLException rbe ) {
                logger.log( Logger.ERROR, BrokerResources.X_DB_ROLLBACK_FAILED, rbe );
            }

            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } if ( e instanceof IOException ) {
                ex = DBManager.wrapIOException("[" + sql + "]", (IOException)e);
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + sql + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.E_TAKEOVER_TXN_2_FAILED,
                    targetBrokerID ), ex );
        } finally {
            Util.close( null, pstmt, null );
        }

        return list;
    }

    /**
     * Get the TransactionState object.
     * @param conn database connection
     * @param txnUID the transaction ID
     * @return TransactionState object
     * @throws BrokerException
     */
    public TransactionState getTransactionState( Connection conn,
        TransactionUID txnUID ) throws BrokerException {

        TransactionState txnState = null;
        long id = txnUID.longValue();

        boolean myConn = false;
        PreparedStatement pstmt = null;
        ResultSet rs = null;
        try {
            // Get a connection
            if ( conn == null ) {
                conn = DBManager.getDBManager().getConnection( true );
                myConn = true;
            }

            pstmt = conn.prepareStatement( selectTxnStateSQL );
            pstmt.setLong( 1, id );
            rs = pstmt.executeQuery();
            if ( rs.next() ) {
                try {
                    txnState = (TransactionState)Util.readObject( rs, 1 );
                } catch ( IOException e ) {
                    // fail to parse TransactionState object; just log it
                    logger.logStack( Logger.ERROR,
                        BrokerResources.X_PARSE_TRANSACTION_FAILED, e );
                }
            } else {
                throw new BrokerException(
                    br.getKString( BrokerResources.E_TRANSACTIONID_NOT_FOUND_IN_STORE,
                    String.valueOf( id ) ), Status.NOT_FOUND );
            }
        } catch ( Exception e ) {
            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + selectTxnStateSQL + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.X_LOAD_TRANSACTION_FAILED,
                    txnUID ), ex );
        } finally {
            if ( myConn ) {
                Util.close( rs, pstmt, conn );
            } else {
                Util.close( rs, pstmt, null );
            }
        }

        return txnState;
    }

    /**
     * Get the time when the transaction was last accessed.
     * @param conn database connection
     * @param txnUID the transaction ID
     * @return Timestamp when the transaction was last accessed
     * @throws BrokerException
     */
    public long getAccessedTime( Connection conn, TransactionUID txnUID )
        throws BrokerException {

        long accessedTime = -1;
        long id = txnUID.longValue();

        boolean myConn = false;
        PreparedStatement pstmt = null;
        ResultSet rs = null;
        try {
            // Get a connection
            if ( conn == null ) {
                conn = DBManager.getDBManager().getConnection( true );
                myConn = true;
            }

            pstmt = conn.prepareStatement( selectAccessedTimeSQL );
            pstmt.setLong( 1, id );
            rs = pstmt.executeQuery();
            if ( rs.next() ) {
                accessedTime = rs.getLong( 1 );
            } else {
                throw new BrokerException(
                    br.getKString( BrokerResources.E_TRANSACTIONID_NOT_FOUND_IN_STORE,
                    String.valueOf( id ) ), Status.NOT_FOUND );
            }
        } catch ( Exception e ) {
            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + selectAccessedTimeSQL + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.X_LOAD_TRANSACTION_FAILED,
                    txnUID ), ex );
        } finally {
            if ( myConn ) {
                Util.close( rs, pstmt, conn );
            } else {
                Util.close( rs, pstmt, null );
            }
        }

        return accessedTime;
    }

    /**
     * Get the broker ID that owns the transaction.
     * @param conn database connection
     * @param txnUID the transaction ID
     * @return Broker ID that owns the transaction
     * @throws BrokerException
     */
    public String getBroker( Connection conn, TransactionUID txnUID )
        throws BrokerException {

        String brokerID = null;
        long id = txnUID.longValue();

        boolean myConn = false;
        PreparedStatement pstmt = null;
        ResultSet rs = null;
        try {
            // Get a connection
            if ( conn == null ) {
                conn = DBManager.getDBManager().getConnection( true );
                myConn = true;
            }

            pstmt = conn.prepareStatement( selectBrokerSQL );
            pstmt.setLong( 1, id );
            rs = pstmt.executeQuery();
            if ( rs.next() ) {
                brokerID = rs.getString( 1 );
            } else {
                throw new BrokerException(
                    br.getKString( BrokerResources.E_TRANSACTIONID_NOT_FOUND_IN_STORE,
                    String.valueOf( id ) ), Status.NOT_FOUND );
            }
        } catch ( Exception e ) {
            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + selectBrokerSQL + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.X_LOAD_TRANSACTION_FAILED,
                    txnUID ), ex );
        } finally {
            if ( myConn ) {
                Util.close( rs, pstmt, conn );
            } else {
                Util.close( rs, pstmt, null );
            }
        }

        return brokerID;
    }

    /**
     * Retrieve all transaction IDs owned by a broker.
     * @param conn database connection
     * @param brokerID the broker ID
     * @return a List of TransactionUID objects; an empty List is returned
     * if the broker does not have any transactions
     */
    public List getTransactionsByBrokerID( Connection conn, String brokerID )
        throws BrokerException {

        List list = new ArrayList();

        boolean myConn = false;
        PreparedStatement pstmt = null;
        ResultSet rs = null;
        try {
            // Get a connection
            if ( conn == null ) {
                conn = DBManager.getDBManager().getConnection( true );
                myConn = true;
            }

            pstmt = conn.prepareStatement( selectTakeoverSQL );
            pstmt.setString( 1, brokerID );
            rs = pstmt.executeQuery();
            while ( rs.next() ) {
                list.add( new TransactionUID( rs.getLong( 1 ) ) );
            }
        } catch ( Exception e ) {
            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + selectTakeoverSQL + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.X_LOAD_TXNS_FOR_BROKER_FAILED,
                    brokerID ), ex );
        } finally {
            if ( myConn ) {
                Util.close( rs, pstmt, conn );
            } else {
                Util.close( rs, pstmt, null );
            }
        }

        return list;
    }

    /**
     * Retrieve all transactions.
     * @param conn database connection
     * @return a List of Destination objects; an empty Map is returned
     * if no transactions exist in the store
     */
    public HashMap getAllTransactionStates( Connection conn ) throws BrokerException {

        HashMap map = new HashMap();

        boolean myConn = false;
        PreparedStatement pstmt = null;
        ResultSet rs = null;
        try {
            // Get a connection
            DBManager dbMgr = DBManager.getDBManager();
            if ( conn == null ) {
                conn = dbMgr.getConnection( true );
                myConn = true;
            }

            pstmt = conn.prepareStatement( selectAllStatesSQL );
            pstmt.setString( 1, dbMgr.getBrokerID() );
            rs = pstmt.executeQuery();

            while ( rs.next() ) {
                try {
                    long id = rs.getLong( 1 );
                    int state = rs.getInt( 2 );
                    TransactionState txnState =
                        (TransactionState)Util.readObject( rs, 3 );

                    // update state in TransactionState object
                    txnState.setState( state );

                    map.put( new TransactionUID( id ), txnState );
                } catch ( IOException e ) {
                    // fail to parse destination object; just log it
                    logger.logStack( Logger.ERROR,
                        BrokerResources.X_PARSE_TRANSACTION_FAILED, e );
                }
            }
        } catch ( Exception e ) {
            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + selectAllStatesSQL + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.X_LOAD_TRANSACTIONS_FAILED ), ex );
        } finally {
            if ( myConn ) {
                Util.close( rs, pstmt, conn );
            } else {
                Util.close( rs, pstmt, null );
            }
        }

        return map;
    }

    /**
     * Return the number of messages and the number of consumer states that
     * that associate with the specified transaction ID.
     * @param conn database connection
     * @param txnUID the transaction ID
     * @return an array of int whose first element contains the number of messages
     * and the second element contains the number of consumer states.
     */
    public int[] getTransactionUsageInfo( Connection conn, TransactionUID txnUID )
        throws BrokerException {

        int[] data = { 0, 0 };

        long id = txnUID.longValue();

        boolean myConn = false;
        PreparedStatement pstmt = null;
        ResultSet rs = null;
        try {
            // Get a connection
            if ( conn == null ) {
                conn = DBManager.getDBManager().getConnection( true );
                myConn = true;
            }

            pstmt = conn.prepareStatement( selectUsageInfoSQL );
            pstmt.setLong( 1, id );
            pstmt.setLong( 2, id );
            rs = pstmt.executeQuery();
            if ( rs.next() ) {
                data[0] = rs.getInt( 1 );   // # of messages
                data[1] = rs.getInt( 2 );   // # of consumer states
            }
        } catch ( Exception e ) {
            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + selectUsageInfoSQL + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.X_LOAD_TRANSACTION_FAILED,
                    txnUID ), ex );
        } finally {
            if ( myConn ) {
                Util.close( rs, pstmt, conn );
            } else {
                Util.close( rs, pstmt, null );
            }
        }

        return data;
    }

    /**
     * Check whether the specified transaction exists.
     * @param conn database connection
     * @param id transaction ID
     * @return return true if the specified transaction exists
     */
    public boolean hasTransaction( Connection conn, long id )
        throws BrokerException {

        boolean found = false;

        boolean myConn = false;
        PreparedStatement pstmt = null;
        ResultSet rs = null;
        try {
            // Get a connection
            if ( conn == null ) {
                conn = DBManager.getDBManager().getConnection( true );
                myConn = true;
            }

            pstmt = conn.prepareStatement( selectAccessedTimeSQL );
            pstmt.setLong( 1, id );
            rs = pstmt.executeQuery();
            if ( rs.next() ) {
                found = true;
            }
        } catch ( Exception e ) {
            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + selectAccessedTimeSQL + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.X_LOAD_TRANSACTION_FAILED, 
                    String.valueOf(id) ), ex );
        } finally {
            if ( myConn ) {
                Util.close( rs, pstmt, conn );
            } else {
                Util.close( rs, pstmt, null );
            }
        }

        return found;
    }

    /**
     * Check whether the specified transaction exists.
     * @param conn database connection
     * @param id transaction ID
     * @throws com.sun.messaging.jmq.jmsserver.util.BrokerException if the
     * transaction does not exists in the store
     */
    public void checkTransaction( Connection conn, long id ) throws BrokerException {

        if ( !hasTransaction( conn, id ) ) {
            throw new BrokerException(
                br.getKString( BrokerResources.E_TRANSACTIONID_NOT_FOUND_IN_STORE,
                String.valueOf( id ) ), Status.NOT_FOUND );
        }
    }

    /**
     * Get debug information about the store.
     * @param conn database connection
     * @return a HashMap of name value pair of information
     */
    public HashMap getDebugInfo( Connection conn ) {

        HashMap map = new HashMap();
        int size = -1;

        boolean myConn = false;
        PreparedStatement pstmt = null;
        ResultSet rs = null;
        try {
            // Get a connection
            if ( conn == null ) {
                conn = DBManager.getDBManager().getConnection( true );
                myConn = true;
            }

            pstmt = conn.prepareStatement( selectCountSQL );
            rs = pstmt.executeQuery();

            if ( rs.next() ) {
                size = rs.getInt( 1 );
            }
        } catch ( Exception e ) {
            logger.log( Logger.ERROR, BrokerResources.X_JDBC_QUERY_FAILED,
                selectCountSQL, e );
        } finally {
            try {
                if ( myConn ) {
                    Util.close( rs, pstmt, conn );
                } else {
                    Util.close( rs, pstmt, null );
                }
            } catch ( BrokerException be ) {
                logger.log( Logger.ERROR, be.getMessage(), be.getCause() );
            }
        }

        map.put( "Transactions(" + tableName + ")", String.valueOf( size ) );
        return map;
    }
}
