/*
 * @(#)MessageDAO.java	1.13 08/31/05
 *
 * Copyright 2004 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms.
 *
 */

package com.sun.messaging.jmq.jmsserver.persist.jdbc;

import com.sun.messaging.jmq.jmsserver.util.BrokerException;
import com.sun.messaging.jmq.jmsserver.core.ConsumerUID;
import com.sun.messaging.jmq.jmsserver.core.Destination;
import com.sun.messaging.jmq.jmsserver.core.DestinationUID;
import com.sun.messaging.jmq.jmsserver.data.TransactionUID;
import com.sun.messaging.jmq.io.Packet;
import com.sun.messaging.jmq.io.SysMessageID;

import java.sql.Connection;
import java.io.IOException;
import java.util.List;
import java.util.Enumeration;
import java.util.HashMap;

/**
 * This class is an interface for the Message table which will be implemented
 * by database specific code.
 *
 * @version	1.13
 */
public interface MessageDAO extends BaseDAO {

    /**
     * Message table:
     * Holds persisted messages.
     *
     * CREATE TABLE IMQMSG<schemaVersion>[C<clusterID>|S<brokerID>] (
     *      ID              VARCHAR(100),
     *      SYSMESSAGE_ID   LONGVARBINARY,
     *      MESSAGE         LONGVARBINARY,
     *      MESSAGE_SIZE    BIGINT,
     *      BROKER_ID	    VARCHAR(100),
     *      DESTINATION_ID  VARCHAR(100),
     *      TRANSACTION_ID  BIGINT,
     *      CREATED_TS      BIGINT,
     *      PRIMARY KEY(ID)
     * );
     *
     * ID - SysMessageID for the message
     * SYSMESSAGE_ID - Serialized SysMessageID object
     * MESSAGE - Wire format of message packet
     * MESSAGE_SIZE - Byte count of this message
     * BROKER_ID - Unique name of the Broker responsible for routing the message
     * DESTINATION_ID - Unique name of the Destination of the message
     * TRANSACTION_ID - Transaction ID associated with an acknowledgement
     * 	(Sent when a message has sent in a transaction but not committed)
     * CREATED_TS - Timestamp when the message is created
     */
    public static final String TABLE_NAME_PREFIX = "MQMSG" + DBConstants.SCHEMA_VERSION;
    public static final String ID_COLUMN = "ID";
    public static final String SYSMESSAGE_ID_COLUMN = "SYSMESSAGE_ID";
    public static final String MESSAGE_COLUMN = "MESSAGE";
    public static final String MESSAGE_SIZE_COLUMN = "MESSAGE_SIZE";
    public static final String BROKER_ID_COLUMN = "BROKER_ID";
    public static final String DESTINATION_ID_COLUMN = "DESTINATION_ID";
    public static final String TRANSACTION_ID_COLUMN = "TRANSACTION_ID";
    public static final String CREATED_TS_COLUMN = "CREATED_TS";

    void insert( Connection conn, Packet message, ConsumerUID[] consumerUIDs, 
        int[] states, String brokerID, long createdTime, boolean checkMsgExist )
        throws BrokerException;

    void moveMessage( Connection conn, Packet message,
        DestinationUID from, DestinationUID to, ConsumerUID[] consumerUIDs,
        int[] states ) throws IOException, BrokerException;

    void delete( Connection conn, SysMessageID sysMsgID,
        DestinationUID dstUID ) throws BrokerException;

    void deleteByDestination( Connection conn, Destination dst )
        throws BrokerException;

    List takeover( Connection conn, String brokerID, String targetBrokerID )
        throws BrokerException;

    Packet getMessage( Connection conn, SysMessageID sysMsgID )
        throws BrokerException;

    Packet getMessage( Connection conn, String id ) throws BrokerException;

    String getBroker( Connection conn, String id ) throws BrokerException;

    List getMessagesByDestination( Connection conn, Destination dst )
        throws BrokerException;

    Enumeration messageEnumeration( Destination dst ) throws BrokerException;

    boolean hasMessageBeenAcked( Connection conn, SysMessageID sysMsgID )
        throws BrokerException;
    
    boolean hasMessage( Connection conn, String id ) throws BrokerException;

    void checkMessage( Connection conn, String id ) throws BrokerException;

    int getMessageCount( Connection conn, String brokerID )
        throws BrokerException;

    HashMap getMessageStorageInfo( Connection conn, Destination dst )
        throws BrokerException;
}
