/*
 * @(#)DBConnectionPool.java	1.6 09/15/05
 *
 * Copyright 2004 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms. 
 *
 */

package com.sun.messaging.jmq.jmsserver.persist.jdbc;

import com.sun.messaging.jmq.util.log.Logger;
import com.sun.messaging.jmq.jmsserver.Globals;
import com.sun.messaging.jmq.jmsserver.resources.*;
import com.sun.messaging.jmq.jmsserver.persist.Store;
import com.sun.messaging.jmq.jmsserver.util.BrokerException;

import java.sql.*;
import java.util.*;

/**
 * The DBConnection class represents a pool of connections to one database.
 */
public class DBConnectionPool implements DBConstants {

    public static final String NUM_CONN_PROP =
        DBManager.JDBC_PROP_PREFIX + "connection.limit";
    static final String USE_TRUNCATE_PROP =
        DBManager.JDBC_PROP_PREFIX + "sql.useTruncate";
    static final int DEFAULT_NUM_CONN = 5;

    private static int numConnections = DEFAULT_NUM_CONN;

    private static boolean initialized = false;
    private static Object lock = new Object();

    private static LinkedList idleConnections = new LinkedList();
    private static HashSet activeConnections = new HashSet();

    private static DBManager dbmgr = null;
    private static Logger logger = Globals.getLogger();
    private static BrokerResources br = Globals.getBrokerResources();

    /**
     * Establish a pool of database connections.
     */
    static void init() throws BrokerException {

        if ( !initialized ) {
            synchronized (lock) {
                if ( initialized ) {
                    return;
                }

                dbmgr = DBManager.getDBManager();

                numConnections = Globals.getConfig().getIntProperty(
                    NUM_CONN_PROP, DEFAULT_NUM_CONN);

                if (numConnections < 1) {
                    numConnections = DEFAULT_NUM_CONN;
                    logger.log(Logger.DEBUG,
                        "Bad number of connection specified, set to default of 5");
                }

                // With embedded DB, if autocreate store is enabled then we
                // need to create the DB now; otherwise we run into a chicken and
                // egg problem because we will not be able to create a connection
                // to check if the store exists.               
                if (dbmgr.getCreateDBURL() != null &&
                    Globals.getConfig().getBooleanProperty(Store.CREATE_STORE_PROP, false)) {
                    try {
                        Connection conn = dbmgr.connectToCreate();
                        conn.close();
                    } catch (Exception e) {
                        String url = dbmgr.getCreateDBURL();
                        logger.log(Logger.ERROR, BrokerResources.E_CREATE_DATABASE_TABLE_FAILED, url, e);
                        throw new BrokerException(br.getString(
                            BrokerResources.E_CREATE_DATABASE_TABLE_FAILED, url, e));
                    }
                }

                for (int i = 0; i < numConnections; i++) {
                    Connection conn = dbmgr.newConnection(true);
                    idleConnections.add(conn);
                }

                initialized = true;
            }
        }
    }

    /**
     * Closes all available connections. Should be called when the broker is
     * shutting down and all store operations are done so there should not
     * be any connection in the activeConnections list.
     */
    static void reset() {

        if (!initialized) {
	    return;
        }

        synchronized (lock) {
            // Close all connections
            Iterator itr = idleConnections.iterator();
            while (itr.hasNext()) {
                Connection c = (Connection)itr.next();
                try {
                    c.close();
                } catch (SQLException e) {
                    logger.log(Logger.WARNING, BrokerResources.X_CLOSE_DATABASE_FAILED, e);
                }
            }

            idleConnections.clear();
            initialized = false;
        }
    }

    /**
     * Checks out a connection from the pool.
     * @throws BrokerException
     */
    static Connection getConnection() throws BrokerException {

        synchronized (lock) {
	    while (idleConnections.isEmpty()) {
		try {
		    lock.wait();
		} catch (Exception e) {
		    logger.log(Logger.DEBUG, "DBConnectionPool.getConnection(): " + e);
		}
	    }

	    Connection conn = (Connection)idleConnections.removeFirst();

            // Check if connection is closed
            boolean isClosed = false;
            try {
                isClosed = conn.isClosed();
            } catch (SQLException e) {}

	    if ( isClosed ) {
		// if we have an invalid connection, reconnect
		try {
                    Connection newconn = dbmgr.newConnection(true);
		    conn = newconn;

		    logger.log(Logger.INFO, br.getString(
                        BrokerResources.I_RECONNECT_TO_DB, dbmgr.getOpenDBURL()));

		} catch (BrokerException e) {
		    logger.log(Logger.ERROR,
			br.getString(BrokerResources.X_RECONNECT_TO_DB_FAILED,
                            dbmgr.getOpenDBURL()), e);

		    // just stick the bad connection back to the idle list
		    // so that we will try to reconnect again
		    idleConnections.add(conn);
		    throw e;
		}
	    }

	    // move the connection in the activeConnections list
	    activeConnections.add(conn);

            if (Store.DEBUG) {
	        logger.log(Logger.DEBUG,
                    "DBConnectionPool.getConnection(): check out connection: " +
                    conn);
            }

	    return conn;
	}
    }

    /**
     * Checks in a connection to the pool.
     */
    static void freeConnection(Connection conn) {

        if (Store.DEBUG) {
	    logger.log(Logger.DEBUG,
		"DBConnectionPool.freeConnection(): check in connection: " + conn);
        }

	synchronized (lock) {
	    activeConnections.remove(conn);
	    idleConnections.add(conn);
	    lock.notify();
	}
    }

    /**
     * The validation check runs a simple SQL query over the connection
     * to see if it throws an exception.
     */
    static boolean validate( Connection conn ) {

        try {
            conn.getMetaData();
        } catch (Exception e) {
            logger.log(Logger.DEBUG,
                "DBConnectionPool.validate(): Lost database connection to "
                + dbmgr.getOpenDBURL(), e);
            return false;
        }
        return true;
    }

    // returning true indicates that the caller should retry the operation
    // will return false if the connection is still good
    static boolean handleException(Connection conn, Throwable t) {

        // test to see if the connection is good
        if ( validate( conn ) ) {

            logger.log(Logger.DEBUG,
                "connection is good; return false (no need to retry)");

            // successful so connection is ok
            return false; // no retry
        }

        return true;
    }
}
