/*
 * @(#)FileStore.java	1.109 12/02/05
 *
 * Copyright 2004 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL. Use is subject to license terms. 
 *
 */

package com.sun.messaging.jmq.jmsserver.persist.file;

import com.sun.messaging.jmq.io.SysMessageID;
import com.sun.messaging.jmq.io.Packet;
import com.sun.messaging.jmq.io.DestMetricsCounters;
import com.sun.messaging.jmq.util.log.Logger;
import com.sun.messaging.jmq.util.FileUtil;
import com.sun.messaging.jmq.jmsserver.core.*;
import com.sun.messaging.jmq.jmsserver.data.TransactionAcknowledgement;
import com.sun.messaging.jmq.jmsserver.data.TransactionUID;
import com.sun.messaging.jmq.jmsserver.data.TransactionState;
import com.sun.messaging.jmq.jmsserver.util.*;
import com.sun.messaging.jmq.jmsserver.Globals;
import com.sun.messaging.jmq.jmsserver.Broker;
import com.sun.messaging.jmq.jmsserver.cluster.BrokerState;
import com.sun.messaging.jmq.jmsserver.resources.*;
import com.sun.messaging.jmq.jmsserver.persist.*;

import java.io.*;
import java.util.*;

/**
 * FileStore provides file based persistence.
 * <br>
 * Note that some methods are NOT synchronized.
 *
 * @version	1.109
 */
public class FileStore extends Store {

    // current version of store
    public static final int OLD_STORE_VERSION_200 = 200;
    public static final int OLD_STORE_VERSION = 350;
    public static final int STORE_VERSION = 370;

    /**
     * properties used:
     *
     * - whether to sync persistent operations; default is false
     * jmq.persist.file.sync.enable=[true|false]
     *
     * - whether to sync all operations, including those operations that
     *   truncate a file or tag a file 'FREE' are not sync'ed.
     * jmq.persist.file.sync.all = [true|false]	# private
     */
    static final String FILE_PROP_PREFIX = Globals.IMQ + ".persist.file.";

    static final String SYNC_ENABLED_PROP
				= FILE_PROP_PREFIX + "sync.enabled";

    static final String SYNC_ALL_PROP
				= FILE_PROP_PREFIX + "sync.all";

    // default for jmq.persist.file.sync.enabled
    static final boolean DEFAULT_SYNC_ENABLED = false;

    // default for jmq.persist.file.sync.all
    static final boolean DEFAULT_SYNC_ALL = false;

    /**
     * directory hierarchy of the file based persistent store
     * "fs370" - top of directory hierarchy, 'fs' + version #
     * "message" - directory to store messages, one message per file with
     *		   numeric file names
     * "config" - directory to store cluster configuration information
     * "interest" - file under "fs370" to store interest objects
     * "destination" - a file under "fs370" to store destinations
     * "property" - a file under "fs370" to store name/value pairs
     * "tid" - a file under "fs370" to store transaction ids
     * "txnack" - files under "fs370" to store transaction
     *			acknowledgements of all transaction ids
     */
    static final String FILESTORE_BASENAME = "fs";
    static final String FILESTORE_TOP =
			FILESTORE_BASENAME + STORE_VERSION;

    // root directory of old (350) file store
    static final String FILESTORE350_TOP =
                        FILESTORE_BASENAME + OLD_STORE_VERSION;

    // root directory of old (200) file store
    static final String FILESTORE200_TOP = "filestore";

    // version file in old (200) file store
    static final String VERSIONFILE = "version";

    /**
     * static varibles used by all other classes in the package.
     */
    // whether to sync data-writing operations
    static boolean syncEnabled = false;

    // whether to sync all write operations
    static boolean syncAll = false;

    /**
     * Instance variables
     */

    // root directory of the file store
    private File rootDir = null;

    // object encapsulates persistence of messages and their interest lists
    private MsgStore msgStore = null;

    // object encapsulates persistence of interests
    private InterestStore intStore = null;

    // object encapsulates persistence of destinations
    private DestinationList dstList = null;

    // object encapsulates persistence of transaction ids
    private TidList tidList = null;

    // object encapsulates persistence of configuration change record
    private ConfigChangeRecord configStore = null;

    // object encapsulates persistence of properties
    private PropertiesFile propFile = null;

    // object encapsulates persistence of ha broker info
    private HABrokers haBrokersStore = null;

    /**
     * When instantiated, the object configures itself by reading the
     * properties specified in BrokerConfig.
     */
    public FileStore() throws BrokerException {

	// get the jmq.persist.file.sync.enabled and
	// jmq.persist.file.sync.all properties
	syncEnabled = config.getBooleanProperty(SYNC_ENABLED_PROP,
						DEFAULT_SYNC_ENABLED);

    	String fstop = Globals.JMQ_INSTANCES_HOME + File.separator +
                Globals.getConfigName() + File.separator;
    	logger.logToAll(Logger.INFO, br.getString(BrokerResources.I_FILE_STORE_INFO, fstop));
				
	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG, SYNC_ENABLED_PROP+"="+syncEnabled);
	}

	syncAll = config.getBooleanProperty(SYNC_ALL_PROP, DEFAULT_SYNC_ALL);

	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG, SYNC_ALL_PROP+"="+syncAll);
	}

	// instance root directory
	String instancename = Globals.JMQ_INSTANCES_HOME + File.separator +
			Globals.getConfigName() + File.separator;

	File instanceDir = new File(instancename);

	// check file store & determine whether we need to upgrade an old store
	int upgrade = checkFileStore(instanceDir);

	// the file store root directory
	rootDir = new File(instanceDir, FILESTORE_TOP);

	// check if we need to remove the store
	if (removeStore) {
	    try {
		// remove everything and return
		FileUtil.removeFiles(rootDir, true);
		return;
	    } catch (IOException e) {
		logger.log(Logger.ERROR,
                    BrokerResources.E_REMOVE_STORE_FAILED, rootDir, e);
		throw new BrokerException(br.getString(
                    BrokerResources.E_REMOVE_STORE_FAILED, rootDir), e);
	    }
	} else if (!rootDir.exists() && !rootDir.mkdirs()) {
	    logger.log(Logger.ERROR,
                BrokerResources.E_CANNOT_CREATE_STORE_HIERARCHY, rootDir);
	    throw new BrokerException(br.getString(
                BrokerResources.E_CANNOT_CREATE_STORE_HIERARCHY, rootDir));
	}

        if (upgrade > 0) {
            File oldRoot = null; // Old file store root dir, i.e. 350 or 200!
	    try {
		// log message to do upgrade
		Object[] args = {(new Integer(STORE_VERSION)), FILESTORE_TOP};
		logger.logToAll(Logger.INFO,
                    BrokerResources.I_UPGRADE_STORE_IN_PROGRESS, args);

		if (resetMessage) {
		    // log message to remove old message
		    logger.logToAll(Logger.INFO,
                        BrokerResources.I_RESET_MESSAGES_IN_OLD_STORE);
		    logger.logToAll(Logger.INFO,
                        BrokerResources.I_UPGRADE_REMAINING_STORE_DATA);
		} else if (resetInterest) {
		    // log message to remove old interest
		    logger.logToAll(Logger.INFO,
                        BrokerResources.I_RESET_INTERESTS_IN_OLD_STORE);
		    logger.logToAll(Logger.INFO,
                        BrokerResources.I_UPGRADE_REMAINING_STORE_DATA);
		}

                if (upgrade == OLD_STORE_VERSION) {
                    // Upgrading from 350 to 370, we copies all files under
                    // fs350 to fs370 and migrate txn and txn acks table
                    oldRoot = new File(instanceDir, FILESTORE350_TOP);

                    try {
                        FileUtil.copyDirectory(oldRoot, rootDir);
                        FileUtil.removeFiles(new File(rootDir, TidList.BASENAME), false);
                        FileUtil.removeFiles(new File(rootDir, TxnAckList.BASENAME), false);
                    } catch (IOException e) {
                        // Upgrade failed - unable to copy files
                        String errorMsg =
                            "Failed to copy old persistent data under " +
                            oldRoot + " to " + rootDir;
                        logger.log(Logger.ERROR, errorMsg, e);
                        throw new BrokerException(errorMsg, e);
                    }

                    // always load destinations first
                    dstList = new DestinationList(this, rootDir, false);

                    msgStore = new MsgStore(this, rootDir, resetMessage);

                    intStore = new InterestStore(rootDir, resetInterest);

                    // transaction ids and associated ack lists
                    tidList = new TidList(this, rootDir, oldRoot);

                    // configuration change record
                    configStore = new ConfigChangeRecord(rootDir, false);

                    // properties
                    propFile = new PropertiesFile(rootDir, false);
                } else {
                    // Upgrading from 200 to 370
                    oldRoot = new File(instanceDir, FILESTORE200_TOP);

                    // always load destinations first
                    dstList = new DestinationList(this, rootDir, oldRoot);

                    msgStore = new MsgStore(this, rootDir, oldRoot,
                        resetMessage);

                    intStore = new InterestStore(this, rootDir, oldRoot,
                        resetInterest);

                    // transaction ids and associated ack lists
                    // Note: for 370, txn and txnack is externalize instead of
                    // serialize so the upgrade strategy is to drop old data!
                    tidList = new TidList(rootDir, true);

                    // properties
                    propFile = new PropertiesFile(this, rootDir, oldRoot);

                    // configuration change record
                    configStore = new ConfigChangeRecord(this, rootDir, oldRoot);
                }

                if (Store.DEBUG) {
                    logger.log(Logger.DEBUG,
                        "FileStore upgraded successfully.");
                }

                if (upgradeNoBackup) {
                    // remove the remaining old store; just remove everything
                    try {
                        FileUtil.removeFiles(oldRoot, true);
                    } catch (IOException e2) {
                        // log something
                        logger.log(Logger.ERROR,
                            BrokerResources.E_REMOVE_STORE_FAILED, oldRoot, e2);
                    }
                }

                logger.logToAll(Logger.INFO, BrokerResources.I_UPGRADE_STORE_DONE);

                if (!upgradeNoBackup) {
                    // log message about the old store
                    logger.logToAll(Logger.INFO,
                        BrokerResources.I_REMOVE_OLD_FILESTORE, oldRoot);
                }
	    } catch (BrokerException e) {
		logger.log(Logger.INFO,
                    BrokerResources.I_REMOVE_NEW_STORE, rootDir);

		// upgrade failed with exception; remove the new store
		try {
		    FileUtil.removeFiles(rootDir, true);
		} catch (IOException e2) {
		    // log something
		    logger.log(Logger.ERROR,
                        BrokerResources.E_REMOVE_STORE_FAILED, rootDir, e);
		}
		throw e;
	    }

	} else {
	    // always load destinations first
	    dstList = new DestinationList(this, rootDir, resetStore);

	    msgStore = new MsgStore(this, rootDir,
					(resetStore || resetMessage));

	    intStore = new InterestStore(rootDir,
					(resetStore || resetInterest));

	    // transaction ids and associated ack lists
	    tidList = new TidList(rootDir, resetStore);

	    // configuration change record
	    configStore = new ConfigChangeRecord(rootDir, resetStore);

	    // properties
	    propFile = new PropertiesFile(rootDir, resetStore);

	    if (Store.DEBUG) {
		logger.log(Logger.DEBUG,
				"FileStore instantiated successfully.");
	    }
	}
    }

    /**
     * Used for Store backup/resotre utility only.
     * Instance directory is given explicitly. Depending on the clean
     * input, store is cleaned and instantiated rather than just instantiated.
     */
    public FileStore(String dir, boolean clean) throws BrokerException {

	// get the jmq.persist.file.sync.enabled and
	// jmq.persist.file.sync.all properties
	syncEnabled = config.getBooleanProperty(SYNC_ENABLED_PROP,
						DEFAULT_SYNC_ENABLED);

	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG, SYNC_ENABLED_PROP + "=" + syncEnabled);
	}

	syncAll = config.getBooleanProperty(SYNC_ALL_PROP, DEFAULT_SYNC_ALL);

	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG, SYNC_ALL_PROP + "=" + syncAll);
	}

	// instance root directory
	String instancename = dir + File.separator +
			Globals.getConfigName() + File.separator;

        logger.log(Logger.INFO, br.getString(BrokerResources.I_FILE_STORE_INFO, instancename));
				
	File instanceDir = new File(instancename);

	// the file store root directory
	rootDir = new File(instanceDir, FILESTORE_TOP);

	// check if we need to clean the store
	if (clean) {
	    try {
		// remove everything under root dir but not the root dir.
		FileUtil.removeFiles(rootDir, false);
                return;
                
	    } catch (IOException e) {
		logger.log(Logger.ERROR, BrokerResources.E_REMOVE_STORE_FAILED, rootDir, e);
		throw new BrokerException(
				br.getString(BrokerResources.E_REMOVE_STORE_FAILED, rootDir),
				e);
	    }
	} else if (!rootDir.exists() && !rootDir.mkdirs()) {
	    logger.log(Logger.ERROR, BrokerResources.E_CANNOT_CREATE_STORE_HIERARCHY,
				rootDir.toString());
	    throw new BrokerException(br.getString(
					BrokerResources.E_CANNOT_CREATE_STORE_HIERARCHY,
					rootDir.toString()));
	}

        // always load brokers first
        haBrokersStore = new HABrokers(rootDir, false);
        
        dstList = new DestinationList(this, rootDir, false);

        msgStore = new MsgStore(this, rootDir, false);

        intStore = new InterestStore(rootDir, false);

        // transaction ids and associated ack lists
        tidList = new TidList(rootDir, false);

        // configuration change record
        configStore = new ConfigChangeRecord(rootDir, false);

        // properties
        propFile = new PropertiesFile(rootDir, false);

        if (Store.DEBUG) {
            logger.log(Logger.DEBUG,
                            "FileStore instantiated successfully.");
        }
    }

    /**
     * Return the LoadException for loading destinations; null if there's
     * none.
     */
    public LoadException getLoadDestinationException() {
	return dstList.getLoadException();
    }

    /**
     * Return the LoadException for loading consumers; null if there's none.
     */
    public LoadException getLoadConsumerException() {
	return intStore.getLoadException();
    }

    /**
     * Return the LoadException for loading Properties; null if there's none.
     */
    public LoadException getLoadPropertyException() {
	return propFile.getLoadException();
    }

    /**
     * Return the LoadException for loading transactions; null if there's none.
     */
    public LoadException getLoadTransactionException() {
	return tidList.getLoadException();
    }

    /**
     * Return the LoadException for loading transaction acknowledgements;
     * null if there's none.
     */
    public LoadException getLoadTransactionAckException() {
	return tidList.getLoadTransactionAckException();
    }

    /**
     * Close the store and releases any system resources associated with
     * it.
     */
    public void close(boolean cleanup) {

	// make sure all operations are done before we proceed to close
	super.setClosedAndWait();

	dstList.close(cleanup);
	tidList.close(cleanup);
	configStore.close(cleanup);
	propFile.close(cleanup);
	intStore.close(cleanup);
	msgStore.close(cleanup);

        // If we're using memory-mapped file for txn and txn ack table,
        // calling System.gc() might free the mapped byte buffers because there
        // is no unmap API. We do this to reduce the risk of getting the error
        // "java.io.IOException: The requested operation cannot be performed on
        // a file with a user-mapped section open" when we re-open the store,
        // e.g. restart the broker with imqcmd. See bug 6354433.
        if (config.getBooleanProperty(TidList.TXN_USE_MEMORY_MAPPED_FILE_PROP,
            TidList.DEFAULT_TXN_USE_MEMORY_MAPPED_FILE)) {
            for (int i = 0; i < 3; i++) {
                System.gc();
                try {
                    // Pause for half a sec
                    Thread.sleep(500);
                } catch (InterruptedException e) {}
            }
        }

	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG, "FileStore.close("+ cleanup +") done.");
	}
    }

    /**
     * Clear the store. Remove all persistent data.
     * Note that this method is not synchronized.
     */
    public void clearAll(boolean sync) throws BrokerException {

	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG, "FileStore.clearAll() called");
	}

	// make sure store is not closed then increment in progress count
	super.checkClosedAndSetInProgress();

	try {
	    msgStore.clearAll(sync);
	    intStore.clearAll(sync);
	    dstList.clearAll(sync, false);// don't worry about messages since
					// they are removed already
	    tidList.clearAll(sync);
	    configStore.clearAll(sync);
	    propFile.clearAll(sync);

	    if (Store.DEBUG) {
		logger.log(Logger.DEBUG, "File store cleared");
	    }

	} finally {
	    // decrement in progress count
	    super.setInProgress(false);
	}
    }

    /**
     * Store a message, which is uniquely identified by it's SysMessageID,
     * and it's list of interests and their states.
     *
     * @param message	the message to be persisted
     * @param iids	an array of interest ids whose states are to be
     *			stored with the message
     * @param states	an array of states
     * @param sync	if true, will synchronize data to disk
     * @exception IOException if an error occurs while persisting the data
     * @exception BrokerException if a message with the same id exists
     *			in the store already
     * @exception NullPointerException	if <code>dst</code>,
     *			<code>message</code>,
     *			<code>iids</code>, or <code>states</code> is
     *			<code>null</code>
     */
    public void storeMessage(DestinationUID dst, Packet message,
	ConsumerUID[] iids, int[] states, boolean sync)
	throws IOException, BrokerException {

	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG, "FileStore.storeMessage() called for "
			+ message.getSysMessageID());
	}

	// make sure store is not closed then increment in progress count
	super.checkClosedAndSetInProgress();

	try {

	    if (dst == null || message == null || iids == null || states == null) {
		throw new NullPointerException();
	    }

	    if (iids.length == 0 || iids.length != states.length) {
		throw new BrokerException(br.getString(BrokerResources.E_BAD_INTEREST_LIST));
	    }

	    msgStore.storeMessage(dst, message, iids, states, sync);

	} finally {
	    // decrement in progress count
	    super.setInProgress(false);
	}
    }

    /**
     * Get the file store version.
     * @return file store version
     */
    public final int getStoreVersion() {
        return STORE_VERSION;
    }

    static final private ConsumerUID[] emptyiid = new ConsumerUID[0];
    static final private int[] emptystate = new int[0];

    /**
     * Store a message which is uniquely identified by it's system message id.
     *
     * @param message	the readonly packet to be persisted
     * @param sync	if true, will synchronize data to disk
     * @exception IOException if an error occurs while persisting the message
     * @exception BrokerException if a message with the same id exists
     *			in the store already
     */
    public void storeMessage(DestinationUID dst, Packet message, boolean sync)
	throws IOException, BrokerException {

	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG, "FileStore.storeMessage() called for "
			+ message.getSysMessageID());
	}

	// make sure store is not closed then increment in progress count
	super.checkClosedAndSetInProgress();

	try {
	    msgStore.storeMessage(dst, message, emptyiid, emptystate, sync);

	} finally {
	    // decrement in progress count
	    super.setInProgress(false);
	}
    }

    /**
     * Remove the message from the persistent store.
     * If the message has an interest list, the interest list will be
     * removed as well.
     *
     * @param id	the system message id of the message to be removed
     * @param sync	if true, will synchronize data to disk
     * @exception IOException if an error occurs while removing the message
     * @exception BrokerException if the message is not found in the store
     */
    public void removeMessage(DestinationUID dst, SysMessageID id, boolean sync)
	throws IOException, BrokerException {

	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG, "FileStore.removeMessage() called for "
			+ dst + ";" + id);
	}

	if (id == null) {
	    throw new NullPointerException();
	}

	// make sure store is not closed then increment in progress count
	super.checkClosedAndSetInProgress();

	try {
	    msgStore.removeMessage(dst, id, sync);
	} finally {
	    // decrement in progress count
	    super.setInProgress(false);
	}
    }

    /**
     * Move the message from one destination to another.
     * The message will be stored in the target destination with the
     * passed in consumers and their corresponding states.
     * After the message is persisted successfully, the message in the
     * original destination will be removed.
     *
     * @param message	message to be moved
     * @param from	destination the message is currently in 
     * @param to	destination to move the message to
     * @param ints	an array of interest ids whose states are to be
     *			stored with the message
     * @param states	an array of states
     * @param sync	if true, will synchronize data to disk.
     * @exception IOException if an error occurs while moving the message
     * @exception BrokerException if the message is not found in source
     *		destination
     * @exception NullPointerException	if <code>message</code>, 
     *			<code>from</code>, <code>to</code>,
     *			<code>iids</code>, or <code>states</code> is
     *			<code>null</code>
     */
    public void moveMessage(Packet message, DestinationUID from,
	DestinationUID to, ConsumerUID[] ints, int[] states, boolean sync)
	throws IOException, BrokerException {

	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG,
		"FileStore.moveMessage() called for: "
			+ message.getSysMessageID() + " from "
			+ from + " to " + to);
	}

	if (message == null || from == null || to == null) {
	    throw new NullPointerException();
	}

	if (ints == null) {
	    ints = emptyiid;
	    states = emptystate;
	}

	// make sure store is not closed then increment in progress count
	super.checkClosedAndSetInProgress();

	try {
	    msgStore.moveMessage(message, from, to, ints, states, sync);
	} finally {
	    // decrement in progress count
	    super.setInProgress(false);
	}
    }

    /**
     * Remove all messages associated with the specified destination
     * from the persistent store.
     *
     * @param destination	the destination whose messages are to be
     *				removed
     * @param sync	if true, will synchronize data to disk
     * @exception IOException if an error occurs while removing the messages
     * @exception BrokerException if the destination is not found in the store
     * @exception NullPointerException	if <code>destination</code> is
     *			<code>null</code>
     */
    public void removeAllMessages(Destination destination, boolean sync)
	throws IOException, BrokerException {

	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG,
		"FileStore.removeAllMessages(Destination) called");
	}

	// make sure store is not closed then increment in progress count
	super.checkClosedAndSetInProgress();

	try {
	    msgStore.removeAllMessages(destination.getDestinationUID(), sync);
	} finally {
	    // decrement in progress count
	    super.setInProgress(false);
	}
    }

    /**
     * Return an enumeration of all persisted messages for the given
     * destination.
     * Use the Enumeration methods on the returned object to fetch
     * and load each message sequentially.
     *
     * <p>
     * This method is to be used at broker startup to load persisted
     * messages on demand.
     *
     * @return an enumeration of all persisted messages, an empty
     *		enumeration will be returned if no messages exist for the
     *		destionation
     * @exception BrokerException if an error occurs while getting the data
     */
    public Enumeration messageEnumeration(Destination dst)
	throws BrokerException {

	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG,
		"FileStore.messageEnumeration(Destination) called");
	}

	// make sure store is not closed then increment in progress count
	super.checkClosedAndSetInProgress();

	try {
	    return msgStore.messageEnumeration(dst.getDestinationUID());
	} finally {
	    // decrement in progress count
	    super.setInProgress(false);
	}
    }

    /**
     * Return the message with the specified message id.
     *
     * @param id	the system message id of the message to be retrieved
     * @return a message
     * @exception BrokerException if the message is not found in the store
     *			or if an error occurs while getting the data
     */
    public Packet getMessage(DestinationUID dst, SysMessageID id)
	throws BrokerException {

	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG, "FileStore.getMessage() called");
	}

	if (id == null) {
	    throw new NullPointerException();
	}

	// make sure store is not closed then increment in progress count
	super.checkClosedAndSetInProgress();

	try {
	    return msgStore.getMessage(dst, id);
	} finally {
	    // decrement in progress count
	    super.setInProgress(false);
	}
    }

    /**
     * Store the given list of interests and their states with the
     * specified message.  The message should not have an interest
     * list associated with it yet.
     *
     * @param mid	system message id of the message that the interest
     *			is associated with
     * @param iids	an array of interest ids whose states are to be
     *			stored
     * @param states	an array of states
     * @param sync	if true, will synchronize data to disk
     * @exception BrokerException if the message is not in the store;
     *				if there's an interest list associated with
     *				the message already; or if an error occurs
     *				while persisting the data
     */
    public void storeInterestStates(DestinationUID dst,
	SysMessageID mid, ConsumerUID[] iids, int[] states, boolean sync)
	throws BrokerException {

	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG, "FileStore.storeInterestStates() called");
	}

	// make sure store is not closed then increment in progress count
	super.checkClosedAndSetInProgress();

	try {
	    if (mid == null || iids == null || states == null) {
		throw new NullPointerException();
	    }

	    if (iids.length == 0 || iids.length != states.length) {
		throw new BrokerException(br.getString(BrokerResources.E_BAD_INTEREST_LIST));
	    }

	    msgStore.storeInterestStates(dst, mid, iids, states, sync);

	} finally {
	    // decrement in progress count
	    super.setInProgress(false);
	}
    }

    /**
     * Update the state of the interest associated with the specified
     * message.  The interest should already be in the interest list
     * of the message.
     *
     * @param mid	system message id of the message that the interest
     *			is associated with
     * @param id	id of the interest whose state is to be updated
     * @param state	state of the interest
     * @param sync	if true, will synchronize data to disk
     * @exception BrokerException if the message is not in the store; if the
     *			interest is not associated with the message; or if
     *			an error occurs while persisting the data
     */
    public void updateInterestState(DestinationUID dst,
	SysMessageID mid, ConsumerUID id, int state, boolean sync)
    	throws BrokerException {

	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG, "FileStore.updateInterestState() called");
	}

	// make sure store is not closed then increment in progress count
	super.checkClosedAndSetInProgress();

	try {
	    if (mid == null || id == null) {
		throw new NullPointerException();
	    }

	    msgStore.updateInterestState(dst, mid, id, state, sync);

	} finally {
	    // decrement in progress count
	    super.setInProgress(false);
	}
    }

    /**
     * Get the state of the interest associated with the specified
     * message.
     *
     * @param mid	system message id of the message that the interest
     *			is associated with
     * @param id	id of the interest whose state is to be returned
     * @return state of the interest
     * @exception BrokerException if the specified interest is not
     *		associated with the message; or if the message is not in the
     *		store
     */
    public int getInterestState(DestinationUID dst, SysMessageID mid,
	ConsumerUID id) throws BrokerException {

	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG, "FileStore.getInterestState() called");
	}

	if (mid == null || id == null) {
	    throw new NullPointerException();
	}

	// make sure store is not closed then increment in progress count
	super.checkClosedAndSetInProgress();

	try {
	    return msgStore.getInterestState(dst, mid, id);
	} finally {
	    // decrement in progress count
	    super.setInProgress(false);
	}
    }

    /**
     * Retrieve all interest IDs associated with the specified message.
     * Note that the state of the interests returned is either
     * INTEREST_STATE_ROUTED or INTEREST_STATE_DELIVERED, i.e., interest
     * whose state is INTEREST_STATE_ACKNOWLEDGED will not be returned in the
     * array.
     *
     * @param mid	system message id of the message whose interests
     *			are to be returned
     * @return an array of ConsumerUID objects associated with the message; a
     *		zero length array will be returned if no interest is
     *		associated with the message
     * @exception BrokerException if the message is not in the store
     */
    public ConsumerUID[] getConsumerUIDs(DestinationUID dst, SysMessageID mid)
	throws BrokerException {

	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG, "FileStore.getConsumerUIDs() called");
	}

	if (mid == null) {
	    throw new NullPointerException();
	}

	// make sure store is not closed then increment in progress count
	super.checkClosedAndSetInProgress();

	try {
	    return msgStore.getConsumerUIDs(dst, mid);
	} finally {
	    // decrement in progress count
	    super.setInProgress(false);
	}
    }

    /**
     * Store an Interest which is uniquely identified by it's id.
     *
     * @param interest the interest to be persisted
     * @param sync	if true, will synchronize data to disk
     * @exception IOException if an error occurs while persisting the interest
     * @exception BrokerException if an interest with the same id exists in
     *			the store already
     */
    public void storeInterest(Consumer interest, boolean sync)
	throws IOException, BrokerException {

	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG, "FileStore.storeInterest() called");
	}

	// make sure store is not closed then increment in progress count
	super.checkClosedAndSetInProgress();

	try {
	    intStore.storeInterest(interest, sync);
	} finally {
	    // decrement in progress count
	    super.setInProgress(false);
	}
    }

    /**
     * Remove the interest from the persistent store.
     *
     * @param interest	the interest to be removed from persistent store
     * @param sync	if true, will synchronize data to disk
     * @exception IOException if an error occurs while removing the interest
     * @exception BrokerException if the interest is not found in the store
     */
    public void removeInterest(Consumer interest, boolean sync)
	throws IOException, BrokerException {

	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG, "FileStore.removeInterest() called");
	}

	// make sure store is not closed then increment in progress count
	super.checkClosedAndSetInProgress();

	try {
	    intStore.removeInterest(interest, sync);
	} finally {
	    // decrement in progress count
	    super.setInProgress(false);
	}
    }

    /**
     * Retrieve all interests in the store.
     * Will return as many interests as we can read.
     * Any interests that are retrieved unsuccessfully will be logged as a
     * warning.
     *
     * @return an array of Interest objects; a zero length array is
     * returned if no interests exist in the store
     * @exception IOException if an error occurs while getting the data
     */
    public Consumer[] getAllInterests() throws IOException, BrokerException {

	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG, "FileStore.getAllInterests() called");
	}

	// make sure store is not closed then increment in progress count
	super.checkClosedAndSetInProgress();

	try {
	    return intStore.getAllInterests();
	} finally {
	    // decrement in progress count
	    super.setInProgress(false);
	}
    }

    /**
     * Store a Destination.
     *
     * @param destination	the destination to be persisted
     * @param sync	if true, will synchronize data to disk
     * @exception IOException if an error occurs while persisting
     *		the destination
     * @exception BrokerException if the same destination exists
     *			the store already
     * @exception NullPointerException	if <code>destination</code> is
     *			<code>null</code>
     */
    public void storeDestination(Destination destination, boolean sync)
	throws IOException, BrokerException {

	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG, "FileStore.storeDestination() called");
	}

	// make sure store is not closed then increment in progress count
	super.checkClosedAndSetInProgress();

	try {
	    if (destination == null) {
		throw new NullPointerException();
	    }

	    dstList.storeDestination(destination, sync);
	} finally {
	    // decrement in progress count
	    super.setInProgress(false);
	}
    }

    /**
     * Update the specified destination.
     *
     * @param destination	the destination to be updated
     * @param sync	if true, will synchronize data to disk
     * @exception BrokerException if the destination is not found in the store
     *				or if an error occurs while updating the
     *				destination
     */
    public void updateDestination(Destination destination, boolean sync)
	throws BrokerException {

	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG, "FileStore.updateDestination() called");
	}

	// make sure store is not closed then increment in progress count
	super.checkClosedAndSetInProgress();

	try {
	    if (destination == null) {
		throw new NullPointerException();
	    }

	    dstList.updateDestination(destination, sync);
	} finally {
	    // decrement in progress count
	    super.setInProgress(false);
	}
    }

    /**
     * Remove the destination from the persistent store.
     * All messages associated with the destination will be removed as well.
     *
     * @param destination	the destination to be removed
     * @param sync	if true, will synchronize data to disk
     * @exception IOException if an error occurs while removing the destination
     * @exception BrokerException if the destination is not found in the store
     */
    public void removeDestination(Destination destination, boolean sync)
	throws IOException, BrokerException {

	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG, "FileStore.removeDestination() called");
	}

	// make sure store is not closed then increment in progress count
	super.checkClosedAndSetInProgress();

	try {
	    if (destination == null) {
		throw new NullPointerException();
	    }

	    dstList.removeDestination(destination, sync);
	} finally {
	    // decrement in progress count
	    super.setInProgress(false);
	}
    }

    /**
     * Retrieve all destinations in the store.
     *
     * @return an array of Destination objects; a zero length array is
     * returned if no destinations exist in the store
     * @exception IOException if an error occurs while getting the data
     */
    public Destination[] getAllDestinations()
	throws IOException, BrokerException {

	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG, "FileStore.getAllDestinations() called");
	}

	// make sure store is not closed then increment in progress count
	super.checkClosedAndSetInProgress();

	try {
	    return dstList.getAllDestinations();
	} finally {
	    // decrement in progress count
	    super.setInProgress(false);
	}
    }

    /**
     * Store a transaction.
     *
     * @param id	the id of the transaction to be persisted
     * @param ts	the transaction state to be persisted
     * @param sync	if true, will synchronize data to disk
     * @exception IOException if an error occurs while persisting
     *		the transaction
     * @exception BrokerException if the same transaction id exists
     *			the store already
     * @exception NullPointerException	if <code>id</code> is
     *			<code>null</code>
     */
    public void storeTransaction(TransactionUID id, TransactionState ts,
        boolean sync) throws IOException, BrokerException {

	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG, "FileStore.storeTransaction() called");
	}

	// make sure store is not closed then increment in progress count
	super.checkClosedAndSetInProgress();

	try {
	    if (id == null || ts == null) {
		throw new NullPointerException();
	    }

	    tidList.storeTransaction(id, ts, sync);

	} finally {
	    // decrement in progress count
	    super.setInProgress(false);
	}
    }

    /**
     * Remove the transaction. The associated acknowledgements
     * will not be removed.
     *
     * @param id	the id of the transaction to be removed
     * @param sync	if true, will synchronize data to disk
     * @exception IOException if an error occurs while removing the transaction
     * @exception BrokerException if the transaction is not found
     *			in the store
     */
    public void removeTransaction(TransactionUID id, boolean sync)
	throws IOException, BrokerException {

	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG, "FileStore.removeTransaction() called");
	}

	// make sure store is not closed then increment in progress count
	super.checkClosedAndSetInProgress();

	try {
	    if (id == null) {
		throw new NullPointerException();
	    }

	    tidList.removeTransaction(id, sync);
	} finally {
	    // decrement in progress count
	    super.setInProgress(false);
	}
    }

    /**
     * Update the state of a transaction
     *
     * @param id	the transaction id to be updated
     * @param ts	the new transaction state
     * @param sync	if true, will synchronize data to disk
     * @exception IOException if an error occurs while persisting
     *		the transaction id
     * @exception BrokerException if the transaction id does NOT exists in
     *			the store already
     * @exception NullPointerException	if <code>id</code> is
     *			<code>null</code>
     */
    public void updateTransactionState(TransactionUID id, TransactionState ts,
        boolean sync) throws IOException, BrokerException {

	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG, "FileStore.updateTransactionState( id=" +
                id + ", ts=" + ts.getState() + ") called");
	}

	// make sure store is not closed then increment in progress count
	super.checkClosedAndSetInProgress();

	try {
	    if (id == null) {
		throw new NullPointerException();
	    }

	    tidList.updateTransactionState(id, ts.getState(), sync);
	} finally {
	    // decrement in progress count
	    super.setInProgress(false);
	}
    }

    /**
     * Retrieve all transaction ids in the store with their state
     *
     * @return A HashMap. The key is a TransactionUID.
     * The value is a TransactionState.
     * @exception IOException if an error occurs while getting the data
     */
    public HashMap getAllTransactionStates()
        throws IOException, BrokerException {

	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG,
			"FileStore.getAllTransactionStates() called");
	}

	// make sure store is not closed then increment in progress count
	super.checkClosedAndSetInProgress();

	try {
	    return tidList.getAllTransactionStates();
	} finally {
	    // decrement in progress count
	    super.setInProgress(false);
	}
    }

    /**
     * Store the acknowledgement for the specified transaction.
     *
     * @param tid	the transaction id with which the acknowledgment is to
     *			be stored
     * @param ack	the acknowledgement to be stored
     * @param sync	if true, will synchronize data to disk
     * @exception BrokerException if the transaction id is not found in the
     *				store, if the acknowledgement already
     *				exists, or if it failed to persist the data
     */
    public void storeTransactionAck(TransactionUID tid,
	TransactionAcknowledgement ack, boolean sync) throws BrokerException {

	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG, "FileStore.storeTransactionAck() called");
	}

	// make sure store is not closed then increment in progress count
	super.checkClosedAndSetInProgress();

	try {
	    if (tid == null || ack == null) {
		throw new NullPointerException();
	    }

	    tidList.storeTransactionAck(tid, ack, sync);
	} finally {
	    // decrement in progress count
	    super.setInProgress(false);
	}
    }

    /**
     * Remove all acknowledgements associated with the specified
     * transaction from the persistent store.
     *
     * @param id	the transaction id whose acknowledgements are
     *			to be removed
     * @param sync	if true, will synchronize data to disk
     * @exception BrokerException if error occurs while removing the
     *			acknowledgements
     */
    public void removeTransactionAck(TransactionUID id, boolean sync)
	throws BrokerException {

	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG, "FileStore.removeTransactionAck() called");
	}

	// make sure store is not closed then increment in progress count
	super.checkClosedAndSetInProgress();

	try {
	    if (id == null)
		throw new NullPointerException();

	    tidList.removeTransactionAck(id, sync);
	} finally {
	    // decrement in progress count
	    super.setInProgress(false);
	}
    }

    /**
     * Retrieve all acknowledgements for the specified transaction.
     *
     * @param tid	id of the transaction whose acknowledgements
     *			are to be returned
     * @exception BrokerException if the operation fails for some reason
     */
    public TransactionAcknowledgement[] getTransactionAcks(
	TransactionUID tid) throws BrokerException {

	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG, "FileStore.getTransactionAcks() called");
	}

	// make sure store is not closed then increment in progress count
	super.checkClosedAndSetInProgress();

	try {
	    if (tid == null) {
		throw new NullPointerException();
	    }

	    return tidList.getTransactionAcks(tid);
	} finally {
	    // decrement in progress count
	    super.setInProgress(false);
	}
    }

    /**
     * Retrieve all acknowledgement list in the persistence store together
     * with their associated transaction id. The data is returned in the
     * form a HashMap. Each entry in the HashMap has the transaction id as
     * the key and an array of the associated TransactionAcknowledgement
     * objects as the value.
     * @return a HashMap object containing all acknowledgement lists in the
     *		persistence store
     */
    public HashMap getAllTransactionAcks() throws BrokerException {
	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG,
			"FileStore.getAllTransactionAcks() called");
	}

	// make sure store is not closed then increment in progress count
	super.checkClosedAndSetInProgress();

	try {
	    return tidList.getAllTransactionAcks();
	} finally {
	    // decrement in progress count
	    super.setInProgress(false);
	}
    }

    /**
     * Persist the specified property name/value pair.
     * If the property identified by name exists in the store already,
     * it's value will be updated with the new value.
     * If value is null, the property will be removed.
     * The value object needs to be serializable.
     *
     * @param name name of the property
     * @param value value of the property
     * @param sync	if true, will synchronize data to disk
     * @exception BrokerException if an error occurs while persisting the
     *			data
     * @exception NullPointerException if <code>name</code> is
     *			<code>null</code>
     */
    public void updateProperty(String name, Object value, boolean sync)
	throws BrokerException {

	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG, "FileStore.updateProperty() called");
	}

	// make sure store is not closed then increment in progress count
	super.checkClosedAndSetInProgress();

	try {
	    if (name == null)
		throw new NullPointerException();

	    propFile.updateProperty(name, value, sync);
	} finally {
	    // decrement in progress count
	    super.setInProgress(false);
	}
    }

    /**
     * Retrieve the value for the specified property.
     *
     * @param name name of the property whose value is to be retrieved
     * @return the property value; null is returned if the specified
     *		property does not exist in the store
     * @exception BrokerException if an error occurs while retrieving the
     *			data
     * @exception NullPointerException if <code>name</code> is
     *			<code>null</code>
     */
    public Object getProperty(String name) throws BrokerException {

	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG, "FileStore.getProperty() called");
	}

	// make sure store is not closed then increment in progress count
	super.checkClosedAndSetInProgress();

	try {
	    if (name == null)
		throw new NullPointerException();

	    return propFile.getProperty(name);
	} finally {
	    // decrement in progress count
	    super.setInProgress(false);
	}
    }

    /**
     * Return the names of all persisted properties.
     *
     * @return an array of property names; an empty array will be returned
     *		if no property exists in the store.
     */
    public String[] getPropertyNames() throws BrokerException {

	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG, "FileStore.getPropertyNames() called");
	}

	// make sure store is not closed then increment in progress count
	super.checkClosedAndSetInProgress();

	try {
	    return propFile.getPropertyNames();
	} finally {
	    // decrement in progress count
	    super.setInProgress(false);
	}
    }

    /**
     * Append a new record to the config change record store.
     * The timestamp is also persisted with the recordData.
     * The config change record store is an ordered list (sorted
     * by timestamp).
     *
     * @param timestamp The time when this record was created.
     * @param recordData The record data.
     * @param sync	if true, will synchronize data to disk
     * @exception BrokerException if an error occurs while persisting
     *			the data or if the timestamp is less than or
     *			equal to 0
     * @exception NullPointerException if <code>recordData</code> is
     *			<code>null</code>
     */
    public void storeConfigChangeRecord(
	long timestamp, byte[] recordData, boolean sync)
	throws BrokerException {

	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG,
			"FileStore.storeConfigChangeRecord() called");
	}

	// make sure store is not closed then increment in progress count
	super.checkClosedAndSetInProgress();

	try {
	    if (timestamp <= 0) {
		logger.log(Logger.ERROR, BrokerResources.E_INVALID_TIMESTAMP,
			new Long(timestamp));
		throw new BrokerException(
			br.getString(BrokerResources.E_INVALID_TIMESTAMP,
					new Long(timestamp)));
	    }

	    if (recordData == null) {
		throw new NullPointerException();
	    }

	    configStore.storeConfigChangeRecord(timestamp, recordData, sync);
	} finally {
	    // decrement in progress count
	    super.setInProgress(false);
	}
    }

    /**
     * Get all the config change records since the given timestamp.
     * Retrieves all the entries with recorded timestamp greater than
     * the specified timestamp.
     * 
     * @return an ArrayList of byte[] recordData objects retrieved
     * from the store. The ArrayList should be empty if there are
     * no records.
     */
    public ArrayList getConfigChangeRecordsSince(long timestamp)
	throws BrokerException {

	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG,
			"FileStore.getConfigChangeRecordsSince() called");
	}

	// make sure store is not closed then increment in progress count
	super.checkClosedAndSetInProgress();

	try {
	    return configStore.getConfigChangeRecordsSince(timestamp);
	} finally {
	    // decrement in progress count
	    super.setInProgress(false);
	}
    }

    /**
     * Return all config records together with their corresponding
     * timestamps.
     * The returned data is an array of Object with 2 elements.
     * The first element contains an ArrayList of all timestamps and
     * the second element contains an ArrayList of all config
     * records, each of which is of type byte[].
     *
     * @return an array of Object whose first element contains an ArrayList
     *		of timestamps and the second element contains an arrayList
     *		of config records.
     * @exception BrokerException if an error occurs while getting the data
     */
    public Object[] getAllConfigRecords() throws BrokerException {
	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG,
			"FileStore.getAllConfigRecords() called");
	}

	// make sure store is not closed then increment in progress count
	super.checkClosedAndSetInProgress();

	try {
	    return configStore.getAllConfigRecords();
	} finally {
	    // decrement in progress count
	    super.setInProgress(false);
	}
    }

    /**
     * Clear all config change records in the store.
     *
     * @param sync	if true, will synchronize data to disk
     * @exception BrokerException if an error occurs while clearing the data
     */
    public void clearAllConfigChangeRecords(boolean sync)
	throws BrokerException {
	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG,
			"FileStore.clearAllConfigChangeRecords() called");
	}

	// make sure store is not closed then increment in progress count
	super.checkClosedAndSetInProgress();

	try {
	    configStore.clearAll(sync);
	} finally {
	    // decrement in progress count
	    super.setInProgress(false);
	}
    }

    /**
     * Return the number of persisted messages and total number of bytes for
     * the given destination.
     *
     * @param dst the destination whose messages are to be counted
     * @return A HashMap of name value pair of information
     * @throws BrokerException if an error occurs while getting the data
     */
    public HashMap getMessageStorageInfo(Destination dst)
        throws BrokerException {

        if (Store.DEBUG) {
            logger.log(Logger.DEBUG,
                "FileStore.getMessageStorageInfo(Destination) called");
        }

        // make sure store is not closed then increment in progress count
        super.checkClosedAndSetInProgress();

        try {
            DestinationUID dstID = dst.getDestinationUID();
            HashMap data = new HashMap(2);
            data.put( DestMetricsCounters.CURRENT_MESSAGES,
                new Integer(msgStore.getMessageCount(dstID)) );
            data.put( DestMetricsCounters.CURRENT_MESSAGE_BYTES,
                new Long(msgStore.getByteCount(dstID)) );
            return data;
        } finally {
            // decrement in progress count
            super.setInProgress(false);
        }
    }

    public String getStoreType() {
	return FILE_STORE_TYPE;
    }

    public boolean isJDBCStore() {
        return false;
    }

    /**
     * Get information about the underlying storage for the specified
     * destination.
     * @return A HashMap of name value pair of information
     */
    public HashMap getStorageInfo(Destination destination)
	throws BrokerException {

	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG, "FileStore.getStorageInfo(" +
			destination + ") called");
	}

	// make sure store is not closed then increment in progress count
	super.checkClosedAndSetInProgress();

	try {
	    return msgStore.getStorageInfo(destination);
	} finally {
	    // decrement in progress count
	    super.setInProgress(false);
	}
    }

    /**
     * Get debug information about the store.
     * @return A Hashtable of name value pair of information
     */
    public Hashtable getDebugState() {
	Hashtable t = new Hashtable();
	t.put("File-based store", rootDir.getPath());
	t.put("Store version", String.valueOf(STORE_VERSION));
	t.putAll(dstList.getDebugState());
	t.putAll(msgStore.getDebugState());
	t.putAll(intStore.getDebugState());
	t.putAll(tidList.getDebugState());
	t.putAll(propFile.getDebugState());
	t.putAll(configStore.getDebugState());
	return t;
    }

    /**
     * Compact the message file associated with the specified destination.
     * If null is specified, message files assocated with all persisted
     * destinations will be compacted..
     */
    public void compactDestination(Destination destination)
	throws BrokerException {

	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG, "FileStore.compactDestination(" +
			destination + ") called");
	}

	// make sure store is not closed then increment in progress count
	super.checkClosedAndSetInProgress();

	try {
	    msgStore.compactDestination(destination);

	} finally {
	    // decrement in progress count
	    super.setInProgress(false);
	}
    }

    /**
     * Get all the HA broker records.
     * 
     * @return a HashMap of HABrokerInfo objects. 
     */
    public HashMap getAllBrokerInfos()
	throws BrokerException {

	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG,
			"FileStore.getAllBrokerInfos() called");
	}

	// make sure store is not closed then increment in progress count
	super.checkClosedAndSetInProgress();

	try {
	    return haBrokersStore.getAllBrokerInfos();
	} finally {
	    // decrement in progress count
	    super.setInProgress(false);
	}
    }

    /*
     * persist broker info record to files.
     * 
     * @param id Broker ID
     * @param takeoverBrokerID Broker ID taken over the store
     * @param url the broker's URL
     * @param version the broker's version
     * @param state the broker's state
     * @param sessionID the broker's session ID
     * @param heartbeat broker's last heartbeat
     * @exception BrokerException if an error occurs while persisting the data
     */
    public void addBrokerInfo(String brokerID, String takeOverBkrID, String URL, BrokerState state,
        int version, long sessionID, long heartbeat) throws BrokerException {
        // make sure store is not closed then increment in progress count
        checkClosedAndSetInProgress();

        try {
                haBrokersStore.addBrokerInfo(brokerID, takeOverBkrID, URL, state.intValue(), version, sessionID, heartbeat);
                            
        } finally {
            // decrement in progress count
            setInProgress(false);
        }
    }

    MsgStore getMsgStore() {
	return msgStore;
    }

    DestinationList getDstStore() {
	return dstList;
    }

    // a file filter that returns true if pathname is a directory
    // whose name starts with "fs"
    private static FilenameFilter storeFilter = new FilenameFilter() {
	public boolean accept(File dir, String name) {
	    return ((new File(dir, name)).isDirectory()
			&& name.startsWith(FILESTORE_BASENAME));
	}
    };

    /**
     * The method does sanity checks on the file store.
     * Newroot is the file store we support in the this release.
     * Oldroot is a version of the store we recognize and might need
     * migration.
     *
     * Return 0 if store doesn't need to be upgrade or the version of the store
     * that needs to be upgrade, e.g. 350 or 200.
     */
    private int checkFileStore(File topDir) throws BrokerException {

	int upgrade = 0;

	if (Store.DEBUG) {
	    logger.log(Logger.DEBUG, "topDir=" + topDir);
	}

	// look for any unsupported file store: a directory
	// whose name starts with 'fs' but is not one of those we expect
	// e.g. a file store of a future release
	String[] names = topDir.list(storeFilter);
	if (names != null) {
	    for (int i = 0; i < names.length; i++) {
		if (!names[i].equals(FILESTORE_TOP)
		    && !names[i].equals(FILESTORE350_TOP)) {

		    File badfs = new File(topDir, names[i]);
		    // unsupported store found
		    logger.log(Logger.ERROR, BrokerResources.E_UNSUPPORTED_FILE_STORE,
					badfs);
		    throw new BrokerException(br.getString(
				BrokerResources.E_UNSUPPORTED_FILE_STORE, badfs));
		}
	    }
	}

	File newRootDir = new File(topDir, FILESTORE_TOP);
        File oldRootDir350 = new File(topDir, FILESTORE350_TOP);
        File oldRootDir200 = new File(topDir, FILESTORE200_TOP);

        boolean storeExist200 = oldRootDir200.exists();
        boolean storeExist350 = oldRootDir350.exists();
        int oldStoreVersion = OLD_STORE_VERSION_200;
        File oldRootDir = oldRootDir200;
        if (storeExist350) {
            // 350 take precedence
            oldStoreVersion = OLD_STORE_VERSION;
            oldRootDir = oldRootDir350;
        } else if (storeExist200) {
            checkOldVersion(new File(oldRootDir, VERSIONFILE), oldStoreVersion);
        }

        if (newRootDir.exists()) {
            // new store exists
            if (!removeStore) {
                // log message only if removeStore is not true
                // log reminder message to remove old store
                if (storeExist350) {
                    logger.logToAll(Logger.INFO,
                        BrokerResources.I_REMOVE_OLDSTORE_REMINDER, oldRootDir );
                } else if (storeExist200) {
                    logger.logToAll(Logger.INFO,
                        BrokerResources.I_REMOVE_OLDSTORE_REMINDER, oldRootDir );
                }
            }
        } else if (storeExist200 || storeExist350) {
            // new store does not exist, but old store exists
            if (removeStore) {
                // do nothing, just log a message and return since
                // the caller will do the remove
                logger.logToAll(Logger.INFO,
                    BrokerResources.I_REMOVE_OLD_PERSISTENT_STORE, oldRootDir);
            } else if (resetStore) {
                // log message to remove old store
                logger.logToAll(Logger.INFO,
                    BrokerResources.I_RESET_OLD_PERSISTENT_STORE, oldRootDir);
                logger.logToAll(Logger.INFO,
                    BrokerResources.I_WILL_CREATE_NEW_STORE);

                // just remove everything
                try {
                    FileUtil.removeFiles(oldRootDir, true);
                } catch (IOException e) {
                    logger.log(Logger.ERROR,
                        BrokerResources.E_RESET_STORE_FAILED, oldRootDir, e);
                    throw new BrokerException(br.getString(
                        BrokerResources.E_RESET_STORE_FAILED, oldRootDir), e);
                }
            } else {
                // log message to do upgrade
                logger.logToAll(Logger.INFO, BrokerResources.I_UPGRADE_STORE_MSG,
                    new Integer(oldStoreVersion));

                if (upgradeNoBackup && !Broker.getBroker().force) {
                    // will throw BrokerException if the user backs out
                    getConfirmation();
                }

                // set return value to the version of old store need upgrading
                upgrade = oldStoreVersion;
            }
        }

	return upgrade;
    }

    // check the version of the old (200) store
    private void checkOldVersion(File versionfile, int version)
	throws BrokerException {

	if (!versionfile.exists()) {
	    // bad store; store with no version file; throw exception
	    logger.log(Logger.ERROR,
                BrokerResources.E_BAD_OLDSTORE_NO_VERSIONFILE, versionfile);
	    throw new BrokerException(br.getString(
                BrokerResources.E_BAD_OLDSTORE_NO_VERSIONFILE, versionfile));
	}

	RandomAccessFile raf = null;
	int integer = 0;
	try {
	    raf = new RandomAccessFile(versionfile, "r");
	    String str = raf.readLine();
	    raf.close();

	    try {
	    	integer = Integer.parseInt(str);
	    } catch (NumberFormatException e) {
		logger.log(Logger.ERROR,
                    BrokerResources.E_BAD_OLDSTORE_VERSION, str,
                    Integer.toString(version), e);
		throw new BrokerException(br.getString(
                    BrokerResources.E_BAD_OLDSTORE_VERSION, str,
                    Integer.toString(version)), e);
	    }

	    if (integer != version) {
		logger.log(Logger.ERROR,
                    BrokerResources.E_BAD_OLDSTORE_VERSION, str,
                    Integer.toString(version));
		throw new BrokerException(br.getString(
                    BrokerResources.E_BAD_OLDSTORE_VERSION, str,
                    Integer.toString(version)));
	    }
	} catch (IOException e) {
	    logger.log(Logger.ERROR,
                BrokerResources.X_STORE_VERSION_CHECK_FAILED, e);
	    throw new BrokerException(br.getString(
                BrokerResources.X_STORE_VERSION_CHECK_FAILED), e);
	}
    }
}

