/*
 * @(#)Protocol.java	1.1 04/18/03
 *
 * Copyright 2003 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms. 
 *
 */

package com.sun.messaging.jmq.jmsserver.multibroker;

import java.io.*;
import java.util.*;
import com.sun.messaging.jmq.jmsserver.Globals;
import com.sun.messaging.jmq.jmsserver.Broker;
import com.sun.messaging.jmq.jmsserver.BrokerNotification;
import com.sun.messaging.jmq.util.log.*;
import com.sun.messaging.jmq.jmsserver.resources.*;
import com.sun.messaging.jmq.jmsserver.core.*;
import com.sun.messaging.jmq.io.*;
import com.sun.messaging.jmq.util.*;
import com.sun.messaging.jmq.jmsserver.util.BrokerException;
import com.sun.messaging.jmq.jmsserver.service.ConnectionUID;
import com.sun.messaging.jmq.jmsserver.core.BrokerAddress;
import com.sun.messaging.jmq.jmsserver.multibroker.ClusterGlobals;
import com.sun.messaging.jmq.jmsserver.multibroker.falcon.FalconProtocol;
import com.sun.messaging.jmq.jmsserver.multibroker.raptor.RaptorProtocol;
import com.sun.messaging.jmq.jmsserver.multibroker.raptor.ProtocolGlobals;
import com.sun.messaging.jmq.jmsserver.cluster.*;
import com.sun.messaging.jmq.jmsserver.cluster.ha.*;

public class CommonProtocol implements Protocol
{
    public static boolean DEBUG = false;

    protected Logger logger = Globals.getLogger();
    protected MessageBusCallback cb = null;
    protected Cluster c = null;
    protected com.sun.messaging.jmq.jmsserver.core.BrokerAddress
        selfAddress = null;

    protected Protocol realProtocol = null;
    protected boolean protocolInitComplete = false;
    protected long startTime = 0;

    private Integer configServerVersion = null;

    public CommonProtocol(MessageBusCallback cb, Cluster c,
        com.sun.messaging.jmq.jmsserver.core.BrokerAddress myaddress)
        throws BrokerException {
        this.cb = cb;
        this.c = c;
        this.selfAddress = myaddress;
        this.startTime =  System.currentTimeMillis();
    }

    public int getHighestSupportedVersion() {
         return ProtocolGlobals.getCurrentVersion();
    }

    public int getClusterVersion() {
        if (!getProtocolInitComplete()) return ClusterGlobals.VERSION_300;
        try {
            if (c.getConfigServer() != null && configServerVersion != null) {
                return configServerVersion.intValue();
            }
        } catch (Exception e) {
            logger.logStack(logger.WARNING,
                BrokerResources.E_INTERNAL_BROKER_ERROR,
                "getConfigServer()", e);
        }
        return realProtocol.getClusterVersion();
    }

    private void initOldProtocol() {
        if (DEBUG)
            logger.log(logger.DEBUG, "Using FALCON cluster protocol.");
        try {
            realProtocol = new
                com.sun.messaging.jmq.jmsserver.multibroker.falcon.FalconProtocol(
                cb, c , selfAddress);
            realProtocol.startClusterIO();
        }
        catch (Exception e) {
            logger.logStack(logger.WARNING,
                BrokerResources.E_INTERNAL_BROKER_ERROR,
                "Initializing the cluster protcol", e);
        }
    }

    private void initNewProtocol() {
        if (DEBUG) {
            logger.log(logger.DEBUG, "Using RAPTOR cluster protocol.");
        }

        try {
            c.useGPackets(true);
        }
        catch (Exception e) {}

        try {
            realProtocol = new
                com.sun.messaging.jmq.jmsserver.multibroker.raptor.RaptorProtocol(
                cb, c , selfAddress, getBrokerInfo());
            realProtocol.startClusterIO();
        } catch (Exception e) {
            logger.logStack(logger.WARNING,
                BrokerResources.E_INTERNAL_BROKER_ERROR,
                "Initializing the cluster protcol", e);
        }
    }

    public BrokerInfo getBrokerInfo() {
        BrokerInfo selfInfo = new BrokerInfo();
        selfInfo.setBrokerAddr(selfAddress);
        selfInfo.setStartTime(startTime);
        selfInfo.setStoreDirtyFlag(false);
        selfInfo.setClusterProtocolVersion(new Integer(ProtocolGlobals.getCurrentVersion()));

        if (Globals.getHAEnabled()) {
            selfInfo.setHeartbeatHostAddress(Globals.getHeartbeatService().getHeartbeatHostAddress());
            selfInfo.setHeartbeatPort(Globals.getHeartbeatService().getHeartbeatPort());
            selfInfo.setHeartbeatInterval(Globals.getHeartbeatService().getHeartbeatInterval());
        }

        return selfInfo;
    }

    public ClusterBrokerInfoReply getBrokerInfoReply(BrokerInfo remote) throws Exception {
        if (c.getConfigServer() != null) {
            ClusterBrokerInfoReply cbi = ClusterBrokerInfoReply.newInstance(
                            getBrokerInfo(), ProtocolGlobals.G_BROKER_INFO_OK);
            return cbi;
        }
        if (!getProtocolInitComplete()) {
            throw new BrokerException(Globals.getBrokerResources().getKString(
                BrokerResources.X_CLUSTER_PROTOCOL_NOT_READY), Status.UNAVAILABLE);
        }

        return realProtocol.getBrokerInfoReply(remote);
    }

    public int addBrokerInfo(BrokerInfo brokerInfo) {
        if (getProtocolInitComplete()) {
            return realProtocol.addBrokerInfo(brokerInfo);
        }

        com.sun.messaging.jmq.jmsserver.core.BrokerAddress configServer;
        try {
            configServer = c.getConfigServer();
        }
        catch (Exception e) {
            logger.logStack(logger.WARNING,
                BrokerResources.E_INTERNAL_BROKER_ERROR,
                "getConfigServer()", e);
            return ADD_BROKER_INFO_RETRY;
        }

        if (configServer != null &&
            configServer.equals(brokerInfo.getBrokerAddr())) {
            boolean useNewProtocol = false;

            Integer protoVersion = brokerInfo.getClusterProtocolVersion();
            if (protoVersion != null &&
                protoVersion.intValue() >= ProtocolGlobals.VERSION_350) {
                useNewProtocol = true;
            }

            if (useNewProtocol)
                initNewProtocol();
            else
                initOldProtocol();

            configServerVersion = protoVersion;

            // From this point on, all the calls will be forwarded to
            // the real protocol implementation...
            setProtocolInitComplete(true);
            return realProtocol.addBrokerInfo(brokerInfo);
        }
        else {
            return ADD_BROKER_INFO_RETRY;
        }
    }

    public void removeBrokerInfo(
        com.sun.messaging.jmq.jmsserver.core.BrokerAddress broker) {
        if (realProtocol != null)
            realProtocol.removeBrokerInfo(broker);
    }

    public void setMatchProps(Properties matchProps) {
        c.setMatchProps(matchProps);
    }

    public void startClusterIO() {
        if (DEBUG)
            logger.log(Logger.DEBUG,
            "CommonProtocol.startClusterIO()");

        // If this is the master broker then use the new raptor
        // cluster protocol.
        try {
            com.sun.messaging.jmq.jmsserver.core.BrokerAddress configServer
                = c.getConfigServer();
            if (configServer == null ||
                configServer.equals(selfAddress)) {
                initNewProtocol();
                setProtocolInitComplete(true);
            }
        }
        catch (Exception e) {}

        try {
            c.start();
        }
        catch (Exception e) {
            //should add interface Cluster.getServiceName
            logger.logStack(logger.ERROR, BrokerResources.X_START_SERVICE_EXCEPTION, 
                   "cluster", e.getMessage(), e); 
            Broker.getBroker().exit(1,
                 Globals.getBrokerResources()
                   .getKString(BrokerResources.X_START_SERVICE_EXCEPTION,
                   "cluster", e.getMessage()),
                 BrokerNotification.REASON_EXCEPTION);
        }
    }

    public void stopClusterIO(boolean requestTakeover) {
        if (realProtocol != null)
            realProtocol.stopClusterIO(requestTakeover);

        c.shutdown();
    }

    public void receiveUnicast(
        com.sun.messaging.jmq.jmsserver.core.BrokerAddress sender,
        GPacket gp) {
        if (DEBUG)
            logger.log(logger.DEBUG, "receiveUnicast GPacket");
        realProtocol.receiveUnicast(sender, gp);
    }

    public void receiveBroadcast(
        com.sun.messaging.jmq.jmsserver.core.BrokerAddress sender,
        GPacket gp) {
        if (DEBUG)
            logger.log(logger.DEBUG, "receiveBroadcast GPacket");
        realProtocol.receiveBroadcast(sender, gp);
    }

    public void receiveUnicast(
        com.sun.messaging.jmq.jmsserver.core.BrokerAddress sender,
        int destId, byte []pkt) {
        if (DEBUG)
            logger.log(logger.DEBUG, "receiveUnicast");
        realProtocol.receiveUnicast(sender, destId, pkt);
    }

    public void receiveBroadcast(
        com.sun.messaging.jmq.jmsserver.core.BrokerAddress sender,
        int destId, byte []pkt) {
        if (DEBUG)
            logger.log(logger.DEBUG, "receiveBroadcast");
        realProtocol.receiveBroadcast(sender, destId, pkt);
    }


    public boolean waitForConfigSync() {
        com.sun.messaging.jmq.jmsserver.core.BrokerAddress configServer =
            null;

        try {
            configServer = c.getConfigServer();
        }
        catch (Exception e) {
            return true; // There is config server but it's unreachable.
        }

        if (configServer == null)
            return false; // There is no config server.

        if (configServer.equals(selfAddress))
            return false; // I am the config server.

        if (! getProtocolInitComplete())
            return true;

        return realProtocol.waitForConfigSync();
    }

    public void reloadCluster() {
        waitForProtocolInit();
        realProtocol.reloadCluster();
    }

    public void stopMessageFlow() throws IOException {
        realProtocol.stopMessageFlow();
    }

    public void resumeMessageFlow() throws IOException {
        realProtocol.resumeMessageFlow();
    }

    public void sendMessage(PacketReference pkt, Collection targets,
                 boolean sendMsgDeliveredAck) {
        realProtocol.sendMessage(pkt, targets, sendMsgDeliveredAck);
    }

    public void sendMessageAck(
        com.sun.messaging.jmq.jmsserver.core.BrokerAddress msgHome, 
        com.sun.messaging.jmq.io.SysMessageID mid,
        com.sun.messaging.jmq.jmsserver.core.ConsumerUID cid,
        int ackType, boolean ackack, Map optionalProps, Long txnID) throws BrokerException { 
        realProtocol.sendMessageAck(msgHome, mid, cid, ackType, ackack, optionalProps, txnID);
    }

    public void clientClosed(ConnectionUID conid, boolean notify) {
        if (realProtocol == null)
            return;

        realProtocol.clientClosed(conid, notify);
    }

    public int lockSharedResource(String resId, ConnectionUID owner) {
        if (! getProtocolInitComplete())
            return ProtocolGlobals.G_LOCK_SUCCESS;

        return realProtocol.lockSharedResource(resId, owner);
    }

    public int lockResource(String resId, long timestamp,
        ConnectionUID owner) {
        if (! getProtocolInitComplete())
            return ProtocolGlobals.G_LOCK_SUCCESS;

        return realProtocol.lockResource(resId, timestamp, owner);
    }

    public void unlockResource(String resId) {
        if (! getProtocolInitComplete())
            return;

        realProtocol.unlockResource(resId);
    }

    public void recordUpdateDestination(Destination d)
        throws BrokerException {
        waitForProtocolInit();
        realProtocol.recordUpdateDestination(d);
    }

    public void recordRemoveDestination(Destination d)
        throws BrokerException {
        waitForProtocolInit();
        realProtocol.recordRemoveDestination(d);
    }

    public void sendNewDestination(Destination d)
                  throws BrokerException {
        waitForProtocolInit();
        realProtocol.sendNewDestination(d);
    }

    public void sendRemovedDestination(Destination d)
                  throws BrokerException {
        waitForProtocolInit();
        realProtocol.sendRemovedDestination(d);
    }

    public void sendUpdateDestination(Destination d)
                  throws BrokerException {
        waitForProtocolInit();
        realProtocol.sendUpdateDestination(d);
    }

    public void recordCreateSubscription(Subscription sub)
        throws BrokerException {
        waitForProtocolInit();
        realProtocol.recordCreateSubscription(sub);
    }

    public void recordUnsubscribe(Subscription sub)
        throws BrokerException {
        waitForProtocolInit();
        realProtocol.recordUnsubscribe(sub);
    }

    public void sendNewSubscription(Subscription sub, Consumer cons,
        boolean active) throws BrokerException {
        if (realProtocol == null)
            return;

        realProtocol.sendNewSubscription(sub, cons, active);
    }

    public void sendNewConsumer(Consumer intr, boolean active)
                  throws BrokerException {
        if (realProtocol == null)
            return;

        realProtocol.sendNewConsumer(intr, active);
    }

    public void sendRemovedConsumer(Consumer intr)
                  throws BrokerException {
        if (realProtocol == null)
            return;

        realProtocol.sendRemovedConsumer(intr);
    }

    public void handleGPacket(MessageBusCallback mbcb, BrokerAddress sender, GPacket pkt) {
        if (realProtocol == null) {
            logger.logStack(Logger.ERROR, "No protocol", new Exception("No protocol"));
            return;
        }

        realProtocol.handleGPacket(mbcb, sender, pkt);
    }

	public void preTakeover(String brokerID, UID storeSession, String brokerHost, UID brokerSession) {
        if (realProtocol == null) {
            logger.logStack(Logger.ERROR, "No protocol", new Exception("No protocol"));
            return;
        }

        realProtocol.preTakeover(brokerID, storeSession, brokerHost, brokerSession);
    }
    
	public void postTakeover(String brokerID, UID storeSession, boolean aborted) {
        if (realProtocol == null) {
            logger.logStack(Logger.ERROR, "No protocol", new Exception("No protocol"));
            return;
        }

        realProtocol.postTakeover(brokerID, storeSession, aborted);
    }

    private Object protocolInitWaitObject = new Object();

    private void waitForProtocolInit() {
        synchronized (protocolInitWaitObject) {
            while (! getProtocolInitComplete()) {
                try {
                    protocolInitWaitObject.wait();
                }
                catch (Exception e) {}
            }
        }
    }

    private boolean getProtocolInitComplete() {
        synchronized (protocolInitWaitObject) {
            return protocolInitComplete;
        }
    }

    private void setProtocolInitComplete(boolean protocolInitComplete) {
        synchronized (protocolInitWaitObject) {
            this.protocolInitComplete = protocolInitComplete;
            protocolInitWaitObject.notifyAll();
        }
    }
}
