/*
 * @(#)ProducerHandler.java	1.43 03/25/05
 *
 * Copyright 2000 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms. 
 *
 */

package com.sun.messaging.jmq.jmsserver.data.handlers;

import java.util.*;
import java.io.*;
import java.net.*;
import com.sun.messaging.jmq.jmsserver.data.PacketHandler;
import com.sun.messaging.jmq.io.*;
import com.sun.messaging.jmq.util.DestType;
import com.sun.messaging.jmq.jmsserver.service.Connection;
import com.sun.messaging.jmq.jmsserver.core.Producer;
import com.sun.messaging.jmq.jmsserver.core.ProducerUID;
import com.sun.messaging.jmq.jmsserver.core.DestinationUID;
import com.sun.messaging.jmq.jmsserver.core.Destination;
import com.sun.messaging.jmq.jmsserver.core.Session;
import com.sun.messaging.jmq.jmsserver.core.SessionUID;
import com.sun.messaging.jmq.jmsserver.resources.BrokerResources;
import com.sun.messaging.jmq.jmsserver.util.BrokerException;
import com.sun.messaging.jmq.jmsserver.Globals;
import com.sun.messaging.jmq.jmsserver.service.imq.IMQConnection;
import com.sun.messaging.jmq.jmsserver.util.memory.MemoryGlobals;
import com.sun.messaging.jmq.util.log.*;



/**
 * Handles the create Message
 */
public class ProducerHandler extends PacketHandler 
{


    public ProducerHandler() {
    }

    /**
     * Method to handle Producers
     */
    public boolean handle(IMQConnection con, Packet msg) 
        throws BrokerException 
    {

        Packet reply = new Packet(con.useDirectBuffers());
        reply.setPacketType(msg.getPacketType() + 1);
        reply.setConsumerID(msg.getConsumerID());

        boolean isIndemp = msg.getIndempontent();

        int status = Status.OK;
        String reason = null;

        Hashtable props = null;
        try {
            props = msg.getProperties();
        } catch (Exception ex) {
            throw new RuntimeException("Can not load props", ex);
        }

        Hashtable returnprop = new Hashtable();

        Destination d = null;

        try {

            if (msg.getPacketType() == PacketType.ADD_PRODUCER) {
    
                String dest = (String)props.get("JMQDestination");
                Integer type = (Integer)props.get("JMQDestType");
                assert dest != null : " bad protocol ";
                assert type != null : " bad protocol ";
    

                Long lsessionid = (Long)props.get("JMQSessionID");
                if (lsessionid != null) { // 3.5 protocol
                    SessionUID sessionID = new SessionUID(
                               lsessionid.longValue());
                    // single threaded .. we dont have to worry about
                    // someone else creating it
                    Session session = con.getSession(sessionID);
                    if (session == null) {
                        throw new BrokerException("Internal Error: client sent "
                              + "invalid sessionUID w/ ADD_PRODUCER " 
                               + sessionID + " session does not exist");
                    }
                }
    
                if (!con.isAdminConnection() && MemoryGlobals.MEM_DISALLOW_PRODUCERS) {
                    status = Status.ERROR;
                    reason = "Low memory";
                    logger.log(Logger.WARNING,BrokerResources.W_LOW_MEM_REJECT_PRODUCER);
    
                } else {
    
   
                    if (dest != null && type != null) {
                        while (true) {
                            d = Destination.getDestination(dest, 
                                type.intValue(), true, 
                                !con.isAdminConnection());
                            if (d != null) {
                                try {
                                    d.incrementRefCount();
                                } catch (BrokerException ex) {
                                    // was destroyed under me
                                    // try again
                                    continue;
                                } catch (IllegalStateException ex) {
                                    throw new BrokerException(
                                        Globals.getBrokerResources().getKString(
                                        BrokerResources.X_SHUTTING_DOWN_BROKER),
                                        BrokerResources.X_SHUTTING_DOWN_BROKER,
                                        ex,
                                        Status.ERROR);
                                } 
                             }
                             break; // got a lock on the dest
                        }
                        if (d == null) {
                            logger.log(Logger.DEBUG, "Unable to add "
                                + "producer to "  + dest 
                                + " :" + DestType.toString(type.intValue())
                                + " destination can not be autocreated ");
                            reason = "can not create destination";
                            status = Status.NOT_FOUND;
                        }
                    } else if (dest == null || type == null) {
                        reason = "no destination passed [dest,type] = [" +
                               dest + "," + type + "]";
                        status = Status.ERROR;
                    }
    
                    assert d != null;
    
                    if (status == Status.OK) { // no null destination

                        Producer p = null;
                        boolean active = true;
                        boolean processed = false;
                        if (isIndemp) {
                             p =  Producer.getProducer(msg.getSysMessageID());
                             active = d.isProducerActive(p.getProducerUID());

                        }
                        if (p == null) {
                            p = Producer.createProducer(
                                 d.getDestinationUID(),
                                 con.getConnectionUID(), 
                                 (SysMessageID)msg.getSysMessageID().clone());
    
                            assert p != null;
    
    
 
                            con.addProducer(p);
    
                            if (d != null) {
                                active = d.addProducer(p);
                            }
                        }
    
                        ProducerUID pid = p.getProducerUID();
    
                        assert pid != null;
    
                        returnprop.put("JMQProducerID", new Long(pid.longValue()));
                        returnprop.put("JMQDestinationID", d.getDestinationUID().toString());
                        if (active) {
                            returnprop.put("JMQBytes", new Long(d.getBytesProducerFlow()));
                            returnprop.put("JMQSize", new Integer(d.getSizeProducerFlow()));
                        } else {
                            returnprop.put("JMQBytes", new Long(0));
                            returnprop.put("JMQSize", new Integer(0));
                        }
                    }
                }
    
            } else {
                assert msg.getPacketType() == PacketType.DELETE_PRODUCER;
    
                Long pid_l = (Long)props.get("JMQProducerID");
    
                assert pid_l != null;
    
                ProducerUID pid = new ProducerUID( pid_l == null ? 0
                                   : pid_l.longValue());
    
    
                Producer p = Producer.getProducer(pid);

                if (p == null && isIndemp) {
                    // dont flag error, we already processed it
                } else if (p == null) {
                    logger.log(Logger.INFO,
                       BrokerResources.E_INTERNAL_BROKER_ERROR,
                       "Internal error Unable to find producer "
                         + pid + "\n\t checking if producer was removed recently " +
                          Producer.checkProducer(pid));
                    reason = "unknown producer";
                    status = Status.ERROR;
                } else if (p.getConnectionUID() != con.getConnectionUID()) {
                    logger.log(Logger.INFO,
                       BrokerResources.E_INTERNAL_BROKER_ERROR,
                        " error connection "
                         + "removing producer it doesnt own" 
                         + "\n\tPID=" + pid 
                         + "\n\tconnectionUID of request " + con.getConnectionUID()
                         + "\n\tconnectionUID of creator " + p.getConnectionUID()
                         + "\n\tchecking producer state: " +
                          Producer.checkProducer(pid));
                    reason = "unknown producer";
                    status = Status.ERROR;
                } else {
                    con.removeProducer(pid, "Producer closed requested:\n\tconnection: "
                         + con.getConnectionUID() + "\n\tproducerID: " + pid +
                         "\n\trequest sysmsgid message: " + msg.getSysMessageID());
                }
            }

        } catch (BrokerException ex) {
            status = ex.getStatusCode();
            reason = ex.getMessage();
            logger.log(Logger.INFO, reason);
        } catch (Exception ex) {
            logger.logStack(Logger.INFO,
                 BrokerResources.E_INTERNAL_BROKER_ERROR,
                 "producer message ", ex);
            reason = ex.getMessage();
            status = Status.ERROR;
        } finally {
            if (d != null)
                d.decrementRefCount();
        }


        returnprop.put("JMQStatus", new Integer(status));
        if (reason != null)
            returnprop.put("JMQReason", reason);
        if (con.DUMP_PACKET || con.OUT_DUMP_PACKET)
            returnprop.put("JMQReqID", msg.getSysMessageID().toString());


        reply.setProperties(returnprop);
        con.sendControlMessage(reply);
        return true;
    }

}
