/*
 * @(#)InfoRequestHandler.java	1.7 07/25/05
 *
 * Copyright 2005 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms. 
 *
 */

package com.sun.messaging.jmq.jmsserver.data.handlers;

import java.util.*;
import java.io.*;
import com.sun.messaging.jmq.jmsserver.data.PacketHandler;
import com.sun.messaging.jmq.util.log.Logger;
import com.sun.messaging.jmq.io.*;
import com.sun.messaging.jmq.jmsserver.Globals;
import com.sun.messaging.jmq.jmsserver.resources.BrokerResources;
import com.sun.messaging.jmq.jmsserver.util.BrokerException;
import com.sun.messaging.jmq.jmsserver.cluster.*;
import com.sun.messaging.jmq.jmsserver.service.imq.IMQConnection;


/**
 * Handler class which deals with the INFO_REQUEST message 
 * sent by a client after failover.
 */
public class InfoRequestHandler extends PacketHandler 
{

    private Logger logger = Globals.getLogger();
    private BrokerResources rb = Globals.getBrokerResources();

    public InfoRequestHandler()
    {
    }


    public static final int REQUEST_STATUS_INFO = 1;
    public static final int REQUEST_CLUSTER_INFO = 2;

    /**
     * Method to handle INFO_REQUEST messages
     */
    public boolean handle(IMQConnection con, Packet msg) 
        throws BrokerException 
    { 
          Hashtable pktprops = null;

          try {
              pktprops = msg.getProperties();
          } catch (Exception ex) {
              
              logger.log(Logger.INFO,"Internal Error: error "
                   + " retrieving properties from INFO_REQUEST message ", ex);
              pktprops = new Hashtable();

          }

          Integer level = (Integer)pktprops.get("JMQRequestType");

          if (level == null) {
              logger.log(logger.INFO,"No JMQRequestType set ");
              level = new Integer(-1); //pick and invalid value
          }
          sendInfoPacket(level.intValue(), con, msg.getConsumerID());
          return true;

    }


    // used by other method to get the INFO packet

    public static void sendInfoPacket(int requestType, IMQConnection con, long consumerUID) {
          Logger logger = Globals.getLogger();
          ClusterManager cfg =  Globals.getClusterManager();
          String reason = null;
          int status = Status.OK;

          Packet pkt = new Packet(con.useDirectBuffers());
          pkt.setPacketType(PacketType.INFO);
          pkt.setConsumerID(consumerUID);
          Hashtable replyprops = new Hashtable();        

          Hashtable bodyProperties = null;

          if (cfg == null) {
              status = Status.ERROR;
              reason = "Internal Error: NOT VALID CLUSTER ";
              logger.log(Logger.INFO,reason);
          } else if (requestType == REQUEST_STATUS_INFO) {
               ClusteredBroker cb = cfg.getLocalBroker();
               // OK, set properties
              bodyProperties = getInfo(cb);
              try {
                  if (cb.getState() == BrokerState.SHUTDOWN_STARTED) {
                      long timeout = Globals.getBrokerStateHandler()
                              .getShutdownRemaining(); 
                      bodyProperties.put("ShutdownMS", new Long(timeout));
                  }
              } catch (BrokerException ex) {
                  logger.logStack(Logger.INFO,"Internal Error: setting timeout ", ex);
              }
          } else if (requestType == REQUEST_CLUSTER_INFO) {
              bodyProperties = getAllInfo();
          } else {
              status = Status.BAD_REQUEST;
              reason = "Internal Error: Bad JMQRequestType set " + requestType;
              logger.log(Logger.INFO,reason);
          }

//XXX - TBD
// should we write out the contents in a less intensive way
// (e.g. use the same format used for properties)
// or just serialize Hashtable (which is whay we are doing now)
//

          String list = null;
          Iterator itr = cfg.getKnownBrokers();
          Set s = new HashSet();
          // ok get rid of dups
          while (itr.hasNext()) {
              ClusteredBroker cb = (ClusteredBroker)itr.next();
              s.add(cb.getBrokerURL().toString());
          }   
              // OK .. now convert to a string

          itr = s.iterator();

          while (itr.hasNext()) {
              if (list == null) {
                  list = itr.next().toString();
              } else {
                  list += "," + itr.next().toString();
              }
          }   
          if (list != null) {
              replyprops.put("JMQBrokerList", list);
          }   


          if (bodyProperties != null) {
              try {
                  ByteArrayOutputStream bos = new ByteArrayOutputStream();
                  ObjectOutputStream oos = new ObjectOutputStream(bos);
                  oos.writeObject(bodyProperties);
                  oos.flush();
                  bos.flush();
                  pkt.setMessageBody(bos.toByteArray());
                  bos.close();
              } catch (Exception ex) {
                   logger.log(Logger.WARNING,"XXX-L10N Internal Error "
                         + " sending back broker data", ex);
              }
          }
          // OK deal with properties, etc
          replyprops.put("JMQRequestType", new Integer(requestType));
          replyprops.put("JMQStatus", new Integer(status));
          if (reason != null) {
              replyprops.put("JMQReason", reason);
          }
          pkt.setProperties(replyprops);
	  con.sendControlMessage(pkt);
    }


    static Hashtable getAllInfo() {

        ClusterManager cfg =  Globals.getClusterManager();
        if (cfg == null) return null;
        Iterator itr = cfg.getKnownBrokers();
        Hashtable ht = new Hashtable();
        String nameList = "";

        while (itr.hasNext()) {
            ClusteredBroker bc = (ClusteredBroker)itr.next();
            if (nameList.length() == 0) {
                nameList = bc.getBrokerName();
            } else {
                nameList += ","+bc.getBrokerName();
            }
            ht.put(bc.getBrokerName(), getInfo(bc));
        }
        ht.put("BrokerIDs", nameList);
        return ht;
    }

    static Hashtable getInfo(ClusteredBroker bc) 
    {

        BrokerState state = null;
        Logger logger = Globals.getLogger();
        ClusterManager cfg =  Globals.getClusterManager();

        try {
            state = bc.getState();
        } catch (BrokerException ex) {
            logger.log(Logger.INFO,"L10N-XXX Internal Error "
                     + " unable to retrieve state", ex);
            state = BrokerState.INITIALIZING;
        }

        Hashtable ht = new Hashtable();
        ht.put("BrokerID", bc.getBrokerName());
        ht.put("brokerURL", bc.getBrokerURL().toString());
        ht.put("State", new Integer( mapStateToInt(state) ));
        ht.put("StateString", state.toString());        
        ht.put("isLocal", new Boolean( bc.isLocalBroker()));
        return ht;
    }

    static final int INITIALIZING = 1;
    static final int RUNNING = 2;
    static final int QUIESCE_STARTED = 3;
    static final int QUIESCE_COMPLETE = 4;
    static final int FAILOVER_STARTED = 5;
    static final int FAILOVER_COMPLETE = 6;
    static final int SHUTDOWN_STARTED = 7;
    static final int SHUTDOWN_COMPLETE = 8;

    static int mapStateToInt(BrokerState state)
    {
        if (state == BrokerState.INITIALIZING) {
            return INITIALIZING;
        } else if (state == BrokerState.OPERATING) {
            return RUNNING;
        } else if (state == BrokerState.QUIESCE_STARTED) {
            return QUIESCE_STARTED;
        } else if (state == BrokerState.QUIESCE_COMPLETED) {
            return QUIESCE_COMPLETE;
        } else if (state == BrokerState.SHUTDOWN_STARTED) {
            return SHUTDOWN_STARTED;
        } else if (state == BrokerState.SHUTDOWN_FAILOVER) {
            return SHUTDOWN_COMPLETE;
        } else if (state == BrokerState.SHUTDOWN_COMPLETE) {
            return SHUTDOWN_COMPLETE;
        } else if (state == BrokerState.FAILOVER_PENDING) {
            return FAILOVER_STARTED;
        } else if (state == BrokerState.FAILOVER_STARTED) {
            return FAILOVER_STARTED;
        } else if (state == BrokerState.FAILOVER_COMPLETE) {
            return FAILOVER_COMPLETE;
        } else if (state == BrokerState.FAILOVER_FAILED) {
            return FAILOVER_COMPLETE;
        }
        return 0;
    }



}

