/*
 * @(#)DestinationHandler.java	1.40 11/18/05
 *
 * Copyright 2000 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms. 
 *
 */

package com.sun.messaging.jmq.jmsserver.data.handlers;

import java.util.*;
import java.io.*;
import com.sun.messaging.jmq.jmsserver.data.PacketHandler;
import com.sun.messaging.jmq.jmsserver.core.DestinationUID;
import com.sun.messaging.jmq.jmsserver.core.Destination;
import com.sun.messaging.jmq.io.*;
import com.sun.messaging.jmq.util.DestType;
import com.sun.messaging.jmq.jmsserver.service.Connection;
import com.sun.messaging.jmq.jmsserver.util.BrokerException;
import com.sun.messaging.jmq.jmsserver.resources.BrokerResources;
import com.sun.messaging.jmq.util.log.Logger;
import com.sun.messaging.jmq.jmsserver.Globals;
import com.sun.messaging.jmq.jmsserver.service.imq.IMQConnection;



/**
 * Handler class which deals with adding and removing destination from the broker
 */
public class DestinationHandler extends PacketHandler 
{
    private Logger logger = Globals.getLogger();
    public static boolean DEBUG = false;
  

    public DestinationHandler() {
    }

    /**
     * Method to handle Destination (create or delete) messages
     */
    public boolean handle(IMQConnection con, Packet msg) 
        throws BrokerException
    {

        int status = Status.OK;
        String reason = null;

        // XXX - REVISIT 2/25/00 racer
        // do we need to create a reply packet each time ?

        Packet pkt = new Packet(con.useDirectBuffers());
        pkt.setConsumerID(msg.getConsumerID());
        Hashtable hash = new Hashtable();

        Hashtable props = null;
        try {
            props = msg.getProperties();
        } catch (Exception ex) {
            assert false;
            logger.logStack(Logger.ERROR, BrokerResources.E_INTERNAL_BROKER_ERROR, "Unable to create/destroy destination - no properties",ex);
            throw new BrokerException(Globals.getBrokerResources().getString(
           BrokerResources.X_INTERNAL_EXCEPTION,"Can not handle create/destroy destination"));
        }
    
        String destination = (String )props.get("JMQDestination");

        Integer inttype = (Integer )props.get("JMQDestType");

        int type = (inttype == null) ? 0 : inttype.intValue();

        pkt.setPacketType(msg.getPacketType() + 1);
        if (msg.getPacketType() == PacketType.CREATE_DESTINATION) {

            if (DEBUG) {
                logger.log(Logger.DEBUGHIGH, "ConsumerHandler: handle() [ Received AddDestination message {0}]", msg.toString());
            }

            assert destination != null;
            assert inttype != null;

            if (con.isAdminConnection()) {
               type |= DestType.DEST_ADMIN | DestType.DEST_LOCAL 
                     | DestType.DEST_AUTO;
            }
            assert pkt.getPacketType() == PacketType.CREATE_DESTINATION_REPLY;
            try {
                Destination d = null;
                if (DestType.isTemporary(type)) {
                    // deal w/ versioning .. only store
                    // 3.5 or later
                    boolean storeTemps = con.getConnectionUID().
                            getCanReconnect();
                    long reconnectTime = con.getReconnectInterval();
                    d = Destination.createTempDestination(
                         destination, type, con.getConnectionUID(), 
                         storeTemps, reconnectTime);
                    if (con.getConnectionUID().equals(d.getConnectionUID())) {
                        con.attachTempDestination(d.getDestinationUID());
                    }

                } else if (destination.startsWith(Globals.INTERNAL_PREFIX)) {
                    // do nothing
                } else {
                    d = Destination.getDestination(destination, type, true, !con.isAdminConnection());
                }

                hash.put("JMQDestType", new Integer(type));
                // XXX-LKS - should be use UID not a string ???
                hash.put("JMQDestUID", destination);
            } catch (BrokerException ex) {
                status = ex.getStatusCode();
                reason = ex.getMessage();
                if (status != Status.CONFLICT) {
                    logger.log(Logger.WARNING, 
                        BrokerResources.W_CREATE_DEST_FAILED, destination, ex);
                } else if (DEBUG) {
                    logger.log(Logger.DEBUG, 
                        BrokerResources.W_CREATE_DEST_FAILED, destination, ex);
                }
            } catch (IOException ex) {
                status = Status.ERROR;
                reason = ex.getMessage();
                logger.log(Logger.WARNING, 
                    BrokerResources.W_CREATE_DEST_FAILED, destination, ex);
            }
        } else { // removing Interest
            assert msg.getPacketType() == PacketType.DESTROY_DESTINATION;
            assert pkt.getPacketType() == PacketType.DESTROY_DESTINATION_REPLY;

            Destination d =null;

            try {
                DestinationUID rmuid = DestinationUID.getUID(destination, DestType.isQueue(type));

                if (destination == null) {
                    throw new BrokerException(
                        Globals.getBrokerResources().getString(
                           BrokerResources.X_INTERNAL_EXCEPTION,
                       "protocol error,  destination is null"),
                           Status.NOT_FOUND);
                }
                assert Destination.getDestination(rmuid) != null;
                Destination.removeDestination(rmuid, true, 
                     Globals.getBrokerResources().getString(
                        BrokerResources.M_CLIENT_REQUEST, con.getConnectionUID()));
            } catch (BrokerException ex) {
                status = ex.getStatusCode();
                reason = ex.getMessage();
                logger.log(Logger.WARNING, 
                    BrokerResources.W_DESTROY_DEST_FAILED, destination,ex);
            } catch (IOException ex) {
                status = Status.ERROR;
                reason = ex.getMessage();
                logger.log(Logger.WARNING, 
                    BrokerResources.W_DESTROY_DEST_FAILED, destination,ex);
            }

        }
        hash.put("JMQStatus", new Integer(status));
        if (reason != null)
            hash.put("JMQReason", reason);
        if (con.DUMP_PACKET || con.OUT_DUMP_PACKET)
            hash.put("JMQReqID", msg.getSysMessageID().toString());


        pkt.setProperties(hash);
        con.sendControlMessage(pkt);
        return true;
    }

}
