/*
 * @(#)DataHandler.java	1.76 07/25/05
 *
 * Copyright 2000 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms. 
 *
 */

package com.sun.messaging.jmq.jmsserver.data.handlers;

import java.util.*;
import java.io.*;
import com.sun.messaging.jmq.jmsserver.data.TransactionUID;
import com.sun.messaging.jmq.jmsserver.data.TransactionList;
import com.sun.messaging.jmq.jmsserver.data.PacketRouter;
import com.sun.messaging.jmq.jmsserver.data.PacketHandler;
import com.sun.messaging.jmq.jmsserver.core.Destination;
import com.sun.messaging.jmq.jmsserver.core.Producer;
import com.sun.messaging.jmq.jmsserver.core.ProducerUID;
import com.sun.messaging.jmq.io.*;
import com.sun.messaging.jmq.jmsserver.service.Connection;
import com.sun.messaging.jmq.jmsserver.service.imq.IMQConnection;
import com.sun.messaging.jmq.jmsserver.util.BrokerException;
import com.sun.messaging.jmq.jmsserver.core.Consumer;
import com.sun.messaging.jmq.jmsserver.core.PacketReference;
import com.sun.messaging.jmq.jmsserver.util.BrokerException;
import com.sun.messaging.jmq.jmsserver.resources.BrokerResources;
import com.sun.messaging.jmq.jmsserver.Globals;
import com.sun.messaging.jmq.jmsserver.FaultInjection;
import com.sun.messaging.jmq.jmsserver.Broker;
import com.sun.messaging.jmq.jmsserver.BrokerStateHandler;
import com.sun.messaging.jmq.jmsserver.util.memory.MemoryGlobals;
import com.sun.messaging.jmq.util.log.Logger;



/**
 * Handler class which deals with normal (topic or queue) JMS messages
 */
public class DataHandler extends PacketHandler 
{
    TransactionList translist = null;
    protected Logger logger = Globals.getLogger();
    public static boolean DEBUG = false;
    

    private int msgProcessCnt = 0; // used for fault injection
    private FaultInjection fi = null;

    public DataHandler(TransactionList list) 
    {
        this.translist = list;
        fi = FaultInjection.getInjection();
        
    }

    /**
     * Method to handle normal data messages
     */
    public boolean handle(IMQConnection con, Packet msg) 
        throws BrokerException
    {
        boolean sentReply = false;

        return handle(con, msg, false);

    }
 
    /**
     * Method to handle normal/admin data messages
     */
    protected boolean handle(IMQConnection con, Packet msg, boolean isadmin) 
        throws BrokerException
    {
        if (DEBUG) {
            logger.log(Logger.DEBUGHIGH, "DataHandler: handle() [ Received JMS  Message] {0} ", msg.toString());
        }


        Hashtable props = null; // used for fault injection
        if (!isadmin && fi.FAULT_INJECTION) {
           msgProcessCnt ++; // for fault injection
           try {
               props = msg.getProperties();
           } catch (Exception ex) {
               props = new Properties();
           }
        } else {
           msgProcessCnt = 0;
        }


        boolean ack = msg.getSendAcknowledge();
        long cid = msg.getConsumerID();
        String refid = (con.DUMP_PACKET || con.OUT_DUMP_PACKET) ?
                  msg.getSysMessageID().toString() : "";

        boolean isIndemp = msg.getIndempontent();

       String reason = null;
       PacketReference ref = null;
       int status = Status.OK;
       boolean removeMsg = false;
       Set s = null;
       Destination d= null;
       boolean route = false;
       Producer pausedProducer = null;
       boolean transacted = false;
       try {
           d = Destination.getDestination(
                  msg.getDestination(), msg.getIsQueue());
           if (d == null) {
               throw new BrokerException("Unknown Destination:" + msg.getDestination());
           }


           
           // check and clearout the F bit (before anything)
           if (msg.getFlowPaused()) {
               con.flowPaused(0);
               msg.setFlowPaused(false);
           } 
           long pid = msg.getProducerID();
           ProducerUID puid = new ProducerUID(pid);
           Producer p = Producer.getProducer(puid);
           if (p != null)
               p.addMsg(); // increment counter
           // see if we need to resume flow
           if (msg.getConsumerFlow()) {
               pausedProducer = p;
               if (pausedProducer == null) {
                   logger.log(Logger.INFO,"Internal Error: Unknown ProducerUID " + puid);
               } else if (pausedProducer.getConnectionUID() != con.getConnectionUID()) {
                   logger.log(Logger.INFO,"Internal Error: Producer " + pausedProducer
                      + " not on this connection " + con.getConnectionUID());

               }
               msg.setConsumerFlow(false);
           }

           transacted = (msg.getTransactionID() != 0);


            // OK .. handle Fault Injection
            if (!isadmin && fi.FAULT_INJECTION) {
                Map m = new HashMap();
                if (props != null)
                    m.putAll(props);
                m.put("mqMsgCount", new Integer(msgProcessCnt));
                m.put("mqIsTransacted", new Boolean(transacted));
                fi.checkFaultAndExit(FaultInjection.FAULT_SEND_MSG_1,
                     m, 2, false);
            }


            // OK generate a ref. This checks message size and
            // will be needed for later operations
            ref = PacketReference.createReference(msg, con);
            if (isadmin) {
                ref.overridePersistence(false);
            }
            // dont bother calling route if there are no messages
            //
            // to improve performance, we route and later forward
            route = d.queueMessage(ref, transacted);


            // ok ... 
            if (route && ack && !ref.isPersistent()) {
                sendAcknowledge(refid, cid, status, con, reason, props, transacted);
                ack = false;
            }

            if (transacted) {
                // if we are transacted, we just store the msg and
                // go on (we dont route)
                try {
                    ref.store();
                    translist.addMessage(ref.getTransactionID(), 
                        ref.getSysMessageID());
                } catch (Exception ex) {
                    ref.destroy();
                    logger.logStack((BrokerStateHandler.shuttingDown? 
                            Logger.DEBUG : Logger.WARNING),
                            BrokerResources.E_INTERNAL_BROKER_ERROR, 
                            "transaction failed", ex);
                    reason = "transaction failed: " + ex.getMessage();
                    status = Status.ERROR;
                }
            } else {
                if (route)
                    s = d.routeNewMessage(ref);
            }

            // handle producer flow control
            if (pausedProducer != null)
                d.producerFlow(con, pausedProducer);

        } catch (BrokerException ex) {

            // dont log on dups if indemponent
            int loglevel = (isIndemp && ex.getStatusCode() 
                   == Status.NOT_MODIFIED) ? Logger.DEBUG 
                      : Logger.WARNING;
            logger.log(loglevel, 
                      BrokerResources.W_MESSAGE_STORE_FAILED,
                      con.toString(), ex);
            reason =  ex.getMessage();
            status = ex.getStatusCode();
        } catch (IOException ex) {
            logger.log(Logger.WARNING, BrokerResources.W_MESSAGE_STORE_FAILED,
                      con.toString(), ex);
            reason =  ex.getMessage();
            status = Status.ERROR;
        } catch (SecurityException ex) {
            logger.log(Logger.WARNING, BrokerResources.W_MESSAGE_STORE_FAILED,
                      con.toString(), ex);
            reason =  ex.getMessage();
            status = Status.FORBIDDEN;
        } catch (OutOfMemoryError err) {
            logger.logStack(Logger.WARNING, BrokerResources.W_MESSAGE_STORE_FAILED,
                      con.toString() + ":" + msg.getPacketSize(), err);
            reason =  err.getMessage();
            status = Status.ERROR;
        } catch (Exception ex) {

            logger.logStack(Logger.WARNING, BrokerResources.W_MESSAGE_STORE_FAILED,
                      con.toString(), ex);
             reason =  ex.getMessage();
             status = Status.ERROR;
        }

        if (status == Status.ERROR && ref != null && d != null ) {
            // make sure we remove the message
            try {
                d.removeMessage(ref.getSysMessageID(), null);
            } catch (Throwable thr) {
               // we should never throw anything .. but this is
               // a just in case [since we dont have any one else
               // to catch it in this case]
               // if something goes wrong removing the message 
               // there is nothing we can do
            }
        }

        if (ack)
            sendAcknowledge(refid, cid, status, con, reason, props, transacted);

        if (route && d != null && s != null) {
            //con.forwardMsg(s,ref);
            d.forwardMessage(s, ref);
        }
 
        return isadmin; // someone else will free

    }

    public void sendAcknowledge(String refid,
          long cid, int status, IMQConnection con,
          String reason, Hashtable props /* fi only */, 
          boolean transacted /*fi only */)
    {
            // OK .. handle Fault Injection
            if (!con.isAdminConnection() && fi.FAULT_INJECTION) {
                Map m = new HashMap();
                if (props != null)
                    m.putAll(props);
                m.put("mqMsgCount", new Integer(msgProcessCnt));
                m.put("mqIsTransacted", new Boolean(transacted));
                fi.checkFaultAndExit(FaultInjection.FAULT_SEND_MSG_2,
                     m, 2, false);
            }
        // send the reply (if necessary)
            Packet pkt = new Packet(con.useDirectBuffers());
            pkt.setPacketType(PacketType.SEND_REPLY);
            pkt.setConsumerID(cid);
            Hashtable hash = new Hashtable();
            hash.put("JMQStatus", new Integer(status));
            if (reason != null)
                hash.put("JMQReason", reason);
            if (con.DUMP_PACKET || con.OUT_DUMP_PACKET)
                hash.put("JMQReqID", refid);
            pkt.setProperties(hash);
            con.sendControlMessage(pkt);
            // OK .. handle Fault Injection
            if (!con.isAdminConnection() && fi.FAULT_INJECTION) {
                Map m = new HashMap();
                if (props != null)
                    m.putAll(props);
                m.put("mqMsgCount", new Integer(msgProcessCnt));
                m.put("mqIsTransacted", new Boolean(transacted));
                fi.checkFaultAndExit(FaultInjection.FAULT_SEND_MSG_3,
                     m, 2, false);
            }
    }


}
