/*
 * @(#)PacketRouter.java	1.29 05/19/05
 *
 * Copyright 2000 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms. 
 *
 */

package com.sun.messaging.jmq.jmsserver.data;


import com.sun.messaging.jmq.io.*;
import java.security.AccessControlException;
import com.sun.messaging.jmq.jmsserver.service.*;
import com.sun.messaging.jmq.jmsserver.service.imq.IMQConnection;
import com.sun.messaging.jmq.jmsserver.util.BrokerException;
import com.sun.messaging.jmq.jmsserver.util.PacketUtil;
import com.sun.messaging.jmq.jmsserver.auth.AccessController;
import com.sun.messaging.jmq.jmsserver.resources.BrokerResources;
import com.sun.messaging.jmq.jmsserver.Globals;
import com.sun.messaging.jmq.util.log.Logger;

/**
 * Class which handles passing messages read in from
 * a protocol to the correct Message Handler
 */


public class PacketRouter
{

    /**
     * dump all packets comming on
     */
    public static boolean DEBUG = false;

    private final Logger logger = Globals.getLogger();

    /**
     * list of message handlers for specific message types 
     */
    private PacketHandler list[] = new PacketHandler[PacketType.LAST];

    /**
     * default handler which handles errors and messages which do not
     * have handlers registered 
     */
    private ErrHandler defaultHandler = null;

    /**
     * Constructor for PacketRouter class.
     */
    public PacketRouter() {
	defaultHandler = new DefaultHandler();
    }

    /**
     * registers a new handler for a specific message id
     *
     * @param id the messageID
     * @param handler the handler to add
     * @throws ArrayIndexOutOfBoundsException if the id is too high
     */
    public void addHandler(int id, PacketHandler handler) 
        throws ArrayIndexOutOfBoundsException 
    {
        if (id > PacketType.LAST) {
            throw new ArrayIndexOutOfBoundsException(Globals.getBrokerResources().getString(
           BrokerResources.X_INTERNAL_EXCEPTION, "Trying to add handler which has no corresponding packet type [ " + id + "]"));
        }
        list[id] = handler;
    }

    /**
     * registers a new handler for a group of message id
     *
     * @param sid the starting message id
     * @param eid the ending message id
     * @param handler the handler to add
     * @throws ArrayIndexOutOfBoundsException if the id is too high
     */
    public void addHandler(int sid, int eid, PacketHandler handler) 
        throws ArrayIndexOutOfBoundsException 
    {
        // NOTE: this is not that efficient, but it should ONLY happen at initialization
        // so I'm not worrying about it
        for (int i = sid; i < eid; i ++ )
            addHandler(i, handler);

    }

    /**
     * Return the handler for a specific packet type.
     *
     * @param id the packet type
     * @throws ArrayIndexOutOfBoundsException if the id is too high
     */
    public PacketHandler getHandler(int id)
        throws ArrayIndexOutOfBoundsException 
    {
        if (id > PacketType.LAST) {
            throw new ArrayIndexOutOfBoundsException(id);
        }
        return list[id];
    }

    /**
     * This routine handles passing messages off to the correct handler.
     * Messages should be first checked for authorization (if any)
     * and then sent to the correct handler.
     * If there is no handler, it should be sent to the default
     * handler (aka error handler).
     *
     * @param con the connection the message was received from
     * @param msg the message received
     *
     */
    public void handleMessage(IMQConnection con, Packet msg) {
        int id = msg.getPacketType();

        if (id < 0) {
            logger.log(Logger.ERROR, Globals.getBrokerResources().getString(
           BrokerResources.X_INTERNAL_EXCEPTION, "invalid packet type {0}",
                    String.valueOf(id)));
            defaultHandler.sendError(con, ErrHandler.ERROR, "invalid packet type " + id, msg);
            return;
        }

        PacketHandler handler = null;

        if (id >= PacketType.LAST) {
            handler = defaultHandler;
        } else {
            handler = list[id];
        }

        if (handler == null) {
            handler = defaultHandler;
        }
        if (handler != defaultHandler &&
            !accessControlCheck(msg, con, handler, id)) {
            return;
        }
        try {
            boolean freepkt = handler.handle(con, msg);
            if (freepkt == true) {
                msg.destroy();
                msg = null;
            }
        } catch (BrokerException ex) {
            assert defaultHandler != null;

            if (defaultHandler != null)
                defaultHandler.sendError(con, ex, msg);
        } catch (Exception ex) {
                defaultHandler.sendError(con, new BrokerException(
                    Globals.getBrokerResources().getKString(
                    BrokerResources.X_INTERNAL_EXCEPTION,
                    "Unexpected Error processing message"), ex), msg);
        }
    }

    private boolean accessControlCheck(Packet msg, IMQConnection con,
                                          PacketHandler handler, int pktype) {
        AccessController ac = con.getAccessController(); 
        if (pktype != PacketType.HELLO && pktype != PacketType.PING &&
                pktype != PacketType.AUTHENTICATE &&
                       pktype != PacketType.GOODBYE) {

            if (!ac.isAuthenticated()) {
                String emsg = Globals.getBrokerResources().getKString(
                        BrokerResources.E_UNEXPECTED_PACKET_NOT_AUTHENTICATED,
                                                 PacketType.getString(pktype));
                if (defaultHandler != null) {
                defaultHandler.sendError(con, ErrHandler.ERROR, emsg, msg); 
                }
                else {
                logger.log(Logger.ERROR, emsg);
                }
                return false;
            }
            try {
                handler.checkPermission(msg, con);
                return true;
            } catch (AccessControlException e) {
                try {                 
                    handler.handleForbidden(con, msg, pktype+1);
                } catch (BrokerException ex) {
                    assert defaultHandler != null;
                    if (defaultHandler != null)
                        defaultHandler.sendError(con, ex, msg);
                } catch (Exception ex) {
                    if (defaultHandler != null) {
                    defaultHandler.sendError(con, 
			new BrokerException(
                            Globals.getBrokerResources().getKString(
                            BrokerResources.X_INTERNAL_EXCEPTION,
                           "Unexpected Error processing message"), ex), msg);
                    }
                }
            } catch (Exception ex) {
                if (defaultHandler != null) {
                defaultHandler.sendError(con,
                    new BrokerException(
                       Globals.getBrokerResources().getKString(
                       BrokerResources.X_INTERNAL_EXCEPTION,
                       "Unexpected Error processing message"), ex), msg);
                }
            }
            return false;
        }    
        else {
            return true;
        }
    }

}
