/*
 * @(#)PacketHandler.java	1.39 10/17/05
 *
 * Copyright 2004 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms. 
 *
 */

package com.sun.messaging.jmq.jmsserver.data;

import java.io.IOException;
import java.util.Hashtable;
import java.security.Principal;
import java.security.AccessControlException;
import com.sun.messaging.jmq.io.*;
import com.sun.messaging.jmq.util.DestType;
import com.sun.messaging.jmq.util.ServiceType;
import com.sun.messaging.jmq.util.log.*;
import com.sun.messaging.jmq.jmsserver.resources.BrokerResources;
import com.sun.messaging.jmq.jmsserver.service.*;
import com.sun.messaging.jmq.jmsserver.service.imq.*;
import com.sun.messaging.jmq.jmsserver.auth.AccessController;
import com.sun.messaging.jmq.jmsserver.util.*;
import com.sun.messaging.jmq.jmsserver.Globals;
import com.sun.messaging.jmq.jmsserver.core.Destination;
import com.sun.messaging.jmq.jmsserver.GlobalProperties;
import com.sun.messaging.jmq.util.admin.MessageType;
import com.sun.messaging.jmq.jmsserver.core.DestinationUID;

/**
 * super classes which deal with handling specific
 * message types
 */
public abstract class PacketHandler 
{
    public static boolean DEBUG = false;
    protected Logger logger = Globals.getLogger();

    /**
     * method to handle processing the specific packet associated
     * with this PacketHandler
     * @returns true if the packet can be freed
     */
    public abstract boolean handle(IMQConnection con, Packet msg) throws
        BrokerException;

    public void handleForbidden(IMQConnection con, Packet msg,
                                int replyType) throws BrokerException {
        Packet reply = new Packet(con.useDirectBuffers());
        if (DEBUG) {
        logger.log(Logger.DEBUG, "handle forbidden: sending "+PacketType.getString(replyType));
        }
        reply.setPacketType(replyType);
        reply.setConsumerID(msg.getConsumerID());
        Hashtable hash = new Hashtable();
        hash.put("JMQStatus", new Integer(Status.FORBIDDEN));
        reply.setProperties(hash);
        con.sendControlMessage(reply);
    }
   
    /**
     * entry point for destination access control check
     */
    public void checkPermission(Packet msg, IMQConnection con)
                            throws AccessControlException, IOException,
                                   ClassNotFoundException, BrokerException
    {
        int id = msg.getPacketType();
        String op = PacketType.mapOperation(id);
        if (op == null) {
            return;
        }
        Hashtable prop = msg.getProperties();

        String destination = (String)prop.get("JMQDestination");
        //all non-null op should have destination
        if (destination == null) { 
            throw new BrokerException(Globals.getBrokerResources().getString(
               BrokerResources.X_INTERNAL_EXCEPTION,
               "checkPermission() no JMQDestionation"));
        }

        Integer dtype = (Integer)prop.get("JMQDestType");
        if (dtype == null) {
            throw new BrokerException(Globals.getBrokerResources().getString(
                BrokerResources.X_INTERNAL_EXCEPTION,
                "checkPermission() no JMQDestType"));
        }
        int destTypeInt = dtype.intValue();

        //Temporary destination should return null
        String destTypeStr = DestType.queueOrTopic(destTypeInt);
        if (destTypeStr == null) { 
            return;
        }

        Service service  = con.getService();
        int serviceType = service.getServiceType();

        if (!checkIsNonAdminDest(con, service, serviceType, destination)) {
            return;
        }

        String acdestination = destination;

        //if autocreate false, return to normal path
        if (id == PacketType.CREATE_DESTINATION) {
            if (!checkIsAutoCreate(destination, destTypeInt)) { 
                return;
            }
            DestinationUID duid = DestinationUID.getUID(destination, DestType.isQueue(destTypeInt));
            if (duid == null) {
                throw new BrokerException(Globals.getBrokerResources().getString(
                    BrokerResources.X_INTERNAL_EXCEPTION,
                    "checkPermission() can't get destination uid for " + destination));
            }
            Destination d = Destination.getDestination(duid);
            if (d != null && !d.isAutoCreated()) {
                return;
            }
            acdestination = null;
        } 
        checkPermission(con, service, serviceType,
                        op, acdestination, destTypeStr, destination);

//XXX:Audit:Removed
//	// audit logging for destination authorization
//	Globals.getAuditSession().destinationAuth(
//			con.getUserName(), con.remoteHostString(),
//			destTypeStr, acdestination, op, true);
   }

    /**
     * call this on DESTINATION_CREATE checkPermission
     * @return true if need access control check on create
     *         false if no need access control check
     */
    private boolean checkIsAutoCreate(String destination, int destType) { 
        if (DestType.isQueue(destType)) {
            if (!GlobalProperties.getGlobalProperties().AUTOCREATE_QUEUE) {
                return false;
            }
        }
        else if (DestType.isTopic(destType)) {
            if (!GlobalProperties.getGlobalProperties().AUTOCREATE_TOPIC) {
                return false;
            }
        }
        return true;
    }

    /**
     * @return true destination is not JMQ_ADMIN_DEST
     *         false ADMIN service access JMQ_ADMIN_DEST
     * @exception non ADMIN service access JMQ_ADMIN_DEST
     * @exception restricted ADMIN service access non JMQ_ADMIN_DEST
     */
    private boolean checkIsNonAdminDest(IMQConnection con,
                    Service service, int serviceType, String destination)
                            throws AccessControlException, BrokerException {

        if (!destination.equals(MessageType.JMQ_ADMIN_DEST)) {
            if (serviceType == ServiceType.ADMIN 
                && con.getAccessController().isRestrictedAdmin()) {
            String emsg = Globals.getBrokerResources().getKString(
                        BrokerResources.X_RESTRICTED_ADMIN_NON_JMQADMINDEST,
                        destination, service.getName());
            logger.log(Logger.WARNING, emsg);
            throw new AccessControlException(emsg);
            }
            return true;
        }
        /*
         * Protect JMQ_ADMIN_DEST to ADMIN service only
         * ADMIN service (when get here the connection has been
         * authenticated and service type connection access control
         * has been applied) should automatically to be allowed to
         * access JMQ_ADMIN_DEST
         */
        if (serviceType == ServiceType.ADMIN) {
            return false;
        }
        String name = "";
        Principal pp = con.getAccessController().getAuthenticatedName();
        if (pp!= null) {
            name = pp.getName();
        }
        String[] args = {name, service.getName(),
                         ServiceType.getServiceTypeString(serviceType)};
        String emsg = Globals.getBrokerResources().getKString(
                       BrokerResources.X_FORBIDDEN_JMQ_ADMIN_DEST, args);
        logger.log(Logger.WARNING, emsg);
        throw new AccessControlException(emsg);
    }

    /**
     * delegate to AccessController.checkDestinationPermission
     *
     * @param con connection
     * @param service
     * @param serviceType
     * @param op operation 
     * @param destination null if op = create otherwise = dest
     * @param destTypeStr 
     * @param dest the destination as JMQDestination property
     */
    private void checkPermission(IMQConnection con, 
                                 Service service, 
                                 int serviceType,
                                 String op, 
                                 String destination,
                                 String destType,
                                 String dest) 
                                 throws AccessControlException {
        try {

        con.getAccessController().checkDestinationPermission(
                      service.getName(),
                      ServiceType.getServiceTypeString(serviceType),
                      op, destination, destType);

        } catch (AccessControlException e) {

        if (destination !=  null) {
        String[] args = {op, destType, destination};
        String emsg = Globals.getBrokerResources().getKString(
                         BrokerResources.W_DESTINATION_ACCESS_DENIED, args);
        logger.log(Logger.WARNING, emsg + " - " + e.getMessage(), e);
        }
        else { //AC_DESTCREATE
        String[] args = {op, destType, dest};
        String emsg = Globals.getBrokerResources().getKString(
                         BrokerResources.W_DESTINATION_CREATE_DENIED, args);
        logger.log(Logger.WARNING, emsg + " - " + e.getMessage(), e);
        }

        throw e;
        }
    }
}
