/*
 * @(#)PacketVariableHeader.java	1.8 03/04/11
 *
 * Copyright 2001 Sun Microsystems, Inc. All Rights Reserved
 *
 */

package com.sun.messaging.jmq.io;

import java.io.*;
import java.nio.ByteBuffer;
import java.nio.charset.*;

/**
 *
 * @version 1.8 03/04/11
 *
 */
public class PacketVariableHeader {

    // Needed to convert between UTF-8 and String
    protected static CharsetDecoder decoder = null;
    protected static CharsetEncoder encoder = null;
    protected static Charset        charset = null;

    // Buffer to hold variable portion of header
    protected ByteBuffer buffer = null;

    protected boolean bufferDirty = false;
    protected boolean bufferParsed = false;

    // The variable portion of the packet contains primarily strings
    // Currently transactionID and producerID are the only exception.
    protected long      transactionID      = 0;
    protected long      producerID         = 0;
    protected String[]  stringItems = new String[PacketString.LAST];

    public PacketVariableHeader() {
	this.reset();

        synchronized (this.getClass()) {
            if (charset == null) {
                try {
                    charset = Charset.forName("UTF-8");
                    decoder = charset.newDecoder();
                    encoder = charset.newEncoder();
                } catch (Exception e) {
                    // Should never happen
                    System.err.println("Could not get charset: " + e);
                }
            }
        }
    }

    /**
     * Set the variable header portion as bytes
     * WARNING! The buffer is NOT copied or duplicated!
     */
    public synchronized void setBytes(ByteBuffer buf) {
        // Clear all data members and set buffer
        reset();

        if (buf == null) {
            buffer = null;
        } else {
            buffer = buf;
            buffer.rewind();
        }

        bufferParsed = false;
        bufferDirty = false;
    }

    /**
     * Return the variable header portion as bytes
     * WARNING! The buffer is NOT a copy or duplicate.
     */
    public synchronized ByteBuffer getBytes()
        throws IOException {

        if (bufferDirty) {
            updateBuffer();
        }

        if (buffer == null) {
            return null;
        }

        buffer.rewind();
        return buffer;
    }

    /**
     * Return the variable header portion as bytes in a way
     * that is compatible with 2.0 clients.
     *
     * This routine will not return null. If there is no data in
     * the variable header a buffer will be allocated that just
     * contains the terminating NULL entry. This is for backwards
     * compatibility with 2.0 clients that had a bug and always
     * expected something in this part of the packet.
     * 
     * WARNING! The buffer is NOT a copy or duplicate.
     */
    public synchronized ByteBuffer getBytes2()
        throws IOException {

        if (bufferDirty || buffer == null) {
            updateBuffer();
        }

        return getBytes();
    }

    /**
     * Get the string value for 'field' from the buffer
     */
    public synchronized String getStringField(int field) {

        if (!bufferParsed) {
            parseBuffer();
        }

        if (field < PacketString.LAST) {
            return stringItems[field];
        } else {
            return null;
        }
    }


    /**
     * Get the long value for 'field' from the variable header portion of
     * the packet.
     */
    public synchronized long getLongField(int field) {

        if (!bufferParsed) {
            parseBuffer();
        }

        switch (field) {

        case PacketString.TRANSACTIONID:
            return transactionID;
        case PacketString.PRODUCERID:
            return producerID;
        default:
            return 0;
        }
    }

    public synchronized void setStringField(int field, String value) {

        // We must do this so we don't loose other field values if
        // updateBuffer is called.
        if (!bufferParsed) {
            parseBuffer();
        }

        if (field < PacketString.LAST) {
            stringItems[field] = value;
            bufferDirty = true;
        }
    }

    public synchronized void setLongField(int field, long value) {

        // We must do this so we don't loose other field values if
        // updateBuffer is called.
        if (!bufferParsed) {
            parseBuffer();
        }

        switch (field) {

        case PacketString.TRANSACTIONID:
            transactionID = value;
            bufferDirty = true;
            break;
        case PacketString.PRODUCERID:
            producerID = value;
            bufferDirty = true;
            break;
        default:
            break;
        }
    }

    /**
     * Reset packet to initial values
     */
    protected void reset() {
        for (int n = 0; n < PacketString.LAST; n++) {
            stringItems[n] = null;
        }
        transactionID = 0;
        producerID    = 0;

        //buffer = null;
        if (buffer != null) {
            buffer.clear();
        }
        bufferDirty = false;
        bufferParsed = true;
    }

    /**
     * Parse buffer and populate class with values
     */
    void parseBuffer() {

        int type, len = 0;

        if (buffer == null) {
            bufferParsed = true;
            return;
        }

        buffer.rewind();

        type = buffer.getShort();
	while (type != PacketString.NULL) {
            switch(type) {

            case PacketString.TRANSACTIONID:
                // Skip length. TransactinID is a long
                len = buffer.getShort();
                transactionID = buffer.getLong();
                break;

            case PacketString.PRODUCERID:
                // Skip length. ProducerID is a long
                len = buffer.getShort();
                producerID = buffer.getLong();
                break;

            case PacketString.DESTINATION:
            case PacketString.MESSAGEID:
            case PacketString.CORRELATIONID:
            case PacketString.REPLYTO:
            case PacketString.TYPE:
            case PacketString.DESTINATION_CLASS:
            case PacketString.REPLYTO_CLASS:
                len = buffer.getShort();

                int currentLimit = buffer.limit();
                int currentPosition = buffer.position();

                // Set limit so we can decode
                buffer.limit(currentPosition + len);
                try {
                    synchronized(decoder) {
                        stringItems[type] = decoder.decode(buffer).toString();
                    }
                } catch (CharacterCodingException e) {
                    // Should never get
                    System.out.println("Could not decode string " + e);
                }

                //reset limit
                buffer.limit(currentLimit);
                break;

             default:
                // Skip unknown field
                len = buffer.getShort();
                buffer.position(buffer.position() + len);
                break;
            }
            type = buffer.getShort();
        }

        bufferParsed = true;
        return;
    }

    /**
     * Update buffer to contain data held in class fields
     */
    private void updateBuffer()
        throws IOException {

	byte[] pad = new byte[4];	// Four nulls

        // ByteArrayOutputStream will grow buf if necessary.
        byte[] buf = new byte[512];
	JMQByteArrayOutputStream bos =
			new JMQByteArrayOutputStream(buf);

	DataOutputStream dos = new DataOutputStream(bos);

        // Make sure transactionID is first in buffer
	if (transactionID != 0) {
            writeLong(dos, PacketString.TRANSACTIONID, transactionID);
	}

	if (producerID != 0) {
            writeLong(dos, PacketString.PRODUCERID, producerID);
	}

        // Write string values to buffer. DESTINATION should be first
        for (int n = 0; n < PacketString.LAST; n++) {
            if (stringItems[n] != null) {
	        writeString(dos, n, stringItems[n]);
            }
        }

	//Teminate list
	dos.writeShort(PacketString.NULL);
	dos.flush();

	// Pad to nearest 32 bit boundary
        int padding = 4 - (bos.getCount() % 4);
        bos.write(pad, 0, padding);
	bos.flush();

        // Wrap a ByteBuffer around the streams backing buffer.
        buffer = ByteBuffer.wrap(bos.getBuf(), 0, bos.getCount());
        bufferDirty = false;

        // Since buffer matches fields we can set this to true
        bufferParsed = true;

	bos.close();
        dos.close();

	return;
    }

    /**
     * Write a header string item to the specified output stream
     */
    private void writeString(DataOutputStream dos, int type, String value)
	throws IOException {
	if (value != null) {
	    dos.writeShort(type);
	    dos.writeUTF(value);
	}
    }

    /**
     * Write a long field to the variable portion of the packet
     */
    private void writeLong(DataOutputStream dos, int type, long value)
	throws IOException {
	dos.writeShort(type);
        dos.writeShort(8);
	dos.writeLong(value);
    }

}
