/*
 * @(#)BasicConnectionFactory.java	1.5 04/07/14
 *
 * Copyright 2004 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms.
 *
 */

package com.sun.messaging;

import com.sun.messaging.jmq.jmsclient.UnifiedConnectionImpl;
import com.sun.messaging.jmq.ClientConstants;
import java.util.Properties;
import javax.jms.*;

/**
 * A <code>BasicConnectionFactory</code> encapsulates Sun MQ specific configuration information
 * for Sun MQ <code>ConnectionFactory</code> objects and is used to create Connections with
 * a Sun MQ Java Message Service (JMS) provider.
 *
 * @version     1.5 04/07/14
 * @author      George Tharakan
 *
 * @see         javax.jms.ConnectionFactory javax.jms.ConnectionFactory
 * @see         com.sun.messaging.ConnectionConfiguration com.sun.messaging.ConnectionConfiguration
 */
public class BasicConnectionFactory extends com.sun.messaging.AdministeredObject implements javax.jms.ConnectionFactory {

    /** The default basename for AdministeredObject initialization */
    private static final String defaultsBase = "ConnectionFactory";

    /** The default Connection Handler classname for clients */
    private static final String DEFAULT_IMQ_CONNECTION_HANDLER =
                    "com.sun.messaging.jmq.jmsclient.protocol.tcp.TCPStreamHandler";

    /** The default ConnectionHandler Label */
    private static final String DEFAULT_IMQ_CONNECTION_HANDLER_LABEL = "Connection Handler Classname";

    /** The default Username and Password for Sun MQ client authentication */
    private static final String DEFAULT_IMQ_USERNAME_PASSWORD = "guest";

    /** The default Username Label */
    private static final String DEFAULT_IMQ_USERNAME_LABEL = "Default Username";

    /** The default Password Label */
    private static final String DEFAULT_IMQ_PASSWORD_LABEL = "Default Password";

    /** The default Acknowledgement Timeout for responses from the MQ Message Service */
    private static final String DEFAULT_IMQ_ACKTIMEOUT = "0";

    /** The default AckTimeout Label*/
    private static final String DEFAULT_IMQ_ACKTIMEOUT_LABEL = "Acknowledgement Timeout";

    /** The default valuse for JMSXProperties */
    private static final boolean DEFAULT_JMSXPROP_VALUE = false;

    /** Indicate that admin connections are created */
    private transient boolean adminConnectionCreated = false;

    /** Indicate that admin connections created use a special key */
    private transient boolean adminKeyUsed = false;

    /** The type of connections created */
    private transient String connectionType = ClientConstants.CONNECTIONTYPE_NORMAL;
 
    /**
     * Constructs a BasicConnectionFactory with the default configuration.
     * 
     */
    public BasicConnectionFactory() {
        super(defaultsBase);
    }
 
    /**
     * Constructs a BasicConnectionFactory with the specified configuration.
     * 
     */
    protected BasicConnectionFactory(String defaultsBase) {
        super(defaultsBase);
    }
 
    /**
     * Creates a Connection with the default user identity. The default user identity
     * is defined by the <code>ConnectionFactory</code> properties
     * <code><b>imqDefaultUsername</b></code> and <code><b>imqDefaultPassword</b></code>
     * 
     * @return a newly created Connection.
     * 
     * @exception JMSException if a JMS error occurs.
     * @see ConnectionConfiguration#imqDefaultUsername
     * @see ConnectionConfiguration#imqDefaultPassword
     */  
    public Connection createConnection() throws JMSException {
        return createConnection(getProperty(ConnectionConfiguration.imqDefaultUsername),
                                     getProperty(ConnectionConfiguration.imqDefaultPassword));
    }

    /**
     * Creates a Connection with a specified user identity.
     * 
     * @param username the caller's user name
     * @param password the caller's password
     * 
     * @return a newly created connection.
     * 
     * @exception JMSException if a JMS error occurs.
     */  
    public Connection createConnection(String username, String password) throws JMSException {
        return new UnifiedConnectionImpl(getCurrentConfiguration(), username, password, getConnectionType());
    }

    /**
     * Sets the type of connections created by this <code>BasicConnectionFactory</code>.
     *   
     * @param type The type of connections created by this
     *             <code>BasicConnectionFactory</code>.
     */  
    public final void setConnectionType(String type) {
        if (ClientConstants.CONNECTIONTYPE_ADMINKEY.equals(type) ||
            ClientConstants.CONNECTIONTYPE_ADMIN.equals(type)) {
            connectionType = type;
        } else {
            connectionType = ClientConstants.CONNECTIONTYPE_NORMAL;
        }
    }
 
    /**
     * Returns the type of connections created by this <code>BasicConnectionFactory</code>.
     *   
     * @return The type of connections created by this
     *             <code>BasicConnectionFactory</code>.
     */  
    public final String getConnectionType() {
        //Protect against NPE if a BasicConnectionFactory Object is restored
        //using a non-standard way
        if (connectionType == null) {
            connectionType = ClientConstants.CONNECTIONTYPE_NORMAL;
        }
        return connectionType;
    }
 
    /**
     * Returns a pretty printed version of the provider specific
     * information for this ConnectionFactory object.
     *
     * @return the pretty printed string.
     */
    public String toString() {
        return ("Sun Java System MQ ConnectionFactory" + super.toString());
    }

    /**
     * Sets the minimum <code>BasicConnectionFactory</code> configuration defaults
     * required to connect to the Sun MQ Message Service.
     */  
    public void setDefaultConfiguration() {
        configuration = new Properties();
        configurationTypes = new Properties();
        configurationLabels = new Properties();

        configuration.put(ConnectionConfiguration.imqConnectionHandler,
                                DEFAULT_IMQ_CONNECTION_HANDLER);
        configurationTypes.put(ConnectionConfiguration.imqConnectionHandler,
                                AO_PROPERTY_TYPE_PROPERTYOWNER);
        configurationLabels.put(ConnectionConfiguration.imqConnectionHandler,
                                DEFAULT_IMQ_CONNECTION_HANDLER_LABEL);
 
        configuration.put(ConnectionConfiguration.imqDefaultUsername,
                                DEFAULT_IMQ_USERNAME_PASSWORD);
        configurationTypes.put(ConnectionConfiguration.imqDefaultUsername,
                                AO_PROPERTY_TYPE_STRING);
        configurationLabels.put(ConnectionConfiguration.imqDefaultUsername,
                                DEFAULT_IMQ_USERNAME_LABEL);
 
        configuration.put(ConnectionConfiguration.imqDefaultPassword,
                                DEFAULT_IMQ_USERNAME_PASSWORD);
        configurationTypes.put(ConnectionConfiguration.imqDefaultPassword,
                                AO_PROPERTY_TYPE_STRING);
        configurationLabels.put(ConnectionConfiguration.imqDefaultPassword,
                                DEFAULT_IMQ_PASSWORD_LABEL);
 
        configuration.put(ConnectionConfiguration.imqAckTimeout,
                                DEFAULT_IMQ_ACKTIMEOUT);
        configurationTypes.put(ConnectionConfiguration.imqAckTimeout,
                                AO_PROPERTY_TYPE_INTEGER);
        configurationLabels.put(ConnectionConfiguration.imqAckTimeout,
                                DEFAULT_IMQ_ACKTIMEOUT_LABEL);
    }

}
