package org.codehaus.groovy.tools

import org.codehaus.groovy.sandbox.markup.StreamingMarkupBuilder

import java.io.File

import com.thoughtworks.qdox.JavaDocBuilder
import com.thoughtworks.qdox.model.JavaSource
import com.thoughtworks.qdox.model.JavaClass
import com.thoughtworks.qdox.model.JavaMethod
import com.thoughtworks.qdox.model.JavaParameter
import com.thoughtworks.qdox.model.Type
import java.util.*;


/**
 * Generate documentation about the methods provided by the Groovy Development Kit
 * enhancing the standard JDK classes.
 *
 * @author Guillaume Laforge, John Wilson
 */
class DocGenerator
{
	File           file
	File           outputFile
	JavaDocBuilder builder

	DocGenerator(File fileToParse, File outputFile)
	{
		this.file       = fileToParse
		this.outputFile = outputFile
	}

	/**
	 * Parse the DefaultGroovyMethods class to build a graph representing the structure of the class,
	 * with its methods, javadoc comments and tags.
	 */
	private void parse()
	{
		reader  = file.newReader()
		builder = new JavaDocBuilder()

		builder.addSource(reader)
	}

	/**
	 * Builds an HTML page from the structure of DefaultGroovyMethods.
	 */
		
		generate() {
			parse()
			
			sources = builder.getSources()

			firstSource = sources[0]
			classes = firstSource.getClasses()
			groovyMeth = classes[0]
	
			// categorize all groovy methods per core JDK class to which it applies
			jdkEnhancedClasses = [:]
			methods = groovyMeth.getMethods()
			
            start = System.currentTimeMillis();
			for (method in methods) {
				if (method.isPublic() && method.isStatic()) {
					parameters = method.getParameters()
					jdkClass = parameters[0].getType().toString()

					if (jdkEnhancedClasses.containsKey(jdkClass)) {
					    List l = jdkEnhancedClasses[jdkClass];
						l.add(method)
				    }
					else
						jdkEnhancedClasses[jdkClass] = [method]
				}
			}
            //start = System.currentTimeMillis();
	        println " added classes in ${System.currentTimeMillis() - start} ms"

			headElement = { head {
								mkp.comment 'generated by Groovy using Streaming Markup'
								title 'GDK : Groovy methods'
								style(['type':'text/css'], '   @import url("./style/maven-base.css"); ')
								style(['type':'text/css'], '   @import url("http://codehaus.org/codehaus-style.css"); ')
							}
						  }
							
			sortedClasses = ([] + jdkEnhancedClasses.keySet()).sort()
							
			summary = { mkp.yield {
					            counter = 0
					
								// lets iterate in sorted class name order
								for (String className in sortedClasses) {
					
									b className
					
									listOfMethods = jdkEnhancedClasses[className]
									listOfMethods.sort{ it.name }
									
									table(width:'100%') {
										for (JavaMethod meth in listOfMethods) {
						                    counter++
						                    
						                    tr {
						                    	td(width:'30%') {
						                    		mkp.yield getReturnType(meth)
						                    	}
						                    	td(width:'70%') {
						                    		a(href:"#meth${counter}") { 
						                    			mkp.yield meth.getName()
						                    		}
						                    		mkp.yield "(${getParametersDecl(meth)})"
						                    	}
						                    }
										}
									}
								}
							}
						}
								
			details = { mkp.yield {
								counter = 0
								
								for (className in sortedClasses) {
					
									h2 className
					
									listOfMethods = jdkEnhancedClasses[className]
									listOfMethods.sort{ it.name }
									
									for (JavaMethod meth in listOfMethods) {
					                    counter++

					                    a(name:"meth${counter}")
					                    
					                    p {
					                    	b "${getReturnType(meth)} ${meth.getName()}(${getParametersDecl(meth)})"
					                    }
					                    
				                    	ul {
				                    		//
				                    		// Java comments can contain markup - pass it through as is
				                    		//
				                   			mkp.yieldUnescaped "${getComment(meth)}  "
				                   			
					                    	ul {
												params = meth.getTags()
												count = 0
												
												for (param in params) {
													if (count++ != 0 && param.getName() != "throws" && param.getName() != "return") {
														li "${param.getName()} ${param.getValue()}"
													}
												 }

												returnType = getReturnType(meth)
	
												if (returnType != "") {
												    if (returnType != "void") {
													    li {
													    	b "returns"
															mkp.yield ": ${returnType}"
															    	
													        returnTag = meth.getTagByName("return")
													                
													        if (returnTag != null) {
													        	mkp.yield " - "
													        	i returnTag.getValue()
															}
														}
													}
													
													exceptions = meth.getExceptions()
													
													for (ex in exceptions) {
														if (ex != null) {
															li {
																b "throws"
																mkp.yield ": ${ex}"
	
																exMsg = meth.getTagByName("throws")
																
																if (exMsg != null) {
																	mkp.yield " - "
																	i exMsg.getValue()
																}
															}
														}
													}
												}
											}
					                    }
									}
								}
					  		}
					  	}
					  				
			bodyElement = { body {
								h1 'Groovy JDK methods'
								p 'New methods added to the JDK to make it more groovy.'
								mkp.yield summary
								mkp.yield details
							 }
						   }
					
			 outputFile.getParentFile().mkdirs()
			 outputFile.newPrintWriter() << new StreamingMarkupBuilder().bind{ html {
																						mkp.yield headElement
																						mkp.yield bodyElement
																					 }
																			  }
		}

		/**
	 	* Retrieves a String representing the return type
	 	*/
		private getReturnType(method)
		{
		    returnType = method.getReturns()
		    
		    if (returnType != null) {
		    	return returnType.toString()
		    } else {
		    	return ""
		    }
		}

		/**
		 * Retrieve a String representing the declaration of the parameters of the method passed as parameter.
		 *
		 * @param method a method
		 * @return the declaration of the method (long version)
		 */
		private getParametersDecl(method)
		{
			getParameters(method).collect{ "${it.getType()} ${it.getName()}" }.join(", ")
		}

		/**
		 * Retrieve the parameters of the method.
		 *
		 * @param method a method
		 * @return a list of parameters without the first one
		 */
		private getParameters(method)
		{
		    if (method.getParameters().size() > 1)
			    return method.getParameters().toList()[1..-1]
			else
			    return []
		}

		/**
		 * Retrieve the JavaDoc comment associated with the method passed as parameter.
		 *
		 * @param method a method
		 * @return the JavaDoc comment associated with this method
		 */
		private getComment(method)
		{
			ans = method.getComment()
			if (ans == null) return ""
			return ans
		}

    /**
     * Main entry point.
     */
    static void main(args)
    {
        defaultGroovyMethodSource =
            //new File("D:/cvs-groovy/groovy/groovy-core/src/main/org/codehaus/groovy/runtime/DefaultGroovyMethods.java")
            new File("src/main/org/codehaus/groovy/runtime/DefaultGroovyMethods.java")
        outFileName =
            //new File("D:/cvs-groovy/groovy/groovy-core/target/html/groovy-jdk.html")
            new File("target/html/groovy-jdk.html")

        start = System.currentTimeMillis();

        docGen = new DocGenerator(defaultGroovyMethodSource, outFileName)
        docGen.generate()

        end = System.currentTimeMillis();

        println "Done. in ${end - start} millis"

    }
}