/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2004 
 *					All rights reserved
 *
 *  This file is part of GPAC / Scene Rendering sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */



#include "stacks3d.h"


#ifdef M4_DEF_Text

#ifndef M4_DEF_FontStyle
#error "FontStyle MUST be defined when enabling Text"
#endif

/*default value when no fontStyle*/
#define FSFAMILY	fs ? (const char *)fs->family.vals[0]	: ""

/*here it's tricky since it depends on our metric system...*/
#define FSSIZE		(fs ? fs->size : -1)
#define FSSTYLE		(fs ? (const char *)fs->style.buffer : "")
#define FSMAJOR		( (fs && fs->justify.count && fs->justify.vals[0]) ? (const char *)fs->justify.vals[0] : "FIRST")
#define FSMINOR		( (fs && (fs->justify.count>1) && fs->justify.vals[1]) ? (const char *)fs->justify.vals[1] : "FIRST")
#define FSHORIZ		(fs ? fs->horizontal : 1)
#define FSLTR		(fs ? fs->leftToRight : 1)
#define FSTTB		(fs ? fs->topToBottom : 1)
#define FSLANG		(fs ? fs->language : "")
#define FSSPACE		(fs ? fs->spacing : 1)

typedef struct
{
	M4Path *path;
	M4Path *outline;
	M4Mesh *mesh;
	M4Mesh *outline_mesh;
	M4Rect bounds;
} CachedTextLine;

static CachedTextLine *new_text_line(Chain *line_list)
{
	CachedTextLine *tl = malloc(sizeof(CachedTextLine));
	memset(tl, 0, sizeof(CachedTextLine));
	tl->path = m4_new_path();
	/*until we get decent glyph/bitmap rendering for text, don't use too fine text...*/
	tl->path->fineness = 0.5;
	ChainAddEntry(line_list, tl);
	return tl;
}

static void clean_paths(TextStack *stack)
{
	CachedTextLine *tl;
	/*delete all path objects*/
	while (ChainGetCount(stack->text_lines)) {
		tl = ChainGetEntry(stack->text_lines, 0);
		ChainDeleteEntry(stack->text_lines, 0);
		m4_path_delete(tl->path);
		if (tl->outline) m4_path_delete(tl->outline);
		if (tl->mesh) { mesh_free(tl->mesh); tl->mesh = NULL; }
		if (tl->outline_mesh) { mesh_free(tl->outline_mesh); tl->outline_mesh = NULL; }
		free(tl);
	}
	stack->bounds.width = stack->bounds.height = 0;
}

static void DestroyText(SFNode *node)
{
	TextStack *stack = (TextStack *) Node_GetPrivate(node);
	clean_paths(stack);
	stack2D_predestroy((stack2D *)stack);
	DeleteChain(stack->text_lines);
	free(stack);
}

static void split_text_letters(TextStack *st, B_Text *txt, RenderEffect *eff)
{
	CachedTextLine *tl;
	unsigned short wcTemp[5000];
	unsigned short letter[2];
	u32 i, j, len;
	Float fontSize, start_y, font_height, line_spacing;
	FontRaster *ft_r = st->compositor->font_engine;
	B_FontStyle *fs = (B_FontStyle *)txt->fontStyle;

	fontSize = FSSIZE;
	if (fontSize <= 0) {
		fontSize = 12;
		if (!st->compositor->is_pixel_metrics) fontSize /= st->compositor->scene_width;
    }
	line_spacing = FSSPACE * fontSize;

	if (ft_r->set_font(ft_r, FSFAMILY, FSSTYLE) != M4OK) {
		if (ft_r->set_font(ft_r, NULL, FSSTYLE) != M4OK) {
			return;
		}
	}
	ft_r->set_font_size(ft_r, fontSize);
	ft_r->get_font_metrics(ft_r, &st->ascent, &st->descent, &font_height);
	/*adjust size*/
	fontSize *= fontSize / (font_height);
	ft_r->set_font_size(ft_r, fontSize);
	ft_r->get_font_metrics(ft_r, &st->ascent, &st->descent, &font_height);

			
	if (!strcmp(FSMINOR, "MIDDLE")) {
		start_y = (st->descent + st->ascent)/2;
	}
	else if (!strcmp(FSMINOR, "BEGIN")) {
		start_y = st->descent;
	}
	else if (!strcmp(FSMINOR, "END")) {
		start_y = st->descent + st->ascent;
	}
	else {
		start_y = st->ascent;
	}
	
	st->bounds.width = st->bounds.height = 0;

	for (i=0; i < txt->string.count; i++) {

		char *str = txt->string.vals[i];
		if (!str) continue;
		len = utf8_mbstowcs(wcTemp, 5000, (const char **) &str);
		if (len == (size_t) (-1)) continue;

		letter[1] = (unsigned short) 0;
		for (j=0; j<len; j++) {
			if (FSLTR) {
				letter[0] = wcTemp[j];
			} else {
				letter[0] = wcTemp[len - j - 1];
			}
			tl = new_text_line(st->text_lines);
			ft_r->add_text_to_path(ft_r, tl->path, letter, 0, start_y, 1, 1, st->ascent, &tl->bounds);
			tl->bounds.height = MAX(st->ascent + st->descent, tl->bounds.height);
		}
	}
}

static void split_text_words(TextStack *st, B_Text *txt, RenderEffect *eff)
{
	CachedTextLine *tl;
	unsigned short wcTemp[5000];
	unsigned short letter[5000];
	u32 i, j, len, k, first_char;
	Float fontSize, font_height, line_spacing;
	M4Rect rc;
	FontRaster *ft_r = st->compositor->font_engine;
	B_FontStyle *fs = (B_FontStyle *)txt->fontStyle;

	fontSize = FSSIZE;
	if (fontSize <= 0) {
		fontSize = 12;
		if (!st->compositor->is_pixel_metrics) fontSize /= st->compositor->scene_width;
    }
	line_spacing = FSSPACE * fontSize;

	if (ft_r->set_font(ft_r, FSFAMILY, FSSTYLE) != M4OK) {
		if (ft_r->set_font(ft_r, NULL, FSSTYLE) != M4OK) {
			return;
		}
	}
	ft_r->set_font_size(ft_r, fontSize);
	ft_r->get_font_metrics(ft_r, &st->ascent, &st->descent, &font_height);
	/*adjust size*/
	fontSize *= fontSize / (font_height);
	ft_r->set_font_size(ft_r, fontSize);
	ft_r->get_font_metrics(ft_r, &st->ascent, &st->descent, &font_height);

	st->bounds.width = st->bounds.height = 0;

	for (i=0; i < txt->string.count; i++) {
		char *str = txt->string.vals[i];
		if (!str) continue;
		len = utf8_mbstowcs(wcTemp, 5000, (const char **) &str);
		if (len == (size_t) (-1)) continue;

		first_char = 0;
		for (j=0; j<len; j++) {
			/*we currently only split sentences at spaces*/
			if ((j+1!=len) && (wcTemp[j] != (unsigned short) ' ')) continue;

			if (FSLTR) {
				for (k=0; k<=j - first_char; k++) letter[k] = wcTemp[first_char+k];
			} else {
				for (k=0; k<=j - first_char; k++) letter[k] = wcTemp[len - first_char - k - 1];
			}
			letter[k] = (unsigned short) 0;
			tl = new_text_line(st->text_lines);

			/*word splitting only happen in layout, so we don't need top/left anchors*/
			ft_r->add_text_to_path(ft_r, tl->path, letter, 0, 0, 1, 1, st->ascent, &rc);
			m4_path_get_bounds(tl->path, &tl->bounds);
			tl->bounds.width = rc.width;
			if (tl->bounds.x != 0) tl->bounds.width -= tl->bounds.x;
			tl->bounds.x = 0;
			tl->bounds.height = MAX(st->ascent + st->descent, tl->bounds.height);
			if (tl->bounds.y != 0) tl->bounds.height -= tl->bounds.y;
			tl->bounds.y = 0;
			first_char = j+1;
		}
	}
}

typedef struct
{
	unsigned short *wcText;
	u32 length;
	Float width, height;
	Float x_scaling, y_scaling;
} TextLine;

static void BuildVerticalTextGraph(TextStack *st, B_Text *txt, RenderEffect *eff)
{
	TextLine *lines;
	CachedTextLine *tl;
	unsigned short wcTemp[5000];
	u32 i, int_major, len, k;
	Float fontSize, start_x, start_y, space, line_spacing, tot_width, tot_height, max_scale;
	M4Rect rc, final;
	Float lw, lh, max_lw;
	unsigned short letter[2];
	FontRaster *ft_r = st->compositor->font_engine;
	B_FontStyle *fs = (B_FontStyle *)txt->fontStyle;

	fontSize = FSSIZE;
	if (fontSize <= 0) {
		fontSize = 12;
		if (!st->compositor->is_pixel_metrics) fontSize /= st->compositor->scene_width;
    }

	line_spacing = FSSPACE * fontSize;

	if (ft_r->set_font(ft_r, FSFAMILY, FSSTYLE) != M4OK) {
		if (ft_r->set_font(ft_r, NULL, FSSTYLE) != M4OK) {
			return;
		}
	}
	ft_r->set_font_size(ft_r, fontSize);
	ft_r->get_font_metrics(ft_r, &st->ascent, &st->descent, &space);


	/*compute overall bounding box size*/
	tot_width = 0;
	tot_height = 0;
	lines = (TextLine *) malloc(sizeof(TextLine)*txt->string.count);
	memset(lines, 0, sizeof(TextLine)*txt->string.count);
		
	letter[1] = (unsigned short) '\0';

/*	space = st->ascent + st->descent;
	space = fontSize - st->ascent + st->descent;
*/
	for (i=0; i < txt->string.count; i++) {
		char *str = txt->string.vals[i];
		if (!str) continue;
		lines[i].length = 0;
		len = utf8_mbstowcs(wcTemp, 5000, (const char **) &str);
		if (len == (size_t) (-1)) continue;

		lines[i].wcText = malloc(sizeof(unsigned short) * len);
		memcpy(lines[i].wcText, wcTemp, sizeof(unsigned short) * len);
		lines[i].length = len;
		
		lines[i].y_scaling = lines[i].x_scaling = 1.0;
		lines[i].height = len * space;
		if (!lines[i].height) continue;

		if ((txt->length.count>i) && (txt->length.vals[i]>0) ) 
			lines[i].y_scaling = txt->length.vals[i] / lines[i].height;
	
		if (tot_height < lines[i].height * lines[i].y_scaling) tot_height = lines[i].height * lines[i].y_scaling;
	}

	tot_width = txt->string.count * line_spacing;
	st->bounds.width = tot_width;

	max_scale = 1.0;
	if ((txt->maxExtent>0) && (tot_height>txt->maxExtent)) {
		max_scale = txt->maxExtent / tot_height;
		tot_height = txt->maxExtent;
	}

	if (!strcmp(FSMINOR, "MIDDLE")) {
		if (FSLTR) {
			start_x = -tot_width/2;
			st->bounds.x = start_x;
		} else {
			start_x = tot_width/2 - line_spacing;
			st->bounds.x = - tot_width + line_spacing;
		}
	}
	else if (!strcmp(FSMINOR, "END")) {
		if (FSLTR) {
			start_x = -tot_width;
			st->bounds.x = start_x;
		} else {
			start_x = tot_width-line_spacing;
			st->bounds.x = 0;
		}
	}
	else {
		if (FSLTR) {
			start_x = 0;
			st->bounds.x = start_x;
		} else {
			start_x = -line_spacing;
			st->bounds.x = -tot_width;
		}
	}

	if (!strcmp(FSMAJOR, "MIDDLE")) {
		int_major = 0;
	} else if (!strcmp(FSMAJOR, "END")) {
		int_major = 1;
	} else {
		int_major = 2;
	}

	final.width = final.height = 0;
	for (i=0; i < txt->string.count; i++) {
		switch (int_major) {
		case 0:
			if (FSTTB) 
				start_y = lines[i].height/2;
			else
				start_y = -lines[i].height/2 + space;
			break;
		case 1:
			if (FSTTB)
				start_y = lines[i].height;
			else
				start_y = -lines[i].height + space;
			break;
		default:
			if (FSTTB)
				start_y = 0;
			else
				start_y = space;
			break;
		}

		if (lines[i].length) {
			tl = new_text_line(st->text_lines);

			/*adjust horizontal offset on first column*/
			if (!i) {
				max_lw = 0;
				for (k=0; k<lines[i].length; k++) {
					letter[0] = lines[i].wcText[k];
					/*get glyph width so that all letters are centered on the same vertical line*/
					ft_r->get_text_size(ft_r, letter, &lw, &lh);
					if (max_lw < lw) max_lw = lw;
				}
				st->bounds.width += max_lw/2;
				start_x += max_lw/2;
			}
			
			for (k=0; k<lines[i].length; k++) {
				letter[0] = lines[i].wcText[k];
				/*get glyph width so that all letters are centered on the same vertical line*/
				ft_r->get_text_size(ft_r, letter, &lw, &lh);
				ft_r->add_text_to_path(ft_r, tl->path, letter, start_x - lw/2, start_y, lines[i].x_scaling, lines[i].y_scaling*max_scale, st->ascent, &rc);

				if (FSTTB)
					start_y -= space;
				else
					start_y += space;
			}
			m4_path_get_bounds(tl->path, &rc);
			m4_rect_union(&final, &rc);
		}

		if (FSLTR) {
			start_x += line_spacing;
		} else {
			start_x -= line_spacing;
		}

		/*free unicode buffer*/
		free(lines[i].wcText);
	}

	/*free remaining unicode buffers*/
	for (; i < txt->string.count; i++) free(lines[i].wcText);

	free(lines);

	st->bounds.height = final.height;
	st->bounds.y = final.y;
}


static void BuildTextGraph(TextStack *st, B_Text *txt, RenderEffect *eff)
{
	TextLine *lines;
	CachedTextLine *tl;
	unsigned short wcTemp[5000];
	u32 i, int_major, len, k;
	Float fontSize, start_x, start_y, font_height, line_spacing, tot_width, tot_height, max_scale;
	M4Rect rc, final;
	FontRaster *ft_r = st->compositor->font_engine;
	B_FontStyle *fs = (B_FontStyle *)txt->fontStyle;

	if (!FSHORIZ) {
		BuildVerticalTextGraph(st, txt, eff);
		return;
	}

	fontSize = FSSIZE;
	if (fontSize <= 0) {
		fontSize = 12;
		if (!st->compositor->is_pixel_metrics) fontSize /= st->compositor->scene_width;
    }

	if (ft_r->set_font(ft_r, FSFAMILY, FSSTYLE) != M4OK) {
		if (ft_r->set_font(ft_r, NULL, FSSTYLE) != M4OK) {
			return;
		}
	}
	ft_r->set_font_size(ft_r, fontSize);
	ft_r->get_font_metrics(ft_r, &st->ascent, &st->descent, &font_height);

	/*spacing= FSSPACING * fontSize and fontSize adjusted to m-ascent+m-descent (removed COR)*/
#if 0
	line_spacing = FSSPACE * fontSize;
	/*adjust size*/
	fontSize *= fontSize / (font_height);
	ft_r->set_font_size(ft_r, fontSize);
	ft_r->get_font_metrics(ft_r, &st->ascent, &st->descent, &font_height);
#else
	/*spacing= FSSPACING * (font_height) and fontSize not adjusted */
	line_spacing = FSSPACE * fontSize;
#endif
	
	tot_width = 0;
	lines = (TextLine *) malloc(sizeof(TextLine)*txt->string.count);
	memset(lines, 0, sizeof(TextLine)*txt->string.count);
	
	for (i=0; i < txt->string.count; i++) {
		char *str = txt->string.vals[i];
		if (!str) continue;
		lines[i].length = 0;
		len = utf8_mbstowcs(wcTemp, 5000, (const char **) &str);
		if (len == (size_t) (-1)) continue;

		lines[i].length = len;
		lines[i].wcText = malloc(sizeof(unsigned short) * (len+1));
		if (!FSLTR) {
			for (k=0; k<len; k++) lines[i].wcText[k] = wcTemp[len-k-1];
		} else {
			memcpy(lines[i].wcText, wcTemp, sizeof(unsigned short) * len);
		}
		lines[i].wcText[len] = (unsigned short) '\0';

		lines[i].y_scaling = lines[i].x_scaling = 1.0;
		ft_r->get_text_size(ft_r, lines[i].wcText, &lines[i].width, &lines[i].height);

		if (!lines[i].width) continue;
		if ((txt->length.count>i) && (txt->length.vals[i]>0)) {
			lines[i].x_scaling = txt->length.vals[i] / lines[i].width;
		}
		if (tot_width < lines[i].width * lines[i].x_scaling) tot_width = lines[i].width * lines[i].x_scaling;
	}
	
	max_scale = 1.0;
	if ((txt->maxExtent > 0) && (tot_width>txt->maxExtent)) {
		max_scale = txt->maxExtent / tot_width;
		tot_width = txt->maxExtent;
	}

	tot_height = (txt->string.count-1) * line_spacing + (st->ascent + st->descent);
	st->bounds.height = tot_height;
	
	if (!strcmp(FSMINOR, "MIDDLE")) {
		if (FSTTB) {
			start_y = tot_height/2;
			st->bounds.y = start_y;
		} else {
			start_y = st->descent + st->ascent - tot_height/2;
			st->bounds.y = tot_height/2;
		}
	}
	else if (!strcmp(FSMINOR, "BEGIN")) {
		if (FSTTB) {
			start_y = st->descent;
			start_y = 0;
			st->bounds.y = start_y;
		} else {
			st->bounds.y = st->bounds.height;
			start_y = st->descent + st->ascent;
		}
	}
	else if (!strcmp(FSMINOR, "END")) {
		if (FSTTB) {
			start_y = tot_height;
			st->bounds.y = start_y;
		} else {
			start_y = -tot_height + 2*st->descent + st->ascent;
			st->bounds.y = start_y - (st->descent + st->ascent) + tot_height;
		}
	}
	else {
		start_y = st->ascent;
		st->bounds.y = FSTTB ? start_y : (tot_height - st->descent);
	}
	
	/*major-justification*/
	if (!strcmp(FSMAJOR, "MIDDLE") ) {
		int_major = 0;
	} else if (!strcmp(FSMAJOR, "END") ) {
		int_major = 1;
	} else {
		int_major = 2;
	}
	final.width = final.height = 0;

	for (i=0; i < txt->string.count; i++) {
		switch (int_major) {
		/*major-justification MIDDLE*/
		case 0:
			start_x = -lines[i].width/2;
			break;
		/*major-justification END*/
		case 1:
			start_x = (FSLTR) ? -lines[i].width : 0;
			break;
		/*BEGIN, FIRST or default*/
		default:
			start_x = (FSLTR) ? 0 : -lines[i].width;
			break;
		}

		if (lines[i].length) {
			tl = new_text_line(st->text_lines);
			/*if using the font engine the font is already configured*/
			ft_r->add_text_to_path(ft_r, tl->path, lines[i].wcText, start_x, start_y, lines[i].x_scaling*max_scale, lines[i].y_scaling, st->ascent, &rc);
			m4_rect_union(&final, &rc);
		}
		if (FSTTB) {
			start_y -= line_spacing;
		} else {
			start_y += line_spacing;
		}

		/*free unicode buffer*/
		free(lines[i].wcText);
	}
	/*free remaining unicode buffers*/
	for (; i < txt->string.count; i++) free(lines[i].wcText);

	free(lines);

	st->bounds.width = final.width;
	st->bounds.x = final.x;
}

static void Text_SetupBounds(RenderEffect *eff, TextStack *st)
{
	u32 i;
	/*text is not splitted, handle as a whole node*/
	if (!eff->text_split_mode) {
		if (eff->parent) {
			group_end_text_child(eff->parent, &st->bounds, st->ascent, st->descent, 0);
		} else {
			bbox_from_rect(&eff->bbox, &st->bounds);
		}
		return;
	}
	assert(eff->parent);

	/*otherwise notify each node as a different group*/
	for (i=0; i<ChainGetCount(st->text_lines); i++) {
		CachedTextLine *tl = ChainGetEntry(st->text_lines, i);
		/*the first one is always started by parent group*/
		if (i) group_start_child(eff->parent, NULL);
		group_end_text_child(eff->parent, &tl->bounds, st->ascent, st->descent, i+1);
	}
}

static void Text_SetPick(RenderEffect *eff, TextStack *st)
{
  /*
	u32 i;
	for (i=0; i<ChainGetCount(st->text_lines); i++) {
		CachedTextLine *tl = ChainGetEntry(st->text_lines, i);
	}
  */
}

static void Text_FillTextLine(RenderEffect *eff, CachedTextLine *tl)
{
	if (!tl->mesh) {
		tl->mesh = new_mesh();
		mesh_from_path(tl->mesh, tl->path);
	}
	VS3D_DrawMesh(eff->surface, tl->mesh);
}

static void Text_StrikeTextLine(RenderEffect *eff, CachedTextLine *tl, Aspect2D *asp, Bool vect_outline)
{
	if (!tl->outline_mesh) {
		tl->outline_mesh = new_mesh();
		if (vect_outline) {
			if (!tl->outline) tl->outline = m4_path_get_outline(tl->path, asp->pen_props);
			TesselatePath(tl->outline_mesh, tl->outline, asp->txh ? 2 : 1);
		} else {
			mesh_get_outline(tl->outline_mesh, tl->path);
		}
	}
	if (vect_outline) {
		VS3D_DrawMesh(eff->surface, tl->outline_mesh);
	} else {
		VS3D_StrikeMesh(eff->surface, tl->outline_mesh, Aspect_GetLineWidth(asp), asp->pen_props.dash);
	}
}

static void Text_Draw(RenderEffect *eff, TextStack *st)
{
	u32 i, count;
	Aspect2D asp;
	CachedTextLine *tl;
	Bool vect_outline;
	Render3D *sr = (Render3D*)st->compositor->visual_renderer->user_priv;

	vect_outline = !sr->raster_outlines;
	/*also save polygon bit*/
	VS3D_PushState(eff->surface);

	VS3D_SetAntiAlias(eff->surface, st->compositor->antiAlias);

	/*fill*/
	VS_GetAspect2D(eff, &asp);
	if (VS_setup_texture(eff) || asp.filled) {
		if (asp.filled) VS3D_SetMaterial2D(eff->surface, asp.fill_color, asp.alpha);

		if (eff->split_text_idx) {
			tl = ChainGetEntry(st->text_lines, eff->split_text_idx-1);
			Text_FillTextLine(eff, tl);
		} else {
			count = ChainGetCount(st->text_lines);
			for (i=0; i<ChainGetCount(st->text_lines); i++) {
				tl = ChainGetEntry(st->text_lines, i);
				Text_FillTextLine(eff, tl);
			}
		}
		/*reset texturing in case of line texture*/
		if (asp.txh) tx_disable(asp.txh);
	}

	if (asp.pen_props.width) {
		Float w = asp.pen_props.width;
		if (!asp.pen_props.is_scalable) asp.pen_props.width/=asp.line_scale;
		VS_Set2DStrikeAspect(eff->surface, &asp);

		if (eff->split_text_idx) {
			tl = ChainGetEntry(st->text_lines, eff->split_text_idx-1);
			Text_StrikeTextLine(eff, tl, &asp, vect_outline);
		} else {
			count = ChainGetCount(st->text_lines);
			for (i=0; i<ChainGetCount(st->text_lines); i++) {
				tl = ChainGetEntry(st->text_lines, i);
				Text_StrikeTextLine(eff, tl, &asp, vect_outline);
			}
		}
		asp.pen_props.width = w;
	}
	VS3D_PopState(eff->surface);

}

static void Text_Render(SFNode *n, void *rs)
{
	B_Text *txt = (B_Text *) n;
	TextStack *st = (TextStack *) Node_GetPrivate(n);
	RenderEffect *eff = (RenderEffect *)rs;

	if (!st->compositor->font_engine) return;
	if (!txt->string.count) return;

	/*check for geometry change*/
	if (Node_GetDirty(n)) {
		clean_paths(st);
		stack2D_reset((stack2D *) st);
		Node_ClearDirty(n);

		if (eff->text_split_mode == 2) {
			split_text_letters(st, txt, eff);
		} else if (eff->text_split_mode == 1) {
			split_text_words(st, txt, eff);
		} else {
			BuildTextGraph(st, txt, eff);
		}
	}

	if (eff->trav_flags & TF_SWITCHED_OFF) return;

	/*drawing*/
	if (!eff->traversing_mode) {
		Text_Draw(eff, st);
	}
	/*getting bounds info*/
	else if (eff->traversing_mode==TRAVERSE_GET_BOUNDS) {
		Text_SetupBounds(eff, st);
		st->mesh->bounds = eff->bbox;
	}
	else if (eff->traversing_mode==TRAVERSE_PICK) {
		Text_SetPick(eff, st);
	}
}


void R3D_InitText(Render3D *sr, SFNode *node)
{
	TextStack *stack = malloc(sizeof(TextStack));
	stack2D_setup((stack2D *)stack, sr->compositor, node);
	/*override all funct*/
	stack->ascent = stack->descent = 0;
	stack->text_lines = NewChain();
	Node_SetPrivate(node, stack);
	Node_SetRenderFunction(node, Text_Render);
	Node_SetPreDestroyFunction(node, DestroyText);
}

#endif

