/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2004 
 *					All rights reserved
 *
 *  This file is part of GPAC / Scene Rendering sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */


#include "visual_surface.h"
#include "gl_inc.h"

#ifndef CALLBACK
#define CALLBACK
#endif

typedef struct
{
	/*for tesselation*/
	GLUtesselator *tess_obj;
	M4Mesh *mesh;

	/*vertex indices: we cannot use a static array because reallocating the array will likely change memory 
	address of indices, hence break triangulator*/
	Chain *vertex_index;
} MeshTess;

static void CALLBACK mesh_tess_begin(GLenum which) { assert(which==GL_TRIANGLES); }
static void CALLBACK mesh_tess_end(void) { }
static void CALLBACK mesh_tess_error(GLenum error_code) { fprintf(stdout, "%s\n", gluErrorString(error_code)); }
/*only needed to force GL_TRIANGLES*/
static void CALLBACK mesh_tess_edgeflag(GLenum flag) { }

static void CALLBACK mesh_tess_vertex(void *vertexData, void *user_data)
{
	MeshTess *tess = (MeshTess *) user_data;
	mesh_set_index(tess->mesh, *(u32*)vertexData);
}

static void CALLBACK mesh_tess_combine(GLdouble coords[3], void* vertex_data[4], GLfloat weight[4], void** out_data, void *user_data)
{
	u32 i, idx;
	u32 *new_idx;
	SFVec3f n;
	SFVec2f tx;
	SFColor col;

	MeshTess *tess = (MeshTess *) user_data;

	col.red = col.green = col.blue = 0.0f;
	if (tess->mesh->flags & MESH_HAS_COLOR) {
		for (i=0; i<4; i++) {
			if (weight[i]) {
				idx = * (u32 *) vertex_data[i];
				col.red += weight[i]*tess->mesh->vertices[idx].color.red;
				col.green += weight[i]*tess->mesh->vertices[idx].color.green;
				col.blue += weight[i]*tess->mesh->vertices[idx].color.blue;
			}
		}
	}

	n.x = n.y = n.z = 0.0f;
	if (tess->mesh->flags & MESH_IS_2D) {
		n.z = 1.0f;
	} else {
		for (i=0; i<4; i++) {
			if (weight[i]) {
				idx = * (u32 *) vertex_data[i];
				n.x += weight[i]*tess->mesh->vertices[idx].normal.x;
				n.y += weight[i]*tess->mesh->vertices[idx].normal.y;
				n.z += weight[i]*tess->mesh->vertices[idx].normal.z;
			}
		}
	}
	tx.x = tx.y = 0;
	if (!(tess->mesh->flags & MESH_NO_TEXTURE)) {
		for (i=0; i<4; i++) {
			if (weight[i]) {
				idx = * (u32 *) vertex_data[i];
				tx.x += weight[i]*tess->mesh->vertices[idx].texcoords.x;
				tx.y += weight[i]*tess->mesh->vertices[idx].texcoords.y;
			}
		}
	}

	new_idx = (u32 *) malloc(sizeof(u32));
	ChainAddEntry(tess->vertex_index, new_idx);
	*new_idx = tess->mesh->v_count;
	mesh_set_vertex(tess->mesh, (Float) coords[0], (Float) coords[1], (Float) coords[2], n.x, n.y, n.z, tx.x, tx.y);
	*out_data = new_idx;
}

void TesselatePath(M4Mesh *mesh, M4Path *path, u32 outline_style)
{
	u32 i, j;
	u32 *idx;
	Float w, h;
	GLdouble vertex[3];
	MeshTess *tess;
	if (!mesh || !path || !path->subpathlen) return;
	tess = malloc(sizeof(MeshTess));
	if (!tess) return;
	memset(tess, 0, sizeof(MeshTess));
    tess->tess_obj = gluNewTess();
    if (!tess->tess_obj) {
		free(tess);
		return;
	}
	tess->vertex_index = NewChain();

	mesh_reset(mesh);
	mesh->flags |= MESH_IS_2D;
	if (outline_style==1) mesh->flags |= MESH_NO_TEXTURE;
	
	tess->mesh = mesh;
    gluTessCallback(tess->tess_obj, GLU_TESS_VERTEX_DATA, (void (CALLBACK*)()) &mesh_tess_vertex);
    gluTessCallback(tess->tess_obj, GLU_TESS_BEGIN, (void (CALLBACK*)()) &mesh_tess_begin);
    gluTessCallback(tess->tess_obj, GLU_TESS_END, (void (CALLBACK*)()) &mesh_tess_end);
    gluTessCallback(tess->tess_obj, GLU_TESS_COMBINE_DATA, (void (CALLBACK*)()) &mesh_tess_combine);
    gluTessCallback(tess->tess_obj, GLU_TESS_ERROR, (void (CALLBACK*)()) &mesh_tess_error);
	gluTessCallback(tess->tess_obj, GLU_EDGE_FLAG,(void (CALLBACK*)()) &mesh_tess_edgeflag);

	/*force 0/non0 winding for outlines*/
	if (outline_style) gluTessProperty(tess->tess_obj, GLU_TESS_WINDING_RULE, GLU_TESS_WINDING_POSITIVE);

    gluTessBeginPolygon(tess->tess_obj, tess);

	w = path->max_x - path->min_x;
	h = path->max_y - path->min_y;
	vertex[2] = 0;
	/*since we're not sure whether subpaths overlaps or not, tesselate everything*/
	for (i=0; i<path->subpathlen; i++) {
		M4SubPath *sp = path->subpath[i];
		u32 len = sp->pointlen;
		if (!len) continue;

		gluTessBeginContour(tess->tess_obj);


		for (j=0; j<len; j++) {
			M4Point2D pt = sp->point[j];
			Float u = (Float) (pt.x - path->min_x) / w;
			Float v = (Float) (pt.y - path->min_y) / h;

			idx = (u32 *) malloc(sizeof(u32));
			*idx = mesh->v_count;
			ChainAddEntry(tess->vertex_index, idx);
			mesh_set_vertex(mesh, pt.x, pt.y, 0, 0, 0, 1, u, v);

			vertex[0] = (Double) pt.x;
			vertex[1] = (Double) pt.y;
			gluTessVertex(tess->tess_obj, vertex, idx);
		}
		gluTessEndContour(tess->tess_obj);
	}

    gluTessEndPolygon(tess->tess_obj);

	gluDeleteTess(tess->tess_obj);

	while (ChainGetCount(tess->vertex_index)) {
		u32 *idx = ChainGetEntry(tess->vertex_index, 0);
		ChainDeleteEntry(tess->vertex_index, 0);
		free(idx);
	}
	DeleteChain(tess->vertex_index);
	free(tess);

	mesh->bounds.min_edge.x = path->min_x;
	mesh->bounds.min_edge.y = path->min_y;
	mesh->bounds.max_edge.x = path->max_x;
	mesh->bounds.max_edge.y = path->max_y;
	mesh->bounds.min_edge.z = mesh->bounds.max_edge.z = 0;
	bbox_refresh(&mesh->bounds);
}


#define GetPoint2D(pt, apt) \
	if (!direction) { pt.x = - apt.pos.z; pt.y = apt.pos.y;	}	\
	else if (direction==1) { pt.x = apt.pos.z; pt.y = apt.pos.x; }	\
	else if (direction==2) { pt.x = apt.pos.x; pt.y = apt.pos.y; } \


#define ConvCompare(delta)	\
    ( (delta.x > 0) ? -1 :		\
      (delta.x < 0) ?	1 :		\
      (delta.y > 0) ? -1 :		\
      (delta.y < 0) ?	1 :	\
      0 )

#define ConvGetPointDelta(delta, pprev, pcur )			\
    /* Given a previous point 'pprev', read a new point into 'pcur' */	\
    /* and return delta in 'delta'.				    */	\
    GetPoint2D(pcur, pts[iread]); iread++;			\
    delta.x = pcur.x - pprev.x;					\
    delta.y = pcur.y - pprev.y;					\

#define ConvCross(p, q) p.x*q.y - p.y*q.x;

#define ConvCheckTriple						\
    if ( (thisDir = ConvCompare(dcur)) == -curDir ) {			\
	  ++dirChanges;							\
	  /* if ( dirChanges > 2 ) return NotConvex;		     */ \
    }									\
    curDir = thisDir;							\
    cross = ConvCross(dprev, dcur);					\
    if ( cross > 0 ) { \
		if ( angleSign == -1 ) return M4_PolyComplex;		\
		angleSign = 1;					\
	}							\
    else if (cross < 0) {	\
		if (angleSign == 1) return M4_PolyComplex;		\
		angleSign = -1;				\
	}						\
    pSecond = pThird;		\
    dprev.x = dcur.x;		\
    dprev.y = dcur.y;							\

u32 polygon_check_convexity(M4Vertex *pts, u32 len, u32 direction)
{
	s32 curDir, thisDir = 0, dirChanges = 0, angleSign = 0;
	u32 iread;
	Float cross;
	M4Point2D pSecond, pThird, pSaveSecond;
	M4Point2D dprev, dcur;

	if (len<3) return M4_PolyConvexLine;

	GetPoint2D(pThird, pts[0]);
    /* Get different point, return if less than 3 diff points. */
    if (len < 3 ) return M4_PolyConvexLine;
    iread = 1;
	ConvGetPointDelta(dprev, pThird, pSecond);
    pSaveSecond = pSecond;
	/*initial direction */
    curDir = ConvCompare(dprev);
    while ( iread < len) {
		/* Get different point, break if no more points */
		ConvGetPointDelta(dcur, pSecond, pThird );
		if ( (dcur.x == 0.0f) && (dcur.y == 0.0f) ) continue;
		/* Check current three points */
		ConvCheckTriple;
    }

    /* Must check for direction changes from last vertex back to first */
	/* Prepare for 'ConvexCheckTriple' */
	GetPoint2D(pThird, pts[0]);
    dcur.x = pThird.x - pSecond.x;
    dcur.y = pThird.y - pSecond.y;
    if ( ConvCompare(dcur) ) ConvCheckTriple;

    /* and check for direction changes back to second vertex */
    dcur.x = pSaveSecond.x - pSecond.x;
    dcur.y = pSaveSecond.y - pSecond.y;
	/* Don't care about 'pThird' now */
    ConvCheckTriple;			

    /* Decide on polygon type given accumulated status */
    if ( dirChanges > 2 ) return M4_PolyComplex;
    if ( angleSign > 0 ) return M4_PolyConvexCCW;
    if ( angleSign < 0 ) return M4_PolyConvexCW;
    return M4_PolyConvexLine;
}


void TesselateFaceMesh(M4Mesh *dest, M4Mesh *orig)
{
	u32 poly_type, i, nb_pts, init_idx, direction;
    Float max_nor_coord, c;
	SFVec3f v1, v2, nor;
	u32 *idx;
	GLdouble vertex[3];
	MeshTess *tess;

	/*get normal*/
	v1 = vec_diff(&orig->vertices[1].pos, &orig->vertices[0].pos);
	v2 = vec_diff(&orig->vertices[2].pos, &orig->vertices[1].pos);
	nor = vec_cross(&v1, &v2);
	vec_norm(&nor);
	/*select projection direction*/
	direction = 0;
	max_nor_coord = (Float) fabs(nor.x);
	c = (Float) fabs(nor.y);
	if (c>max_nor_coord) {
		direction = 1;
		max_nor_coord = c;
	}
	c = (Float) fabs(nor.z);
	if (c>max_nor_coord) direction = 2;

	/*if this is a convex polygone don't triangulate*/
	poly_type = polygon_check_convexity(orig->vertices, orig->v_count, direction);
	switch (poly_type) {
	case M4_PolyConvexLine:
		return;
	/*do NOT try to make face CCW otherwise we loos front/back faces...*/
	case M4_PolyConvexCW:
	case M4_PolyConvexCCW:
		init_idx = dest->v_count;
		for (i=0; i<orig->v_count; i++) {
			mesh_set_vertex_vx(dest, &orig->vertices[i]);
		}
		nb_pts = orig->v_count;
		for (i=1; i<nb_pts-1; i++) {
			mesh_set_triangle(dest, init_idx, init_idx + i, init_idx + i+1);
		}
		return;
	default:
		break;
	}

	
	/*tesselate it*/
	tess = malloc(sizeof(MeshTess));
	if (!tess) return;
	memset(tess, 0, sizeof(MeshTess));
    tess->tess_obj = gluNewTess();
    if (!tess->tess_obj) {
		free(tess);
		return;
	}
	tess->vertex_index = NewChain();

	tess->mesh = dest;
    gluTessCallback(tess->tess_obj, GLU_TESS_VERTEX_DATA, (void (CALLBACK*)()) &mesh_tess_vertex);
    gluTessCallback(tess->tess_obj, GLU_TESS_BEGIN, (void (CALLBACK*)()) &mesh_tess_begin);
    gluTessCallback(tess->tess_obj, GLU_TESS_END, (void (CALLBACK*)()) &mesh_tess_end);
    gluTessCallback(tess->tess_obj, GLU_TESS_COMBINE_DATA, (void (CALLBACK*)()) &mesh_tess_combine);
    gluTessCallback(tess->tess_obj, GLU_TESS_ERROR, (void (CALLBACK*)()) &mesh_tess_error);
	gluTessCallback(tess->tess_obj, GLU_EDGE_FLAG,(void (CALLBACK*)()) &mesh_tess_edgeflag);

    gluTessBeginPolygon(tess->tess_obj, tess);
	gluTessBeginContour(tess->tess_obj);

	vertex[2] = 0;

	for (i=0; i<orig->v_count; i++) {
		idx = (u32 *) malloc(sizeof(u32));
		*idx = dest->v_count;
		ChainAddEntry(tess->vertex_index, idx);
		mesh_set_vertex_vx(dest, &orig->vertices[i]);

		vertex[0] = (Double) orig->vertices[i].pos.x;
		vertex[1] = (Double) orig->vertices[i].pos.y;
		gluTessVertex(tess->tess_obj, vertex, idx);
	}

	gluTessEndContour(tess->tess_obj);
    gluTessEndPolygon(tess->tess_obj);
	gluDeleteTess(tess->tess_obj);

	while (ChainGetCount(tess->vertex_index)) {
		u32 *idx = ChainGetEntry(tess->vertex_index, 0);
		ChainDeleteEntry(tess->vertex_index, 0);
		free(idx);
	}
	DeleteChain(tess->vertex_index);
	free(tess);
}


