/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2004 
 *					All rights reserved
 *
 *  This file is part of GPAC / Scene Rendering sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */



#include "stacks3d.h"

#ifdef M4_DEF_Anchor

static void DestroyAnchor(SFNode *n)
{
	AnchorStack *st = (AnchorStack*)Node_GetPrivate(n);
	if (st->compositor->interaction_sensors) st->compositor->interaction_sensors--;
	DeleteGroupingNode((GroupingNode *)st);
	free(st);
}

static void RenderAnchor(SFNode *node, void *rs)
{
	AnchorStack *st = (AnchorStack *) Node_GetPrivate(node);
	B_Anchor *an = (B_Anchor *) node;
	RenderEffect *eff = rs;

	if (!st->compositor->client->EventProc) {
		st->enabled = 0;
		return;
	}

	/*note we don't clear dirty flag, this is done in traversing*/
	if (Node_GetDirty(node) & SG_NODE_DIRTY) {
		st->enabled = 0;
		if (an->url.count && an->url.vals[0].url && strlen(an->url.vals[0].url) )
			st->enabled = 1;
	}
	
	grouping_traverse((GroupingNode*)st, eff);
}

static void OnAnchor(SensorHandler *sh, Bool is_over, Bool is_active, SFVec3f *local_pt, SFVec3f *world_pt, M4Matrix *mx)
{
	AnchorStack *st = (AnchorStack *) Node_GetPrivate(sh->owner);

	if (st->active && !is_active) {
		u32 i;
		M4Event evt;
		B_Anchor *an = (B_Anchor *) sh->owner;
		evt.type = M4E_NAVIGATE;
		i=0;
		while (i<an->url.count) {
			evt.navigate.to_url = an->url.vals[i].url;
			if (!evt.navigate.to_url) break;
			if (st->compositor->client->EventProc(st->compositor->client->opaque, &evt))
				break;

			i++;
		}
	}
	st->active = is_active;
}

static Bool anchor_is_enabled(SFNode *node)
{
	AnchorStack *st = (AnchorStack *) Node_GetPrivate(node);
	return st->enabled;
}

static void on_activate_anchor(SFNode *node)
{
	UserEvent3D ev;
	AnchorStack *st = (AnchorStack *) Node_GetPrivate(node);
	if (!st->enabled) return;

	ev.event_type = M4E_LEFTUP;
	OnAnchor(&st->hdl, 0, 1, NULL, NULL, NULL);
}

SensorHandler *r3d_anchor_get_handler(SFNode *n)
{
	AnchorStack *st = (AnchorStack *) Node_GetPrivate(n);
	return &st->hdl;
}


void R3D_InitAnchor(Render3D *sr, SFNode *node)
{
	B_Anchor *an = (B_Anchor *)node;
	AnchorStack *stack = malloc(sizeof(AnchorStack));
	memset(stack, 0, sizeof(AnchorStack));
	stack->hdl.IsEnabled = anchor_is_enabled;
	stack->hdl.OnUserEvent = OnAnchor;
	stack->hdl.owner = node;
	an->on_activate = on_activate_anchor;
	SetupGroupingNode((GroupingNode*)stack, sr->compositor, node, an->children);
	sr->compositor->interaction_sensors++;
	Node_SetPrivate(node, stack);
	Node_SetPreDestroyFunction(node, DestroyAnchor);
	Node_SetRenderFunction(node, RenderAnchor);
}

#endif


#ifdef M4_DEF_DiscSensor


static void DestroyDiscSensor(SFNode *node)
{
	DiscSensorStack *st = (DiscSensorStack *) Node_GetPrivate(node);
	if (st->compositor->interaction_sensors) st->compositor->interaction_sensors--;
	free(st);
}

static Bool ds_is_enabled(SFNode *n)
{
	return ((B_DiscSensor *) n)->enabled;
}


static void OnDiscSensor(SensorHandler *sh, Bool is_over, Bool is_down, SFVec3f *local_pt, SFVec3f *world_pt, M4Matrix *mx)
{
	SFVec3f pt;
	B_DiscSensor *ds = (B_DiscSensor *)sh->owner;
	DiscSensorStack *stack = (DiscSensorStack *) Node_GetPrivate(sh->owner);
	
	assert(ds->enabled);

	if (is_down && !ds->isActive) {
		/*store inverse matrix*/
		mx_copy(stack->initial_matrix, *mx);
		stack->start_angle = atan2(local_pt->y, local_pt->x);
		ds->isActive = 1;
		Node_OnEventOutSTR(sh->owner, "isActive");
	} else if (!is_down && ds->isActive) {
		if (ds->autoOffset) {
			ds->offset = ds->rotation_changed;
			/*that's an exposedField*/
			Node_OnEventOutSTR(sh->owner, "offset");
		}
		ds->isActive = 0;
		Node_OnEventOutSTR(sh->owner, "isActive");
	}
	else if (is_down) {
		Double result;
		pt = *world_pt;
		mx_apply_vec(&stack->initial_matrix, &pt);
	
		result = atan2(pt.y, pt.x) - stack->start_angle + ds->offset;
		if (ds->minAngle < ds->maxAngle) {
			/*FIXME this doesn't work properly*/
			if (result < ds->minAngle) result = ds->minAngle;
			if (result > ds->maxAngle) result = ds->maxAngle;
		}
		ds->rotation_changed = (Float) result;
		Node_OnEventOutSTR(sh->owner, "rotation_changed");
	   	ds->trackPoint_changed.x = local_pt->x;
	   	ds->trackPoint_changed.y = local_pt->y;
		Node_OnEventOutSTR(sh->owner, "trackPoint_changed");
	}
}

SensorHandler *r3d_ds_get_handler(SFNode *n)
{
	DiscSensorStack *st = (DiscSensorStack *)Node_GetPrivate(n);
	return &st->hdl;
}

void R3D_InitDiscSensor(Render3D *sr, SFNode *node)
{
	DiscSensorStack *st;
	st = malloc(sizeof(DiscSensorStack));
	memset(st, 0, sizeof(DiscSensorStack));
	st->hdl.IsEnabled = ds_is_enabled;
	st->hdl.OnUserEvent = OnDiscSensor;
	st->hdl.owner = node;
	st->compositor = sr->compositor;
	sr->compositor->interaction_sensors++;
	Node_SetPrivate(node, st);
	Node_SetPreDestroyFunction(node, DestroyDiscSensor);
}


#endif


#ifdef M4_DEF_PlaneSensor2D


static void DestroyPlaneSensor2D(SFNode *node)
{
	PS2DStack *st = (PS2DStack *) Node_GetPrivate(node);
	if (st->compositor->interaction_sensors) st->compositor->interaction_sensors--;
	free(st);
}

static Bool ps2D_is_enabled(SFNode *n)
{
	return ((B_PlaneSensor2D *) n)->enabled;
}

static void OnPlaneSensor2D(SensorHandler *sh, Bool is_over, Bool is_down, SFVec3f *local_pt, SFVec3f *world_pt, M4Matrix *mx)
{
	B_PlaneSensor2D *ps = (B_PlaneSensor2D *)sh->owner;
	PS2DStack *stack = (PS2DStack *) Node_GetPrivate(sh->owner);
	
	assert(ps->enabled);

	if (is_down && !ps->isActive) {
		mx_copy(stack->initial_matrix, *mx);
		stack->start_drag.x = local_pt->x - ps->offset.x;
		stack->start_drag.y = local_pt->y - ps->offset.y;
		ps->isActive = 1;
		Node_OnEventOutSTR(sh->owner, "isActive");
	}
	else if (!is_down && ps->isActive) {
		ps->isActive = 0;
		Node_OnEventOutSTR(sh->owner, "isActive");
		if (ps->autoOffset) {
			ps->offset = ps->translation_changed;
			Node_OnEventOutSTR(sh->owner, "translation_changed");
		}
	} else if (is_down) {
		SFVec2f res;
		SFVec3f pt = *world_pt;
		mx_apply_vec(&stack->initial_matrix, &pt);
		res.x = pt.x - stack->start_drag.x;
		res.y = pt.y - stack->start_drag.y;

		/*clip*/
		if (ps->minPosition.x <= ps->maxPosition.x) {
			if (res.x < ps->minPosition.x) res.x = ps->minPosition.x;
			if (res.x > ps->maxPosition.x) res.x = ps->maxPosition.x;
		}
		if (ps->minPosition.y <= ps->maxPosition.y) {
			if (res.y < ps->minPosition.y) res.y = ps->minPosition.y;
			if (res.y > ps->maxPosition.y) res.y = ps->maxPosition.y;
		}
		ps->translation_changed = res;
		Node_OnEventOutSTR(sh->owner, "translation_changed");
		ps->trackPoint_changed.x = local_pt->x;
		ps->trackPoint_changed.y = local_pt->y;
		Node_OnEventOutSTR(sh->owner, "trackPoint_changed");
	}
}

SensorHandler *r3d_ps2D_get_handler(SFNode *n)
{
	PS2DStack *st = (PS2DStack *)Node_GetPrivate(n);
	return &st->hdl;
}

void R3D_InitPlaneSensor2D(Render3D *sr, SFNode *node)
{
	PS2DStack *st = malloc(sizeof(PS2DStack));
	memset(st, 0, sizeof(PS2DStack));
	st->hdl.IsEnabled = ps2D_is_enabled;
	st->hdl.OnUserEvent = OnPlaneSensor2D;
	st->hdl.owner = node;
	st->compositor = sr->compositor;
	st->compositor->interaction_sensors++;
	Node_SetPrivate(node, st);
	Node_SetPreDestroyFunction(node, DestroyPlaneSensor2D);
}


#endif

#ifdef M4_DEF_ProximitySensor2D


static void DestroyProximitySensor2D(SFNode *node)
{
	Prox2DStack *st = (Prox2DStack *) Node_GetPrivate(node);
	if (st->compositor->interaction_sensors) st->compositor->interaction_sensors--;
	free(st);
}

static Bool prox2D_is_enabled(SFNode *n)
{
	return ((B_ProximitySensor2D *) n)->enabled;
}

static Bool prox2D_is_in_sensor(Prox2DStack *st, B_ProximitySensor2D *ps, Float X, Float Y)
{
	if (X < ps->center.x - ps->size.x/2) return 0;
	if (X > ps->center.x + ps->size.x/2) return 0;
	if (Y < ps->center.y - ps->size.y/2) return 0;
	if (Y > ps->center.y + ps->size.y/2) return 0;
	return 1;
}

static void OnProximitySensor2D(SensorHandler *sh, Bool is_over, Bool is_down, SFVec3f *local_pt, SFVec3f *world_pt, M4Matrix *mx)
{
	B_ProximitySensor2D *ps = (B_ProximitySensor2D *)sh->owner;
	Prox2DStack *stack = (Prox2DStack *) Node_GetPrivate(sh->owner);
	
	assert(ps->enabled);
	
	if (is_over) {
		stack->last_time = Node_GetSceneTime(sh->owner);
		if (prox2D_is_in_sensor(stack, ps, local_pt->x, local_pt->y)) {
			ps->position_changed.x = local_pt->x;
			ps->position_changed.y = local_pt->y;
			Node_OnEventOutSTR(sh->owner, "position_changed");

			if (!ps->isActive) {
				ps->isActive = 1;
				Node_OnEventOutSTR(sh->owner, "isActive");
				ps->enterTime = stack->last_time;
				Node_OnEventOutSTR(sh->owner, "enterTime");
			}
			return;
		}
	} 
	/*either we're not over the shape or we're not in sensor*/
	if (ps->isActive) {
		ps->exitTime = stack->last_time;
		Node_OnEventOutSTR(sh->owner, "exitTime");
		ps->isActive = 0;
		Node_OnEventOutSTR(sh->owner, "isActive");
	}
}

SensorHandler *r3d_prox2D_get_handler(SFNode *n)
{
	Prox2DStack *st = (Prox2DStack *)Node_GetPrivate(n);
	return &st->hdl;
}


void R3D_InitProximitySensor2D(Render3D *sr, SFNode *node)
{
	Prox2DStack *st = malloc(sizeof(Prox2DStack));
	memset(st, 0, sizeof(Prox2DStack));
	st->hdl.IsEnabled = prox2D_is_enabled;
	st->hdl.OnUserEvent = OnProximitySensor2D;
	st->hdl.owner = node;
	st->compositor = sr->compositor;
	st->compositor->interaction_sensors++;
	Node_SetPrivate(node, st);
	Node_SetPreDestroyFunction(node, DestroyProximitySensor2D);
}


#endif


#ifdef M4_DEF_TouchSensor

static void DestroyTouchSensor(SFNode *node)
{
	TouchSensorStack *st = (TouchSensorStack *) Node_GetPrivate(node);
	if (st->compositor->interaction_sensors) st->compositor->interaction_sensors--;
	free(st);
}

static Bool ts_is_enabled(SFNode *n)
{
	return ((B_TouchSensor *) n)->enabled;
}

static void OnTouchSensor(SensorHandler *sh, Bool is_over, Bool is_down, SFVec3f *local_pt, SFVec3f *world_pt, M4Matrix *mx)
{
	B_TouchSensor *ts = (B_TouchSensor *)sh->owner;
	assert(ts->enabled);

	/*isActive becomes false, send touch time*/
	if (is_over && !is_down && ts->isActive) {
		ts->touchTime = Node_GetSceneTime(sh->owner);
		Node_OnEventOutSTR(sh->owner, "touchTime");
	}
	if (is_over != ts->isOver) {
		ts->isOver = is_over;
		Node_OnEventOutSTR(sh->owner, "isOver");
	}
	if (is_down != ts->isActive) {
		ts->isActive = is_down;
		Node_OnEventOutSTR(sh->owner, "isActive");
	}
	if (is_over) {
		ts->hitPoint_changed.x = local_pt->x;
		ts->hitPoint_changed.y = local_pt->y;
		ts->hitPoint_changed.z = local_pt->z;
		Node_OnEventOutSTR(sh->owner, "hitPoint_changed");
	}
}

SensorHandler *r3d_touch_sensor_get_handler(SFNode *n)
{
	TouchSensorStack *ts = (TouchSensorStack *)Node_GetPrivate(n);
	return &ts->hdl;
}


void R3D_InitTouchSensor(Render3D *sr, SFNode *node)
{
	TouchSensorStack *st = malloc(sizeof(TouchSensorStack));
	memset(st, 0, sizeof(TouchSensorStack));
	st->hdl.IsEnabled = ts_is_enabled;
	st->hdl.OnUserEvent = OnTouchSensor;
	st->hdl.owner = node;
	st->compositor = sr->compositor;
	st->compositor->interaction_sensors++;
	Node_SetPrivate(node, st);
	Node_SetPreDestroyFunction(node, DestroyTouchSensor);
}



#endif
