/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2004 
 *					All rights reserved
 *
 *  This file is part of GPAC / Scene Rendering sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */



#include "stacks3d.h"

#ifdef M4_DEF_OrderedGroup


M4INLINE B_OrderedGroup *og_getnode(OrderedGroupStack *ptr) { return (B_OrderedGroup *) ptr->owner; }

static void DestroyOrderedGroup(SFNode *node)
{
	OrderedGroupStack *ptr = (OrderedGroupStack *) Node_GetPrivate(node);
	DeleteGroupingNode((GroupingNode *)ptr);
	if (ptr->priorities) free(ptr->priorities);
	free(ptr);
}

static s32 compare_priority(const void* elem1, const void* elem2)
{
	struct og_pos *p1, *p2;
	p1 = (struct og_pos *)elem1;
	p2 = (struct og_pos *)elem2;
	if (p1->priority < p2->priority) return -1;
	if (p1->priority > p2->priority) return 1;
	return 0;
}


static void RenderOrderedGroup(SFNode *node, void *rs)
{
	u32 i, count, mode_back;
	SFNode *child;
	Bool split_text_backup, get_bounds;
	B_OrderedGroup *og;
	OrderedGroupStack *ogs = (OrderedGroupStack *) Node_GetPrivate(node);
	RenderEffect *eff = (RenderEffect *)rs;

	og = og_getnode(ogs);
	if (!og->order.count) {
		grouping_traverse((GroupingNode*)ogs, eff);
		return;
	}
	
	count = ChainGetCount(og->children);

	/*check whether the OrderedGroup node has changed*/
	if (Node_GetDirty(node) & SG_NODE_DIRTY) {
		if (ogs->priorities) free(ogs->priorities);
		ogs->priorities = malloc(sizeof(struct og_pos)*count);
		for (i=0; i<count; i++) {
			ogs->priorities[i].position = i;
			ogs->priorities[i].priority = (i<og->order.count) ? og->order.vals[i] : 0;
		}
		qsort(ogs->priorities, count, sizeof(struct og_pos), compare_priority);
	}

	if (Node_GetDirty(node) & SG_CHILD_DIRTY) {
		/*traverse subtree to recompute bounds*/
		if (eff->traversing_mode!=TRAVERSE_GET_BOUNDS) {
			u32 mode_back = eff->traversing_mode;
			eff->traversing_mode=TRAVERSE_GET_BOUNDS;
			grouping_traverse((GroupingNode*)ogs, eff);
			eff->traversing_mode = mode_back;
		}
	}
	/*sub-tree not dirty and getting bounds, direct copy*/
	else if (eff->traversing_mode==TRAVERSE_GET_BOUNDS) {
		eff->bbox = ogs->bbox;
		Node_ClearDirty(node);
		return;
	}
	Node_ClearDirty(node);

	mode_back=eff->cull_flag;
	if (!eff->traversing_mode && !ogs->dont_cull && !node_cull_or_render(eff, &ogs->bbox)) {
		eff->cull_flag = mode_back;
		return;
	}

	split_text_backup = eff->text_split_mode;
	if (count>1) eff->text_split_mode = 0;
	get_bounds = (eff->traversing_mode==TRAVERSE_GET_BOUNDS) ? 1 : 0;
	if (get_bounds) ogs->dont_cull = ogs->bbox.is_set = eff->bbox.is_set = 0;

	for (i=0; i<count; i++) {
		child = ChainGetEntry(og->children, ogs->priorities[i].position);
		Node_Render(child, eff);
		if (get_bounds) {
			if (eff->bbox.is_set) {
				bbox_union(&ogs->bbox, &eff->bbox);
				eff->bbox.is_set = 0;
			} else if (eff->trav_flags & TF_DONT_CULL) {
				ogs->dont_cull = 1;
				eff->trav_flags &= ~TF_DONT_CULL;
			}
		}
	}
	if (get_bounds) {
		eff->bbox = ogs->bbox;
		if (ogs->dont_cull) eff->trav_flags |= TF_DONT_CULL;
	}
	eff->text_split_mode = split_text_backup;
	eff->cull_flag = mode_back;
}

void R3D_InitOrderedGroup(Render3D *sr, SFNode *node)
{
	OrderedGroupStack *ptr = malloc(sizeof(OrderedGroupStack));
	memset(ptr, 0, sizeof(OrderedGroupStack));
	SetupGroupingNode((GroupingNode*)ptr, sr->compositor, node, ((B_OrderedGroup *)node)->children);
	
	Node_SetPrivate(node, ptr);
	Node_SetPreDestroyFunction(node, DestroyOrderedGroup);
	Node_SetRenderFunction(node, RenderOrderedGroup);
}

#endif


#ifdef M4_DEF_Switch
/*
		Switch 
*/
static void DestroySwitch(SFNode *node)
{
	SwitchStack *st = (SwitchStack *)Node_GetPrivate(node);
	free(st);
}
static void RenderSwitch(SFNode *node, void *rs)
{
	u32 i;
	Bool prev_switch;
	SFNode *child;
	SwitchStack *st = (SwitchStack *)Node_GetPrivate(node);
	B_Switch *sw = (B_Switch *) node;
	RenderEffect *eff; 
	u32 count = ChainGetCount(sw->choice);

	eff = (RenderEffect *)rs;

	prev_switch = eff->trav_flags;

	/*check changes in choice field*/
	if (st->last_switch != sw->whichChoice) {
		eff->trav_flags |= TF_SWITCHED_OFF;
		/*deactivation must be signaled because switch may contain audio nodes (I hate this spec!!!)*/
		for (i=0; i<count; i++) {
			if ((s32) i==sw->whichChoice) continue;
			child = ChainGetEntry(sw->choice, i);
			Node_Render(child, eff);
		}
		eff->trav_flags &= ~TF_SWITCHED_OFF;
		st->last_switch = sw->whichChoice;
	}

	eff->trav_flags = prev_switch;
	if (sw->whichChoice>=0) {
		child = ChainGetEntry(sw->choice, sw->whichChoice);
		Node_Render(child, eff);
	}
}

void R3D_InitSwitch(Render3D *sr, SFNode *node)
{
	SwitchStack *st = malloc(sizeof(SwitchStack));
	st->last_switch = -1;
	Node_SetPrivate(node, st);
	Node_SetPreDestroyFunction(node, DestroySwitch);
	Node_SetRenderFunction(node, RenderSwitch);
}
#endif




#ifdef M4_DEF_ColorTransform

static void DestroyColorTransform(SFNode *n)
{
	ColorTransformStack *ptr = (ColorTransformStack *)Node_GetPrivate(n);
	DeleteGroupingNode((GroupingNode *)ptr);
	free(ptr);
}

/*ColorTransform*/
static void RenderColorTransform(SFNode *node, void *rs)
{
	B_ColorTransform *tr = (B_ColorTransform *)node;
	ColorTransformStack *ptr = (ColorTransformStack  *)Node_GetPrivate(node);
	RenderEffect *eff;
	eff = (RenderEffect *) rs;
	if (Node_GetDirty(node) & SG_NODE_DIRTY) {
		cmat_set(&ptr->cmat, 
			tr->mrr , tr->mrg, tr->mrb, tr->mra, tr->tr, 
			tr->mgr , tr->mgg, tr->mgb, tr->mga, tr->tg, 
			tr->mbr, tr->mbg, tr->mbb, tr->mba, tr->tb, 
			tr->mar, tr->mag, tr->mab, tr->maa, tr->ta); 
	}

	/*note we don't clear dirty flag, this is done in traversing*/
	if (ptr->cmat.identity) {
		grouping_traverse((GroupingNode *) ptr, eff);
	} else {
		M4ColorMatrix cmat_bck;
		Bool prev_cmat = eff->has_cmat;
		cmat_copy(&cmat_bck, &eff->color_mat);
		cmat_multiply(&eff->color_mat, &ptr->cmat);
		eff->has_cmat = 1;
		grouping_traverse((GroupingNode *) ptr, eff);
		/*restore effects*/
		cmat_copy(&eff->color_mat, &cmat_bck);
		eff->has_cmat = prev_cmat;
	}
}

void R3D_InitColorTransform(Render3D *sr, SFNode *node)
{
	ColorTransformStack *stack = malloc(sizeof(ColorTransformStack));
	SetupGroupingNode((GroupingNode *)stack, sr->compositor, node, ((B_ColorTransform *)node)->children);
	cmat_init(&stack->cmat);
	Node_SetPrivate(node, stack);
	Node_SetPreDestroyFunction(node, DestroyColorTransform);
	Node_SetRenderFunction(node, RenderColorTransform);
}

#endif


#ifdef M4_DEF_Group
static void RenderGroup(SFNode *node, void *rs)
{
	GroupingNode *group = (GroupingNode *) Node_GetPrivate(node);
	grouping_traverse(group, (RenderEffect*)rs);
}
void R3D_InitGroup(Render3D *sr, SFNode *node)
{
	GroupingNode *stack = malloc(sizeof(GroupingNode));
	SetupGroupingNode(stack, sr->compositor, node, ((B_Group *)node)->children);
	Node_SetPrivate(node, stack);
	Node_SetPreDestroyFunction(node, DestroyBaseGrouping);
	Node_SetRenderFunction(node, RenderGroup);
}
#endif


static void DestroyTransform(SFNode *n)
{
	TransformStack *ptr = (TransformStack *)Node_GetPrivate(n);
	DeleteGroupingNode((GroupingNode *)ptr);
	free(ptr);
}

static void NewTransformStack(Render3D *sr, SFNode *node, Chain *children)
{
	TransformStack *st= malloc(sizeof(TransformStack));
	memset(st, 0, sizeof(TransformStack));
	mx_init(st->mx);
	SetupGroupingNode((GroupingNode *)st, sr->compositor, node, children);
	Node_SetPrivate(node, st);
	Node_SetPreDestroyFunction(node, DestroyTransform);
}

#ifdef M4_DEF_Transform

static void RenderTransform(SFNode *n, void *rs)
{
	M4Matrix mx_bckup;
	TransformStack *st = Node_GetPrivate(n);
	B_Transform *tr = (B_Transform *)n;
	RenderEffect *eff = (RenderEffect *)rs;

	if (Node_GetDirty(n) & SG_NODE_DIRTY) {
		Bool scale_rot, recenter;
		mx_init(st->mx);
		if (tr->translation.x || tr->translation.y || tr->translation.z) 
			mx_add_translation(&st->mx, tr->translation.x, tr->translation.y, tr->translation.z);
		recenter = (tr->center.x || tr->center.y || tr->center.z) ? 1 : 0;
		if (recenter) 
			mx_add_translation(&st->mx, tr->center.x, tr->center.y, tr->center.z);

		if (tr->rotation.angle) mx_add_rotation(&st->mx, tr->rotation.angle, tr->rotation.xAxis, tr->rotation.yAxis, tr->rotation.zAxis);
		scale_rot = (tr->scaleOrientation.angle) ? 1 : 0;
		if (scale_rot) 
			mx_add_rotation(&st->mx, tr->scaleOrientation.angle, tr->scaleOrientation.xAxis, tr->scaleOrientation.yAxis, tr->scaleOrientation.zAxis);
		if ((tr->scale.x != 1.0f) || (tr->scale.y != 1.0f) || (tr->scale.z != 1.0f)) 
			mx_add_scale(&st->mx, tr->scale.x, tr->scale.y, tr->scale.z);
		if (scale_rot) 
			mx_add_rotation(&st->mx, -tr->scaleOrientation.angle, tr->scaleOrientation.xAxis, tr->scaleOrientation.yAxis, tr->scaleOrientation.zAxis);
		if (recenter) 
			mx_add_translation(&st->mx, -tr->center.x, -tr->center.y, -tr->center.z);

   	}

	/*for fun we allow 3D nodes under 2D layout, form & co*/

	mx_copy(mx_bckup, eff->model_view);
	mx_add_matrix(&eff->model_view, &st->mx);

	VS3D_PushMatrix(eff->surface);
	VS3D_MultMatrix(eff->surface, st->mx.m);

	
	/*note we don't clear dirty flag, this is done in traversing*/
	grouping_traverse((GroupingNode *) st, eff);

	
	VS3D_PopMatrix(eff->surface);

	mx_copy(eff->model_view, mx_bckup);

	if (eff->traversing_mode==TRAVERSE_GET_BOUNDS) mx_apply_bbox(&st->mx, &eff->bbox);
}

void R3D_InitTransform(Render3D *sr, SFNode *node)
{
	NewTransformStack(sr, node, ((B_Transform *)node)->children);
	Node_SetRenderFunction(node, RenderTransform);

}

#endif


#ifdef M4_DEF_Transform2D
static void RenderTransform2D(SFNode *node, void *rs)
{
	M4Matrix mx_bckup;
	B_Transform2D *tr = (B_Transform2D *)node;
	TransformStack *st = (TransformStack*)Node_GetPrivate(node);
	RenderEffect *eff = (RenderEffect *) rs;

	if (Node_GetDirty(node) & SG_NODE_DIRTY) {
		M4Matrix2D mx;
		mx2d_init(mx);
		if ((tr->scale.x != 1.0) || (tr->scale.y != 1.0)) 
			mx2d_add_scale_at(&mx, tr->scale.x, tr->scale.y, 0, 0, tr->scaleOrientation);
		if (tr->rotationAngle) 
			mx2d_add_rotation(&mx, tr->center.x, tr->center.y, tr->rotationAngle);
		if (tr->translation.x || tr->translation.y) 
			mx2d_add_translation(&mx, tr->translation.x, tr->translation.y);

		mx_from_mx2d(&st->mx, &mx);
	}

	mx_copy(mx_bckup, eff->model_view);
	mx_add_matrix(&eff->model_view, &st->mx);

	VS3D_PushMatrix(eff->surface);
	VS3D_MultMatrix(eff->surface, st->mx.m);

	/*note we don't clear dirty flag, this is done in traversing*/
	grouping_traverse((GroupingNode *) st, eff);
	
	VS3D_PopMatrix(eff->surface);

	mx_copy(eff->model_view, mx_bckup);

	if (eff->traversing_mode==TRAVERSE_GET_BOUNDS) 
		mx_apply_bbox(&st->mx, &eff->bbox);
}
void R3D_InitTransform2D(Render3D *sr, SFNode *node)
{
	NewTransformStack(sr, node, ((B_Transform2D *)node)->children);
	Node_SetRenderFunction(node, RenderTransform2D);
}
#endif

#ifdef M4_DEF_TransformMatrix2D

void TM2D_GetMatrix(SFNode *n, M4Matrix *mx)
{
	M4Matrix2D mat;
	B_TransformMatrix2D *tm = (B_TransformMatrix2D*)n;
	mx2d_init(mat);
	mat.m[0] = tm->mxx; mat.m[1] = tm->mxy; mat.m[2] = tm->tx;
	mat.m[3] = tm->myx; mat.m[4] = tm->myy; mat.m[5] = tm->ty;
	mx_from_mx2d(mx, &mat);
}


/*TransformMatrix2D*/
static void RenderTransformMatrix2D(SFNode *node, void *rs)
{
	M4Matrix mx_bckup;
	TransformStack *st = (TransformStack *) Node_GetPrivate(node);
	RenderEffect *eff = (RenderEffect *)rs;

	if (Node_GetDirty(node) & SG_NODE_DIRTY) {
		TM2D_GetMatrix(node, &st->mx);
	}

	mx_copy(mx_bckup, eff->model_view);
	mx_add_matrix(&eff->model_view, &st->mx);

	VS3D_PushMatrix(eff->surface);
	VS3D_MultMatrix(eff->surface, st->mx.m);

	/*note we don't clear dirty flag, this is done in traversing*/
	grouping_traverse((GroupingNode *)st, eff);
	
	VS3D_PopMatrix(eff->surface);

	mx_copy(eff->model_view, mx_bckup);

	if (eff->traversing_mode==TRAVERSE_GET_BOUNDS) mx_apply_bbox(&st->mx, &eff->bbox);
}


void R3D_InitTransformMatrix2D(Render3D *sr, SFNode *node)
{
	NewTransformStack(sr, node, ((B_TransformMatrix2D *)node)->children);
	Node_SetRenderFunction(node, RenderTransformMatrix2D);
}

#endif



