/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2004 
 *					All rights reserved
 *
 *  This file is part of GPAC / Scene Rendering sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */


#include "stacks3d.h"

SensorHandler *get_sensor_handler(SFNode *n, Bool skip_anchor)
{
	SensorHandler *hs;

	switch (Node_GetTag(n)) {
#ifdef M4_DEF_Anchor
	case TAG_Anchor: if (skip_anchor) return NULL; hs = r3d_anchor_get_handler(n); break;
#endif
#ifdef M4_DEF_DiscSensor
	case TAG_DiscSensor: hs = r3d_ds_get_handler(n); break;
#endif
#ifdef M4_DEF_TouchSensor
	case TAG_TouchSensor: hs = r3d_touch_sensor_get_handler(n); break;
#endif
#ifdef M4_DEF_PlaneSensor2D
	case TAG_PlaneSensor2D: hs = r3d_ps2D_get_handler(n); break;
#endif
#ifdef M4_DEF_ProximitySensor2D
	case TAG_ProximitySensor2D: hs = r3d_prox2D_get_handler(n); break;
#endif
	default:
		return NULL;
	}
	if (hs && hs->IsEnabled(n)) return hs;
	return NULL;
}


void group_start_child(GroupingNode *group, SFNode *n)
{
	ChildGroup *cg;
	if (!n) {
		ChildGroup *cg_prev = ChainGetEntry(group->groups, ChainGetCount(group->groups)-1);
		n = cg_prev ? cg_prev->child : NULL;
	}
	if (!n) return;
	cg = malloc(sizeof(ChildGroup));
	memset(cg, 0, sizeof(ChildGroup));
	cg->child = n;
	ChainAddEntry(group->groups, cg);
}

void group_end_child(GroupingNode *group, M4BBox *bounds)
{
	ChildGroup *cg = ChainGetEntry(group->groups, ChainGetCount(group->groups)-1);
	if (!cg) return;
#ifdef M4_DEF_Text
	/*don't override splitted text info*/
	if (cg->is_text_group) return;
#endif
	m4_rect_from_bbox(&cg->original, bounds);
	cg->final = cg->original;
}

#ifdef M4_DEF_Text
void group_end_text_child(GroupingNode *group, M4Rect *bounds, Float ascent, Float descent, u32 split_text_idx)
{
	ChildGroup *cg = ChainGetEntry(group->groups, ChainGetCount(group->groups)-1);
	if (!cg) return;
	cg->split_text_idx = split_text_idx;
	cg->is_text_group = 1;
	cg->ascent = ascent;
	cg->descent = descent;
	cg->final = cg->original = *bounds;
}
#endif

void group_reset_children(GroupingNode *group)
{
	while (ChainGetCount(group->groups)) {
		ChildGroup *cg = ChainGetEntry(group->groups, 0);
		ChainDeleteEntry(group->groups, 0);
		free(cg);
	}
}



/*This is the generic routine for child traversing*/
void grouping_traverse(GroupingNode *group, RenderEffect *eff)
{
	u32 i, count, mode_back;
	Bool split_text_backup, is_parent, get_bounds;
	SFNode *child;

	is_parent = (eff->parent == group) ? 1 : 0;

	if (Node_GetDirty(group->owner) & SG_CHILD_DIRTY) {
		/*traverse subtree to recompute bounds*/
		if (eff->traversing_mode!=TRAVERSE_GET_BOUNDS) {
			mode_back = eff->traversing_mode;
			eff->traversing_mode=TRAVERSE_GET_BOUNDS;
			grouping_traverse(group, eff);
			eff->traversing_mode = mode_back;
		}
	}
	/*not parent (eg form, layout...) sub-tree not dirty and getting bounds, direct copy */
	else if (!is_parent && (eff->traversing_mode==TRAVERSE_GET_BOUNDS)) {
		eff->bbox = group->bbox;
		Node_ClearDirty(group->owner);
		return;
	}
	Node_ClearDirty(group->owner);

	mode_back=eff->cull_flag;
	if (!eff->traversing_mode && !group->dont_cull && !node_cull_or_render(eff, &group->bbox)) {
		eff->cull_flag = mode_back;
		return;
	}

	count = ChainGetCount(group->children);
	split_text_backup = eff->text_split_mode;
	get_bounds = (!is_parent && (eff->traversing_mode==TRAVERSE_GET_BOUNDS)) ? 1 : 0;

	if (!is_parent && (count>1)) eff->text_split_mode = 0;
	if (get_bounds) group->dont_cull = group->bbox.is_set = eff->bbox.is_set = 0;
		
	for (i=0; i<count; i++) {
		child = ChainGetEntry(group->children, i);
		if (is_parent) group_start_child(group, child);
		Node_Render(child, eff);
		if (is_parent) group_end_child(group, &eff->bbox);
		else if (get_bounds) {
			if (eff->bbox.is_set) {
				bbox_union(&group->bbox, &eff->bbox);
				eff->bbox.is_set = 0;
			} else if (eff->trav_flags & TF_DONT_CULL) {
				group->dont_cull = 1;
				eff->trav_flags &= ~TF_DONT_CULL;
			}
		}
	} 

	if (get_bounds) {
		eff->bbox = group->bbox;
		if (group->dont_cull) eff->trav_flags |= TF_DONT_CULL;
	}
	eff->text_split_mode = split_text_backup;
	eff->cull_flag = mode_back;
}

/*final drawing of each group*/
void child_render_done(ChildGroup *cg, RenderEffect *eff)
{
	M4Matrix mx, mx_bckup;
	mx_init(mx);
	mx_add_translation(&mx, cg->final.x - cg->original.x, cg->final.y - cg->original.y, 0);

	mx_copy(mx_bckup, eff->model_view);
	mx_add_translation(&eff->model_view, cg->final.x - cg->original.x, cg->final.y - cg->original.y, 0);

	VS3D_PushMatrix(eff->surface);
	VS3D_MultMatrix(eff->surface, mx.m);
	eff->split_text_idx = cg->split_text_idx;
	Node_Render(cg->child, eff);
	eff->split_text_idx = 0;
	VS3D_PopMatrix(eff->surface);

	mx_copy(eff->model_view, mx_bckup);
}

void child_render_done_complex(ChildGroup *cg, RenderEffect *eff, M4Matrix2D *mat2D)
{
	M4Matrix mx, mx_bckup;
	if (!mat2D) return;

	mx_from_mx2d(&mx, mat2D);

	mx_copy(mx_bckup, eff->model_view);
	mx_add_matrix(&eff->model_view, &mx);

	VS3D_PushMatrix(eff->surface);
	VS3D_MultMatrix(eff->surface, mx.m);
	eff->split_text_idx = cg->split_text_idx;
	Node_Render(cg->child, eff);
	eff->split_text_idx = 0;
	VS3D_PopMatrix(eff->surface);

	mx_copy(eff->model_view, mx_bckup);
}


void SetupGroupingNode(GroupingNode *group, LPSCENERENDER sr, SFNode *node, Chain *children)
{
	memset(group, 0, sizeof(GroupingNode));
	traversable_setup(group, node, sr);
	group->groups = NewChain();
	group->children = children;
}

void DeleteGroupingNode(GroupingNode *group)
{
	/*just in case*/
	group_reset_children(group);
	DeleteChain(group->groups);
}

void DestroyBaseGrouping(SFNode *node)
{
	GroupingNode *group = (GroupingNode *)Node_GetPrivate(node);
	DeleteGroupingNode(group);
	free(group);
}

void NewGroupingNodeStack(LPSCENERENDER sr, SFNode *node, Chain *children)
{
	GroupingNode *st = malloc(sizeof(GroupingNode));
	if (!st) return;
	SetupGroupingNode(st, sr, node, children);
	Node_SetPrivate(node, st);
	Node_SetPreDestroyFunction(node, DestroyBaseGrouping);
	return;
}

