/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2003 
 *					All rights reserved
 *
 *  This file is part of GPAC / Scene Rendering sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#include "stacks2d.h"
#include "visualsurface2d.h"

#ifdef M4_DEF_Viewport

static void DestroyViewport(SFNode *node)
{
	s32 i;
	Chain *stack;
	ViewportStack *tmp, *ptr = (ViewportStack *) Node_GetPrivate(node);
	B_Viewport *vp = (B_Viewport *) ptr->owner;

	while (ChainGetCount(ptr->stack_list)) {
		stack = ChainGetEntry(ptr->stack_list, 0);
		i = ChainDeleteItem(stack, ptr);
		if (i==0) {
			tmp = ChainGetEntry(stack, 0);
			/*we were bound so bind new top*/
			if (tmp) {
				vp->isBound = 1;
				vp->bindTime = Node_GetSceneTime(node);
				Node_OnEventOutSTR(tmp->owner, "isBound");
				Node_OnEventOutSTR(tmp->owner, "bindTime");
			}
		}
		ChainDeleteEntry(ptr->stack_list, 0);
	}
	DeleteChain(ptr->stack_list);
	free(ptr);
}

static void viewport_set_bind(SFNode *node)
{
	Bool on_top;
	u32 i;
	Chain *stack;
	ViewportStack *tmp, *ptr = (ViewportStack *) Node_GetPrivate(node);
	B_Viewport *vp = (B_Viewport *) ptr->owner;

	//notify all stacks using this node
	for (i=0; i<ChainGetCount(ptr->stack_list);i++) {
		stack = ChainGetEntry(ptr->stack_list, i);
		on_top = (ChainGetEntry(stack, 0) == ptr) ? 1 : 0;
	
		if (! vp->set_bind) {
			if (vp->isBound) {
				vp->isBound = 0;
				Node_OnEventOutSTR(node, "isBound");
			}

			if (on_top) {
				tmp = ChainGetEntry(stack, 0);
				ChainDeleteEntry(stack, 0);
				ChainAddEntry(stack, tmp);

				tmp = ChainGetEntry(stack, 0);
				if (tmp != ptr) {
					B_Viewport *vp2 = (B_Viewport *) tmp->owner;
					vp2->isBound = 1;
					Node_OnEventOutSTR(node, "isBound");
				}
			}
		} else {
			if (! vp->isBound) {
				vp->isBound = 1;
				vp->bindTime = Node_GetSceneTime(node);
				Node_OnEventOutSTR(node, "isBound");
				Node_OnEventOutSTR(node, "bindTime");
			}

			if (!on_top) {
				tmp = ChainGetEntry(stack, 0);
				ChainDeleteItem(stack, ptr);
				ChainInsertEntry(stack, ptr, 0);
				if (tmp != ptr) {
					B_Viewport *vp2 = (B_Viewport *) tmp->owner;
					vp2->isBound = 0;
					Node_OnEventOutSTR(tmp->owner, "isBound");
				}
			}
		}
	}
	SR_Invalidate(ptr->compositor, NULL);
}



static Chain *vp_get_stack(ViewportStack *vp, RenderEffect *eff)
{
	Chain *stack = NULL;
	u32 i;	
	stack = eff->view_stack;
	if (!stack) return NULL;

	for (i=0; i<ChainGetCount(vp->stack_list); i++) {
		if (stack == ChainGetEntry(vp->stack_list, i) ) return stack;	
	}
	ChainAddEntry(vp->stack_list, stack);
	ChainAddEntry(stack, vp);
	/*need a callback to user*/
	return stack;
}

static void RenderViewport(SFNode *node, void *rs)
{
	ViewportStack *st = (ViewportStack *) Node_GetPrivate(node);
	B_Viewport *vp = (B_Viewport *) st->owner;

	if (st->first_time) {
		Chain *stack = vp_get_stack(st, (RenderEffect *)rs);
		if (ChainGetEntry(stack, 0) == st) {
			if (! vp->isBound) {
				vp->isBound = 1;
				vp->bindTime = Node_GetSceneTime(node);
				Node_OnEventOutSTR(node, "isBound");
				Node_OnEventOutSTR(node, "bindTime");
			}
		}
		st->first_time = 0;
	}
}

void R2D_InitViewport(Render2D *sr, SFNode *node)
{
	ViewportStack *ptr = malloc(sizeof(ViewportStack));
	memset(ptr, 0, sizeof(ViewportStack));
	ptr->first_time = 1;
	ptr->stack_list = NewChain();

	traversable_setup(ptr, node, sr->compositor);
	Node_SetPrivate(node, ptr);
	Node_SetRenderFunction(node, RenderViewport);
	Node_SetPreDestroyFunction(node, DestroyViewport);
	((B_Viewport*)node)->on_set_bind = viewport_set_bind;
}



void vp_setup(ViewportStack *vps, RenderEffect *eff, M4Rect *surf_clip)
{
	Float ar, sx, sy, w, h, tx, ty;
	B_Viewport *vp;
	M4Matrix2D mat;
	M4Rect rc;
	if (!surf_clip->width || !surf_clip->height) return;

	vp = (B_Viewport *) vps->owner;
	mx2d_init(mat);
	mx2d_add_translation(&mat, -1 * vp->position.x, -1 * vp->position.y);
	mx2d_add_rotation(&mat, 0, 0, -1 * vp->orientation);
	
	mx2d_add_matrix(&eff->transform, &mat);

	mx2d_copy(mat, eff->transform);

	//compute scaling ratio
	m4_rect_center(&rc, vp->size.x, vp->size.y);
	mx2d_apply_rect(&mat, &rc);

	w = surf_clip->width;
	h = surf_clip->height;
	ar = h / w;
	
	surf_clip->width = rc.width;
	surf_clip->height = rc.height;

	switch (vp->fit) {
	//covers all area and respect aspect ratio
	case 2:
		if (rc.width/w > rc.height/h) {
			rc.width *= h/rc.height;
			rc.height = h;
		} else {
			rc.height *= w/rc.width;
			rc.width = w;
		}
		break;
	//fits inside the area and respect AR
	case 1:
		if (rc.width/w> rc.height/h) {
			rc.height *= w/rc.width;
			rc.width = w;
		} else {
			rc.width *= h/rc.height;
			rc.height = h;
		}
		break;
	//fit entirely: nothing to change
	case 0:
		rc.width = w;
		rc.height = h;
		break;
	default:
		return;
	}
	sx = rc.width / surf_clip->width;
	sy = rc.height / surf_clip->height;

	surf_clip->width = rc.width;
	surf_clip->height = rc.height;
	surf_clip->x = - rc.width/2;
	surf_clip->y = rc.height/2;

	mx2d_init(mat);
	if (!vp->fit) {
		mx2d_add_scale(&mat, sx, sy);
		mx2d_add_matrix(&eff->transform, &mat);
		return;
	}

	//setup x-alignment
	switch (vp->alignment.vals[0]) {
	//left align: 
	case -1:
		tx = rc.width/2 - w/2;
		break;
	//right align
	case 1:
		tx = w/2 - rc.width/2;
		break;
	//center
	case 0:
	default:
		tx = 0;
		break;
	}

	//setup y-alignment
	switch (vp->alignment.vals[1]) {
	//left align: 
	case -1:
		ty = rc.height/2 - h/2;
		break;
	//right align
	case 1:
		ty = h/2 - rc.height/2;
		break;
	//center
	case 0:
	default:
		ty = 0;
		break;
	}

	mx2d_add_scale(&mat, sx, sy);
	mx2d_add_translation(&mat, tx, ty);
	mx2d_add_matrix(&eff->transform, &mat);
	surf_clip->x += tx;
	surf_clip->y += ty;
}

#endif
