/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2003 
 *					All rights reserved
 *
 *  This file is part of GPAC / MP4 reader plugin
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *		
 */

#include "ffmpeg_in.h"
#include <gpac/m4_author.h>

#ifndef FF_NO_DEMUX

#define FFD_DATA_BUFFER	800

static u32 FFDemux_Run(void *par)
{
	AVPacket pkt;
	u64 seek_to;
	u64 aud_ts, vid_ts;
	s32 ret;
	NetworkCommand com;
	SLHeader slh;
	FFDemux *ffd = (FFDemux *) par;
	
	memset(&com, 0, sizeof(NetworkCommand));
	com.command_type = CHAN_BUFFER_QUERY;

	memset(&slh, 0, sizeof(SLHeader));

	/*this is still buggy in ffmpeg with some mpeg files*/
	seek_to = (s64) (AV_TIME_BASE*ffd->seek_time);
	ret = av_seek_frame(ffd->ctx, -1, seek_to);
	slh.compositionTimeStampFlag = slh.decodingTimeStampFlag = 1;
	aud_ts = vid_ts = seek_to;

	while (ffd->is_running) {

		pkt.stream_index = -1;
		/*EOF*/
        if (av_read_frame(ffd->ctx, &pkt) <0) break;

		MX_P(ffd->mx);
		if (pkt.pts == AV_NOPTS_VALUE) pkt.pts = pkt.dts;

		if (!ffd->use_packet_durations) {
			if ((u32) pkt.pts<seek_to) {
				slh.decodingTimeStamp = slh.compositionTimeStamp = seek_to;
			} else {
				slh.compositionTimeStamp = pkt.pts;
				slh.decodingTimeStamp = pkt.dts;
			}
		}

		if (ffd->audio_ch && (pkt.stream_index == ffd->audio_st)) {
			if (ffd->use_packet_durations) slh.compositionTimeStamp = slh.decodingTimeStamp = aud_ts;
			NM_OnSLPRecieved(ffd->service, ffd->audio_ch, pkt.data, pkt.size, &slh, M4OK);
			if (ffd->use_packet_durations) aud_ts += pkt.duration;
		} else if (ffd->video_ch && (pkt.stream_index == ffd->video_st)) {
			if (ffd->use_packet_durations) slh.compositionTimeStamp = slh.decodingTimeStamp = vid_ts;
			NM_OnSLPRecieved(ffd->service, ffd->video_ch, pkt.data, pkt.size, &slh, M4OK);
			if (ffd->use_packet_durations) vid_ts += pkt.duration;
		}
		av_free_packet(&pkt);
		MX_V(ffd->mx);

		/*sleep untill the buffer occupancy is too low - note that this work because all streams in this
		demuxer are synchronized*/
		while (1) {
			if (ffd->audio_ch) {
				com.on_channel = ffd->audio_ch;
				NM_OnCommand(ffd->service, &com);
				if (com.buffer_occupancy < ffd->data_buffer_ms) break;
			}
			if (ffd->video_ch) {
				com.on_channel = ffd->video_ch;
				NM_OnCommand(ffd->service, &com);
				if (com.buffer_occupancy < ffd->data_buffer_ms) break;
			}
			Sleep(10);
			
			/*escape if disconnect*/
			if ((ffd->audio_status != NM_Running) && (ffd->video_status != NM_Running)) break;
		}
		if ((ffd->audio_status != NM_Running) && (ffd->video_status != NM_Running)) break;
	}
	/*signal EOS*/
	if (ffd->audio_ch) NM_OnSLPRecieved(ffd->service, ffd->audio_ch, NULL, 0, NULL, M4EOF);
	if (ffd->video_ch) NM_OnSLPRecieved(ffd->service, ffd->video_ch, NULL, 0, NULL, M4EOF);
	ffd->is_running = 0;

	return 0;
}

static Bool FFD_CanHandleURL(NetClientPlugin *plug, const char *url)
{
	Bool has_audio, has_video;
	s32 i;
	Bool ret = 0;
	AVFormatContext *ctx;
	char *ext, szName[1000], szExt[20], *szExtList;

	strcpy(szName, url);
	ext = strrchr(szName, '#');
	if (ext) ext[0] = 0;

	/*disable RTP/RTSP from ffmpeg*/
	if (!strnicmp(szName, "rtsp://", 7)) return 0;
	if (!strnicmp(szName, "rtspu://", 8)) return 0;
	if (!strnicmp(szName, "rtp://", 6)) return 0;

	ext = strrchr(szName, '.');
	if (ext) {
		strcpy(szExt, &ext[1]);
		strlwr(szExt);

		/*check supported extension list - note by default we forbid ffmpeg to handle mp4 and mp3 files*/
		szExtList = PMI_GetOpt(plug, "FileAssociations", "FFMPEG demuxer");
		if (!szExtList) {
			szExtList = "avi mpeg mpg vob ogg raw wmv asf au gif rm wav";
			PMI_SetOpt(plug, "FileAssociations", "FFMPEG demuxer", szExtList);
		}
		if (!strstr(szExtList, szExt)) return 0;
	}

	ctx = NULL;
    if (av_open_input_file(&ctx, szName, NULL, 0, NULL)<0)
		return 0;

    if (!ctx || av_find_stream_info(ctx) <0) goto exit;
	/*figure out if we can use codecs or not*/
	has_video = has_audio = 0;
    for(i = 0; i < ctx->nb_streams; i++) {
        AVCodecContext *enc = &ctx->streams[i]->codec;
        switch(enc->codec_type) {
        case CODEC_TYPE_AUDIO:
            if (!has_audio) has_audio = 1;
            break;
        case CODEC_TYPE_VIDEO:
            if (!has_video) has_video= 1;
            break;
        default:
            break;
        }
    }
	if (!has_audio && !has_video) goto exit;
	ret = 1;

exit:
    if (ctx) av_close_input_file(ctx);
	return ret;
}

static M4Err FFD_ConnectService(NetClientPlugin *plug, LPNETSERVICE serv, const char *url)
{
	M4Err e;
	s32 i;
	char *sOpt;
	FFDemux *ffd = plug->priv;
	char *ext, szName[1000];

	if (ffd->ctx) return M4ServiceError;

	strcpy(szName, url);
	ext = strrchr(szName, '#');
	ffd->service_type = 0;
	if (ext) {
		if (!stricmp(&ext[1], "video")) ffd->service_type = 1;
		else if (!stricmp(&ext[1], "audio")) ffd->service_type = 2;
		ext[0] = 0;
	}
    if (av_open_input_file(&ffd->ctx, szName, NULL, 0, NULL)<0)
		return M4ServiceError;

	e = M4UnsupportedURL;
	ffd->service = serv;

    if (av_find_stream_info(ffd->ctx) <0) goto err_exit;
	/*figure out if we can use codecs or not*/
	ffd->audio_st = ffd->video_st = -1;
    for (i = 0; i < ffd->ctx->nb_streams; i++) {
        AVCodecContext *enc = &ffd->ctx->streams[i]->codec;
        switch(enc->codec_type) {
        case CODEC_TYPE_AUDIO:
            if (ffd->audio_st<0) ffd->audio_st = i;
            break;
        case CODEC_TYPE_VIDEO:
            if (ffd->video_st<0) ffd->video_st = i;
            break;
        default:
            break;
        }
    }
	if ((ffd->service_type==1) && (ffd->video_st<0)) goto err_exit;
	if ((ffd->service_type==2) && (ffd->audio_st<0)) goto err_exit;
	if ((ffd->video_st<0) && (ffd->audio_st<0)) goto err_exit;

	/*setup indexes for BIFS/OD*/
	ffd->bifs_es_id = 2 + MAX(ffd->video_st, ffd->audio_st);
	ffd->od_es_id = ffd->bifs_es_id + 1;

	ffd->status = NM_Connected;
	sOpt = PMI_GetOpt(plug, "FFMPEG", "DataBufferMS"); 
	ffd->data_buffer_ms = 0;
	if (sOpt) ffd->data_buffer_ms = atoi(sOpt);
	if (!ffd->data_buffer_ms) ffd->data_buffer_ms = FFD_DATA_BUFFER;

	sOpt = PMI_GetOpt(plug, "FFMPEG", "UsePacketDuration");
	if (sOpt && !stricmp(sOpt, "yes")) ffd->use_packet_durations = 1;

	/*let's go*/
	NM_OnConnect(serv, NULL, M4OK);
	return M4OK;

err_exit:
    av_close_input_file(ffd->ctx);
	ffd->ctx = NULL;
	ffd->status = NM_Unavailable;
	NM_OnConnect(serv, NULL, e);
	return M4OK;
}


static ESDescriptor *FFD_GetESDescriptor(FFDemux *ffd, Bool for_audio)
{
	BitStream *bs;
	ESDescriptor *esd = (ESDescriptor *) OD_NewESDescriptor(0);
	esd->ESID = 1 + (for_audio ? ffd->audio_st : ffd->video_st);
	esd->decoderConfig->streamType = for_audio ? M4ST_AUDIO : M4ST_VISUAL;
	esd->decoderConfig->avgBitrate = esd->decoderConfig->avgBitrate = 0;

	/*remap std object types - depending on input formats, FFMPEG may not have separate DSI from initial frame. 
	In this case we have no choice but using FFMPEG decoders*/
	if (for_audio) {
		switch (ffd->ctx->streams[ffd->audio_st]->codec.codec_id) {
		case CODEC_ID_MP2:
			esd->decoderConfig->objectTypeIndication = 0x6B;
			break;
		case CODEC_ID_MP3:
			esd->decoderConfig->objectTypeIndication = 0x69;
			break;
		case CODEC_ID_MPEG4AAC:
		case CODEC_ID_AAC:
			if (!ffd->ctx->streams[ffd->audio_st]->codec.extradata_size) goto opaque_audio;
			esd->decoderConfig->objectTypeIndication = 0x40;
			esd->decoderConfig->decoderSpecificInfo->dataLength = ffd->ctx->streams[ffd->audio_st]->codec.extradata_size;
			esd->decoderConfig->decoderSpecificInfo->data = malloc(sizeof(char)*ffd->ctx->streams[ffd->audio_st]->codec.extradata_size);
			memcpy(esd->decoderConfig->decoderSpecificInfo->data, 
					ffd->ctx->streams[ffd->audio_st]->codec.extradata, 
					sizeof(char)*ffd->ctx->streams[ffd->audio_st]->codec.extradata_size);
			break;
		default:
opaque_audio:
			esd->decoderConfig->objectTypeIndication = GPAC_FFMPEG_CODECS_OTI;
			bs = NewBitStream(NULL, 0, BS_WRITE);
			BS_WriteInt(bs, ffd->ctx->streams[ffd->audio_st]->codec.codec_id, 32);
			BS_WriteInt(bs, ffd->ctx->streams[ffd->audio_st]->codec.sample_rate, 32);
			BS_WriteInt(bs, ffd->ctx->streams[ffd->audio_st]->codec.channels, 16);
			BS_WriteInt(bs, ffd->ctx->streams[ffd->audio_st]->codec.bits_per_sample, 16);
			BS_WriteInt(bs, ffd->ctx->streams[ffd->audio_st]->codec.frame_size, 16);
			BS_WriteInt(bs, ffd->ctx->streams[ffd->audio_st]->codec.block_align, 16);

			BS_WriteInt(bs, ffd->ctx->streams[ffd->audio_st]->codec.codec_tag, 32);
			BS_WriteInt(bs, ffd->ctx->streams[ffd->audio_st]->codec.bit_rate, 32);

			if (ffd->ctx->streams[ffd->audio_st]->codec.extradata_size) {
				BS_WriteData(bs, ffd->ctx->streams[ffd->audio_st]->codec.extradata, ffd->ctx->streams[ffd->audio_st]->codec.extradata_size);
			}
			BS_GetContent(bs, (unsigned char **) &esd->decoderConfig->decoderSpecificInfo->data, &esd->decoderConfig->decoderSpecificInfo->dataLength);
			DeleteBitStream(bs);
			break;
		}
	} else {
		switch (ffd->ctx->streams[ffd->video_st]->codec.codec_id) {
		case CODEC_ID_MPEG4:
			/*if dsi not detected force use ffmpeg*/
			if (!ffd->ctx->streams[ffd->video_st]->codec.extradata_size) goto opaque_video;
			/*otherwise use any MPEG-4 Visual*/
			esd->decoderConfig->objectTypeIndication = 0x20;
			esd->decoderConfig->decoderSpecificInfo->dataLength = ffd->ctx->streams[ffd->video_st]->codec.extradata_size;
			esd->decoderConfig->decoderSpecificInfo->data = malloc(sizeof(char)*ffd->ctx->streams[ffd->video_st]->codec.extradata_size);
			memcpy(esd->decoderConfig->decoderSpecificInfo->data, 
					ffd->ctx->streams[ffd->video_st]->codec.extradata, 
					sizeof(char)*ffd->ctx->streams[ffd->video_st]->codec.extradata_size);
			break;
		case CODEC_ID_MPEG1VIDEO:
			esd->decoderConfig->objectTypeIndication = 0x6A;
			break;
		case CODEC_ID_MPEG2VIDEO:
			esd->decoderConfig->objectTypeIndication = 0x65;
			break;
		default:
opaque_video:
			esd->decoderConfig->objectTypeIndication = GPAC_FFMPEG_CODECS_OTI;
			bs = NewBitStream(NULL, 0, BS_WRITE);
			BS_WriteInt(bs, ffd->ctx->streams[ffd->video_st]->codec.codec_id, 32);
			BS_WriteInt(bs, ffd->ctx->streams[ffd->video_st]->codec.width, 32);
			BS_WriteInt(bs, ffd->ctx->streams[ffd->video_st]->codec.height, 32);
			BS_WriteInt(bs, ffd->ctx->streams[ffd->video_st]->codec.codec_tag, 32);
			BS_WriteInt(bs, ffd->ctx->streams[ffd->video_st]->codec.bit_rate, 32);

			if (ffd->ctx->streams[ffd->video_st]->codec.extradata_size) {
				BS_WriteData(bs, ffd->ctx->streams[ffd->video_st]->codec.extradata, ffd->ctx->streams[ffd->video_st]->codec.extradata_size);
			}
			BS_GetContent(bs, (unsigned char **) &esd->decoderConfig->decoderSpecificInfo->data, &esd->decoderConfig->decoderSpecificInfo->dataLength);
			DeleteBitStream(bs);
			break;
		}
	}

	/*we unframe all data*/
	esd->slConfig->useAccessUnitStartFlag = esd->slConfig->useAccessUnitEndFlag = 0;
	/*no proper RAP signal in libavformat yet...*/
	esd->slConfig->useRandomAccessUnitsOnlyFlag = 1;
	esd->slConfig->useTimestampsFlag = 1;
	esd->slConfig->timestampResolution = AV_TIME_BASE;
	return esd;
}

static M4Err FFD_Get_MPEG4_IOD(NetClientPlugin *plug, u32 expect_type, char **raw_iod, u32 *raw_iod_size)
{
	M4Err e;
	ObjectDescriptor *od;
	BitStream *bs;
	ESDescriptor *esd;
	FFDemux *ffd = plug->priv;

	if (!ffd->ctx) return M4ServiceError;

	od = (ObjectDescriptor *) OD_NewDescriptor(ObjectDescriptor_Tag);
	od->objectDescriptorID = 1;

	if (expect_type==NM_OD_AUDIO) {
		esd = FFD_GetESDescriptor(ffd, 1);
		/*if session join, setup sync*/
		if (ffd->video_ch) esd->OCRESID = ffd->video_st+1;
		ChainAddEntry(od->ESDescriptors, esd);
		e = OD_EncDesc((Descriptor *) od, raw_iod, raw_iod_size);
		OD_DeleteDescriptor((Descriptor **)&od);
		return e;
	}
	if (expect_type==NM_OD_VIDEO) {
		esd = FFD_GetESDescriptor(ffd, 0);
		/*if session join, setup sync*/
		if (ffd->audio_ch) esd->OCRESID = ffd->audio_st+1;
		ChainAddEntry(od->ESDescriptors, esd);
		e = OD_EncDesc((Descriptor *) od, raw_iod, raw_iod_size);
		OD_DeleteDescriptor((Descriptor **)&od);
		return e;
	}

	/*setup BIFS stream*/
	esd = (ESDescriptor *) OD_NewESDescriptor(0);
	esd->ESID = ffd->bifs_es_id;
	esd->OCRESID = 0;
	esd->decoderConfig->streamType = M4ST_BIFS;
	esd->decoderConfig->objectTypeIndication = 0x01;
	bs = NewBitStream(NULL, 0, BS_WRITE);
	/*NODEID bits*/
	BS_WriteInt(bs, 0, 5);
	/*ROUTEID bits*/
	BS_WriteInt(bs, 0, 5);
	/*commandstream bit*/
	BS_WriteInt(bs, 1, 1);
	/*pixelMetrics TRUE*/
	BS_WriteInt(bs, 1, 1);
	if (ffd->video_st<0) {
		/*sizeInfo FALSE*/
		BS_WriteInt(bs, 0, 1);
	} else {
		/*sizeInfo TRUE*/
		BS_WriteInt(bs, 1, 1);
		/*width/height*/
		BS_WriteInt(bs, ffd->ctx->streams[ffd->video_st]->codec.width, 16);
		BS_WriteInt(bs, ffd->ctx->streams[ffd->video_st]->codec.height, 16);
	}

	BS_GetContent(bs, (unsigned char **) &esd->decoderConfig->decoderSpecificInfo->data, 
		&esd->decoderConfig->decoderSpecificInfo->dataLength);
	DeleteBitStream(bs);
	/*we only send 1 full AU RAP*/
	esd->slConfig->useAccessUnitStartFlag = esd->slConfig->useAccessUnitEndFlag = 0;
	esd->slConfig->useRandomAccessUnitsOnlyFlag = 1;
	esd->slConfig->useTimestampsFlag = 1;
	esd->slConfig->timestampResolution = 1000;

	OD_AddDescToDesc((Descriptor *)od, (Descriptor *) esd);

	/*setup OD stream (no DSI)*/
	esd = (ESDescriptor *) OD_NewESDescriptor(0);
	esd->ESID = ffd->od_es_id;
	esd->OCRESID = ffd->bifs_es_id;
	esd->decoderConfig->streamType = M4ST_OD;
	esd->decoderConfig->objectTypeIndication = 0x01;
	
	/*we only send 1 full AU RAP*/
	esd->slConfig->useAccessUnitStartFlag = esd->slConfig->useAccessUnitEndFlag = 0;
	esd->slConfig->useRandomAccessUnitsOnlyFlag = 1;
	esd->slConfig->useTimestampsFlag = 1;
	esd->slConfig->timestampResolution = 1000;
	OD_AddDescToDesc((Descriptor *)od, (Descriptor *) esd);

	e = OD_EncDesc((Descriptor *)od, raw_iod, raw_iod_size);
	OD_DeleteDescriptor((Descriptor **) &od);
	return e;
}


static M4Err FFD_CloseService(NetClientPlugin *plug, Bool immediate_shutdown)
{
	FFDemux *ffd = plug->priv;

	ffd->is_running = 0;

	if (ffd->ctx) av_close_input_file(ffd->ctx);
	ffd->ctx = NULL;
	ffd->status = NM_Disconnected;
	ffd->audio_ch = ffd->video_ch = ffd->bifs_ch = ffd->od_ch = NULL;
	ffd->audio_status = ffd->video_status = ffd->bifs_status = ffd->od_status = 0;

	NM_OnDisconnect(ffd->service, NULL, M4OK);
	return M4OK;
}

static M4Err FFD_ConnectChannel(NetClientPlugin *plug, LPNETCHANNEL channel, const char *url, Bool upstream)
{
	M4Err e;
	u32 ESID;
	FFDemux *ffd = plug->priv;

	e = M4OK;
	if (upstream) {
		e = M4InvalidMP4File;
		goto exit;
	}
	if (!strstr(url, "ES_ID=")) {
		e = M4UnsupportedURL;
		goto exit;
	}
	sscanf(url, "ES_ID=%d", &ESID);

	if (ESID==ffd->bifs_es_id) {
		if (ffd->bifs_ch) {
			e = M4ServiceError;
			goto exit;
		}
		ffd->bifs_ch = channel;
		ffd->bifs_status = NM_Connected;
	}
	else if (ESID==ffd->od_es_id) {
		if (ffd->od_ch) {
			e = M4ServiceError;
			goto exit;
		}
		ffd->od_ch = channel;
		ffd->od_status = NM_Connected;
	}
	else if ((s32) ESID == 1 + ffd->audio_st) {
		if (ffd->audio_ch) {
			e = M4ServiceError;
			goto exit;
		}
		ffd->audio_ch = channel;
		ffd->audio_status = NM_Connected;
	}
	else if ((s32) ESID == 1 + ffd->video_st) {
		if (ffd->video_ch) {
			e = M4ServiceError;
			goto exit;
		}
		ffd->video_ch = channel;
		ffd->video_status = NM_Connected;
	}

exit:
	NM_OnConnect(ffd->service, channel, e);
	return M4OK;
}

static M4Err FFD_DisconnectChannel(NetClientPlugin *plug, LPNETCHANNEL channel)
{
	M4Err e;
	FFDemux *ffd = plug->priv;

	e = M4ChannelNotFound;
	if (ffd->bifs_ch == channel) {
		e = M4OK;
		ffd->bifs_ch = NULL;
		ffd->bifs_status = NM_Disconnected;
	}
	else if (ffd->od_ch == channel) {
		e = M4OK;
		ffd->od_ch = NULL;
		ffd->od_status = NM_Disconnected;
	}
	else if (ffd->audio_ch == channel) {
		e = M4OK;
		ffd->audio_ch = NULL;
		ffd->audio_status = NM_Disconnected;
	}
	else if (ffd->video_ch == channel) {
		e = M4OK;
		ffd->video_ch = NULL;
		ffd->video_status = NM_Disconnected;
	}
	NM_OnDisconnect(ffd->service, channel, e);
	return M4OK;
}

static M4Err FFD_GetStatus(NetClientPlugin *plug, LPNETCHANNEL channel, u32 *status)
{
	M4Err e;
	FFDemux *ffd = plug->priv;

	if (!channel) {
		*status = ffd->status;
		return M4OK;
	}

	e = M4ChannelNotFound;
	if (ffd->bifs_ch == channel) {
		e = M4OK;
		*status = ffd->bifs_status;
	}
	else if (ffd->od_ch == channel) {
		e = M4OK;
		*status = ffd->od_status;
	}
	else if (ffd->audio_ch == channel) {
		e = M4OK;
		*status = ffd->audio_status;
	}
	else if (ffd->video_ch == channel) {
		e = M4OK;
		*status = ffd->video_status;
	}
	return e;
}


static M4Err FFD_ServiceCommand(NetClientPlugin *plug, NetworkCommand *com)
{
	FFDemux *ffd = plug->priv;

	if (!com->on_channel) return M4OK;

	switch (com->command_type) {
	/*only BIFS/OD work in pull mode (cf ffmpeg_in.h)*/
	case CHAN_SET_PULL:
		if (ffd->audio_ch==com->on_channel) return M4NotSupported;
		if (ffd->video_ch==com->on_channel) return M4NotSupported;
		return M4OK;
	case CHAN_INTERACTIVE:
		return M4OK;
	case CHAN_BUFFER:
		com->buffer_max = com->buffer_min = 0;
		return M4OK;
	case CHAN_DURATION:
		if (ffd->ctx->duration<0) {
			if ((ffd->video_st>=0) && ffd->ctx->streams[ffd->video_st]->codec.frame_rate) {
				com->duration = ffd->ctx->streams[ffd->video_st]->codec_info_nb_frames;
				com->duration /= ffd->ctx->streams[ffd->video_st]->codec.frame_rate;
			}
		} else {
			com->duration = (Double) ffd->ctx->duration / AV_TIME_BASE;
		}
		return M4OK;
	/*fetch start time*/
	case CHAN_PLAY:
		if (com->speed<0) return M4NotSupported;

		MX_P(ffd->mx);
		ffd->seek_time = (com->start_range>=0) ? com->start_range : 0;
		
		if (ffd->audio_ch==com->on_channel) ffd->audio_status = NM_Running;
		else if (ffd->video_ch==com->on_channel) ffd->video_status = NM_Running;
		else if (ffd->bifs_ch==com->on_channel) ffd->bifs_status = NM_Running;
		else if (ffd->od_ch==com->on_channel) ffd->od_status = NM_Running;

		/*play on media stream, start thread*/
		if ((ffd->audio_ch==com->on_channel) || (ffd->video_ch==com->on_channel)) {
			if (!ffd->is_running) {
				ffd->is_running = 1;
				TH_Run(ffd->thread, FFDemux_Run, ffd);
			}
		}
		MX_V(ffd->mx);
		return M4OK;
	case CHAN_STOP:
		if (ffd->audio_ch==com->on_channel) {
			ffd->audio_status = NM_Connected;
		}
		else if (ffd->video_ch==com->on_channel) {
			ffd->video_status = NM_Connected;
		}
		else if (ffd->bifs_ch==com->on_channel) {
			ffd->bifs_status = NM_Connected;
			ffd->bifs_state = 0;
		}
		else if (ffd->od_ch==com->on_channel) {
			ffd->od_status = NM_Connected;
			ffd->od_state = 0;
		}
		return M4OK;

	default:
		return M4OK;
	}

	return M4OK;
}


M4Err FFD_ChannelGetSLP(NetClientPlugin *plug, LPNETCHANNEL channel, char **out_data_ptr, u32 *out_data_size, struct tagSLHeader *out_sl_hdr, Bool *sl_compressed, M4Err *out_reception_status, Bool *is_new_data)
{
	LPODCODEC odc;
	ODCommand *com;
	ObjectDescriptor *OD;
	ESDescriptor *esd;

	FFDemux *ffd = plug->priv;
	if (ffd->status==NM_Disconnected) return M4BadParam;
	if (channel==ffd->audio_ch) return M4BadParam;
	if (channel==ffd->video_ch) return M4BadParam;

	*sl_compressed = 0;
	memset(out_sl_hdr, 0, sizeof(struct tagSLHeader));
	out_sl_hdr->accessUnitEndFlag = 1;
	out_sl_hdr->accessUnitStartFlag = 1;
	out_sl_hdr->compositionTimeStampFlag = 1;
	out_sl_hdr->compositionTimeStamp = (u32) (1000 * ffd->seek_time);

	if (ffd->bifs_ch==channel) {
		*out_reception_status = (ffd->bifs_state == 2) ? M4EOF : M4OK;
		*is_new_data = ffd->bifs_state ? 0 : 1;
		if ((ffd->audio_st>=0) && (ffd->video_st>=0)) {
			*out_data_ptr = (char *) ISMA_BIFS_AV;
			*out_data_size = sizeof(ISMA_BIFS_AV) / sizeof(char);
		} else if (ffd->audio_st>=0) {
			*out_data_ptr = (char *) ISMA_BIFS_AUDIO;
			*out_data_size = sizeof(ISMA_BIFS_AUDIO) / sizeof(char);
		} else {
			*out_data_ptr = (char *) ISMA_BIFS_VIDEO;
			*out_data_size = sizeof(ISMA_BIFS_VIDEO) / sizeof(char);
		}
		if (!ffd->bifs_state) ffd->bifs_state = 1;
		return M4OK;
	}

	*out_reception_status = (ffd->od_state == 2) ? M4EOF : M4OK;
	if (!ffd->od_au && (ffd->od_state != 2)) {
		/*compute OD AU*/
		com = OD_NewCommand(ODUpdate_Tag);

		if ((ffd->service_type!=1) && (ffd->audio_st>=0)) {
			OD = (ObjectDescriptor *) OD_NewDescriptor(ObjectDescriptor_Tag);
			OD->objectDescriptorID = ISMA_AUDIO_OD_ID;
			esd = FFD_GetESDescriptor(ffd, 1);
			esd->OCRESID = ffd->bifs_es_id;
			OD_AddDescToDesc((Descriptor*)OD, (Descriptor *) esd);
			ChainAddEntry( ((ObjectDescriptorUpdate*)com)->objectDescriptors, OD);

		}
		/*compute video */
		if ((ffd->service_type!=2) && (ffd->video_st>=0)) {
			OD = (ObjectDescriptor *) OD_NewDescriptor(ObjectDescriptor_Tag);
			OD->objectDescriptorID = ISMA_VIDEO_OD_ID;
			esd = FFD_GetESDescriptor(ffd, 0);
			esd->OCRESID = ffd->bifs_es_id;
			OD_AddDescToDesc((Descriptor*)OD, (Descriptor *) esd);
			ChainAddEntry( ((ObjectDescriptorUpdate*)com)->objectDescriptors, OD);
		}
		odc = OD_NewCodec(OD_WRITE);
		OD_AddCommand(odc, com);
		OD_EncodeAU(odc);
		OD_GetEncodedAU(odc, &ffd->od_au, &ffd->od_au_size);
		OD_DeleteCodec(odc);
		*is_new_data = 1;

	} else {
		*is_new_data = 0;
	}
	*out_data_ptr = ffd->od_au;
	*out_data_size = ffd->od_au_size;
	if (!ffd->od_state) ffd->od_state = 1;
	return M4OK;
}

M4Err FFD_ChannelReleaseSLP(NetClientPlugin *plug, LPNETCHANNEL channel)
{
	FFDemux *ffd = plug->priv;
	if (ffd->status==NM_Disconnected) return M4BadParam;
	if (channel==ffd->audio_ch) return M4BadParam;
	if (channel==ffd->video_ch) return M4BadParam;

	if (ffd->bifs_ch==channel) {
		ffd->bifs_state = 2;
		return M4OK;
	}

	ffd->od_state = 2;
	if (ffd->od_au) free(ffd->od_au);
	ffd->od_au = NULL;
	ffd->od_au_size = 0;
	return M4OK;
}

static Bool FFD_CanHandleURLInService(NetClientPlugin *plug, const char *url)
{
	char szURL[2048], *sep;
	FFDemux *ffd = (FFDemux *)plug->priv;
	const char *this_url = NM_GetServiceURL(ffd->service);
	if (!this_url || !url) return 0;

	strcpy(szURL, this_url);
	sep = strrchr(szURL, '#');
	if (sep) sep[0] = 0;

	/*ok this is ugly but will do it untill we have proper file association in place*/
	sep = strrchr(szURL, '.');
	if (sep) {
		if (!stricmp(sep, ".mp3") || !stricmp(sep, ".mp4") || !stricmp(sep, ".3gp")) return 0;
	}

	/*direct addressing in service*/
	if (url[0] == '#') return 1;
	if (strnicmp(szURL, url, sizeof(char)*strlen(szURL))) return 0;
//	if (!strrchr(url, '#')) return 0;
	/*we could also check the track exists but we assume that's fine*/
	return 1;
}

void *New_FFMPEG_Demux() 
{
	FFDemux *priv;
	NetClientPlugin *ffd = malloc(sizeof(NetClientPlugin));
	memset(ffd, 0, sizeof(NetClientPlugin));

	priv = malloc(sizeof(FFDemux));
	memset(priv, 0, sizeof(FFDemux));

    /* register all codecs, demux and protocols */
    av_register_all();
	
	ffd->CanHandleURL = FFD_CanHandleURL;
	ffd->CloseService = FFD_CloseService;
	ffd->ConnectChannel = FFD_ConnectChannel;
	ffd->ConnectService = FFD_ConnectService;
	ffd->DisconnectChannel = FFD_DisconnectChannel;
	ffd->Get_MPEG4_IOD = FFD_Get_MPEG4_IOD;
	ffd->GetStatus = FFD_GetStatus;
	ffd->ServiceCommand = FFD_ServiceCommand;
	/*for BIFS/OD only*/
	ffd->ChannelGetSLP = FFD_ChannelGetSLP;
	ffd->ChannelReleaseSLP = FFD_ChannelReleaseSLP;

	ffd->CanHandleURLInService = FFD_CanHandleURLInService;

	priv->thread = NewThread();
	priv->mx = NewMutex();

	M4_REG_PLUG(ffd, M4STREAMINGCLIENT, "FFMPEG demuxer", "gpac distribution", 0);
	ffd->priv = priv;
	return ffd;
}

void Delete_FFMPEG_Demux(void *ifce)
{
	FFDemux *ffd;
	NetClientPlugin *ptr = (NetClientPlugin *)ifce;

	ffd = ptr->priv;

	TH_Delete(ffd->thread);
	MX_Delete(ffd->mx);

	free(ffd);
	free(ptr);
}

#endif

