/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2004
 *					All rights reserved
 *
 *  This file is part of GPAC / DirectX audio and video render plugin
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *		
 */

#include "SDL_out.h"

#define SDLAUD()	SDLAudCtx *ctx = (SDLAudCtx *)dr->opaque

void sdl_fill_audio(void *udata, Uint8 *stream, int len)
{
	AudioOutput *dr = (AudioOutput *)udata;
	dr->FillBuffer(dr->audio_renderer, stream, len);	
}


static M4Err SDLAud_SetupHardware(AudioOutput *dr, void *os_handle, u32 num_buffers, u32 num_buffers_per_sec)
{
	u32 flags;
	SDL_AudioSpec want_format, got_format;
	SDLAUD();

	/*init sdl*/
	if (!SDLOUT_InitSDL()) return M4IOErr;

	flags = SDL_WasInit(SDL_INIT_AUDIO);
	if (!(flags & SDL_INIT_AUDIO)) {
		if (SDL_InitSubSystem(SDL_INIT_AUDIO)<0) {
			SDLOUT_CloseSDL();
			return M4IOErr;
		}
	}
	/*check we can open the device*/
	memset(&want_format, 0, sizeof(SDL_AudioSpec));
	want_format.freq = 44100;
	want_format.format = AUDIO_S16SYS;
	want_format.channels = 2;
	want_format.samples = 1024;
	want_format.callback = sdl_fill_audio;
	want_format.userdata = dr;
	if ( SDL_OpenAudio(&want_format, &got_format) < 0 ) {
		SDL_QuitSubSystem(SDL_INIT_AUDIO);
		SDLOUT_CloseSDL();
		return M4IOErr;
	}
	SDL_CloseAudio();
	ctx->is_init = 1;
	ctx->num_buffers = num_buffers;
	ctx->num_buffers_per_sec = num_buffers_per_sec;
	return M4OK;
}

static void SDLAud_Shutdown(AudioOutput *dr)
{
	SDLAUD();

	if (ctx->is_running) SDL_CloseAudio();
	if (ctx->is_init) {
		SDL_QuitSubSystem(SDL_INIT_AUDIO);
		SDLOUT_CloseSDL();
		ctx->is_init = 0;
	}
}

static M4Err SDLAud_ConfigureOutput(AudioOutput *dr, u32 *SampleRate, u32 *NbChannels, u32 *nbBitsPerSample)
{
	u32 nb_samples;
	SDL_AudioSpec want_format, got_format;
	SDLAUD();

	if (ctx->is_running) SDL_CloseAudio();
	ctx->is_running = 0;

	memset(&want_format, 0, sizeof(SDL_AudioSpec));
	want_format.freq = *SampleRate;
	want_format.format = (*nbBitsPerSample==16) ? AUDIO_S16SYS : AUDIO_S8;
	want_format.channels = *NbChannels;
	want_format.callback = sdl_fill_audio;
	want_format.userdata = dr;

	if (ctx->num_buffers && ctx->num_buffers_per_sec) {
		nb_samples = want_format.freq;
		nb_samples /= ctx->num_buffers_per_sec;
		if (nb_samples % 2) nb_samples--;
	} else {
		nb_samples = 1024;
	}
	want_format.samples = nb_samples;

	/*if not win32, respect SDL need for power of 2 - otherwise it works with any even value*/
#ifndef WIN32
	want_format.samples = 1;
	while (want_format.samples*2<nb_samples) want_format.samples *= 2;
#endif

	if ( SDL_OpenAudio(&want_format, &got_format) < 0 ) return M4IOErr;
	ctx->is_running = 1;
	ctx->delay_ms = got_format.samples * 1000 / got_format.freq;
	*SampleRate = got_format.freq;
	*NbChannels = got_format.channels;
	switch (got_format.format) {
	case AUDIO_S8:
	case AUDIO_U8:
		*nbBitsPerSample = 8;
		break;
	default:
		*nbBitsPerSample = 16;
		break;
	}
	/*and play*/
	SDL_PauseAudio(0);
	return M4OK;
}

static u32 SDLAud_GetAudioDelay(AudioOutput *dr)
{
	SDLAUD();
	return ctx->delay_ms;
}

static void SDLAud_SetVolume(AudioOutput *dr, u32 Volume)
{
	/*not supported by SDL*/
}

static void SDLAud_SetPan(AudioOutput *dr, u32 pan)
{
	/*not supported by SDL*/
}

static void SDLAud_Pause(AudioOutput *dr, Bool DoFreeze)
{
	SDL_PauseAudio(DoFreeze ? 1 : 0);
}

static void SDLAud_SetPriority(AudioOutput *dr, u32 priority)
{
	/*not supported by SDL*/
}

static u32 SDLAud_QueryOutputSampleRate(AudioOutput *dr, u32 desired_samplerate, u32 NbChannels, u32 nbBitsPerSample)
{
	/*cannot query supported formats in SDL...*/
	return desired_samplerate;
}


void *SDL_NewAudio()
{
	SDLAudCtx *ctx;
	AudioOutput *dr;


	ctx = malloc(sizeof(SDLAudCtx));
	memset(ctx, 0, sizeof(SDLAudCtx));

	dr = malloc(sizeof(AudioOutput));
	memset(dr, 0, sizeof(AudioOutput));
	M4_REG_PLUG(dr, M4_AUDIO_OUTPUT_INTERFACE, "SDL Audio Output", "gpac distribution", 0);

	dr->opaque = ctx;

	dr->SetupHardware = SDLAud_SetupHardware;
	dr->Shutdown = SDLAud_Shutdown;
	dr->ConfigureOutput = SDLAud_ConfigureOutput;
	dr->SetVolume = SDLAud_SetVolume;
	dr->SetPan = SDLAud_SetPan;
	dr->Pause = SDLAud_Pause;
	dr->SetPriority = SDLAud_SetPriority;
	dr->GetAudioDelay = SDLAud_GetAudioDelay;

	dr->QueryOutputSampleRate = SDLAud_QueryOutputSampleRate;
	/*always threaded*/
	dr->SelfThreaded = 1;
	return dr;
}

void SDL_DeleteAudio(void *ifce)
{
	AudioOutput *dr = (AudioOutput*)ifce;
	SDLAUD();

	free(ctx);
	free(ifce);
}

