/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2003 
 *					All rights reserved
 *
 *  This file is part of GPAC / Authoring Tools sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#include <gpac/m4_author.h>
#include <gpac/m4_bifs.h>

static void UpdateODCommand(M4File *mp4, ODCommand *com)
{
	u32 i, j;
	const char *szName;
	char szPath[2048];

	szName = M4_GetFilename(mp4);
	if (com->tag == ODUpdate_Tag) {
		ObjectDescriptorUpdate *odU = (ObjectDescriptorUpdate *)com;
		for (i=0; i<ChainGetCount(odU->objectDescriptors); i++) {
			ObjectDescriptor *od = ChainGetEntry(odU->objectDescriptors, i);
			for (j=0; j<ChainGetCount(od->ESDescriptors); j++) {
				ESDescriptor *esd = ChainGetEntry(od->ESDescriptors, j);
				if (esd->URLString) continue;
				switch (esd->decoderConfig->streamType) {
				case M4ST_OD:
				case M4ST_BIFS:
				case M4ST_OCR:
					break;
				default:
				{
					MuxInfoDescriptor *mi = (MuxInfoDescriptor *) OD_NewDescriptor(MuxInfoDescriptor_Tag);
					ChainAddEntry(esd->extensionDescriptors, mi);
					sprintf(szPath, "%s#%d", szName, esd->ESID);
					mi->file_name = strdup(szPath);
					mi->streamFormat = strdup("MP4");
				}
					break;
				}
			}
		}
		return;
	}
	if (com->tag == ESDUpdate_Tag) {
		ESDescriptorUpdate *esdU = (ESDescriptorUpdate *)com;
		for (i=0; i<ChainGetCount(esdU->ESDescriptors); i++) {
			ESDescriptor *esd = ChainGetEntry(esdU->ESDescriptors, i);
			if (esd->URLString) continue;
			switch (esd->decoderConfig->streamType) {
			case M4ST_OD:
			case M4ST_BIFS:
			case M4ST_OCR:
				break;
			default:
			{
				MuxInfoDescriptor *mi = (MuxInfoDescriptor *) OD_NewDescriptor(MuxInfoDescriptor_Tag);
				ChainAddEntry(esd->extensionDescriptors, mi);
				sprintf(szPath, "%s#%d", szName, esd->ESID);
				mi->file_name = strdup(szPath);
				mi->streamFormat = strdup("MP4");
			}
				break;
			}
		}
		return;
	}
}


M4Err M4SM_LoadContextFromMP4(M4SceneManager *ctx, M4File *mp4, void (*MP4NodeInit)(void *cbk, SFNode *n), void (*MP4OnError)(void *cbk, char *msg), void *mp4_cbck)
{
	u32 i, j, k, di, nbBifs, timeScale;
	M4StreamContext *sc;
	ESDescriptor *esd;
	LPODCODEC oddec;
	LPBIFSDEC bdec;
	LPSCENEGRAPH base_scene;
	M4Err e;

	e = M4OK;
	/*load IOD*/
	ctx->root_od = (ObjectDescriptor *) M4_GetRootOD(mp4);
	if (!ctx->root_od) {
		e = M4_GetLastError(mp4);
		if (e) return e;
	} else if ((ctx->root_od->tag != ObjectDescriptor_Tag) && (ctx->root_od->tag != InitialObjectDescriptor_Tag)) {
		OD_DeleteDescriptor((Descriptor **) &ctx->root_od);
	}
	
	bdec = BIFS_NewDecoder(NULL);
	oddec = OD_NewCodec(OD_READ);

	nbBifs = 0;
	esd = NULL;
	base_scene = NULL;
	/*load each stream*/
	for (i=0; i<M4_GetTrackCount(mp4); i++) {
		u32 type = M4_GetMediaType(mp4, i+1);
		switch (type) {
		case M4_BIFSMediaType:
			break;
		case M4_ODMediaType:
			break;
		default:
			continue;
		}
		esd = M4_GetStreamDescriptor(mp4, i+1, 1);
		if (!esd) {
			continue;
		}
		sc = M4SM_NewStream(ctx, esd->ESID, esd->decoderConfig->streamType, esd->decoderConfig->objectTypeIndication);
		sc->streamType = esd->decoderConfig->streamType;
		sc->ESID = esd->ESID;
		sc->objectType = esd->decoderConfig->objectTypeIndication;
		sc->timeScale = M4_GetMediaTimeScale(mp4, i+1);

		if (esd->decoderConfig->streamType==M4ST_BIFS) {
			if (!esd->dependsOnESID && nbBifs) {
				if (MP4OnError) {
					MP4OnError(mp4_cbck, "Warning: several BIFS namespaces used or improper BIFS dependencies in file - import may not be correct");
				} else {
					fprintf(stdout, "Warning: several BIFS namespaces used or improper BIFS dependencies in file - import may not be correct");
				}
			}

			e = BIFS_ConfigureStream(bdec, esd->ESID, esd->decoderConfig->decoderSpecificInfo->data, esd->decoderConfig->decoderSpecificInfo->dataLength, esd->decoderConfig->objectTypeIndication);
			if (e) goto exit;
			nbBifs++;
		}

		timeScale = M4_GetMediaTimeScale(mp4, i+1);
		/*dump all AUs*/
		for (j=0; j<M4_GetSampleCount(mp4, i+1); j++) {
			M4AUContext *au;
			M4Sample *samp = M4_GetSample(mp4, i+1, j+1, &di);
			if (!samp) {
				e = M4_GetLastError(mp4);
				goto exit;
			}
			au = M4SM_NewAU(sc, samp->DTS, ((Double)samp->DTS) / timeScale, samp->IsRAP);

			if (esd->decoderConfig->streamType==M4ST_BIFS) {
				e = BIFS_DecodeAUMemory(bdec, esd->ESID, samp->data, samp->dataLength, au->commands, MP4NodeInit, mp4_cbck);
				if (!e && !sc->scene_context) {
					for (k=0; k<ChainGetCount(au->commands); k++) {
						SGCommand *com = ChainGetEntry(au->commands, k);
						if (com->tag == SG_SceneReplace) {
							sc->scene_context = com->graph;
							break;
						}
					}
					if (!sc->scene_context) sc->scene_context = base_scene;
					/*needed for scalability/animation stream*/
					if (!base_scene) base_scene = sc->scene_context;
				}
			} else {
				e = OD_SetBuffer(oddec, samp->data, samp->dataLength);
				if (!e) e = OD_DecodeAU(oddec);
				if (!e) {
					while (1) {
						ODCommand *odc = OD_GetCommand(oddec);
						if (!odc) break;
						/*update ESDs if any*/
						UpdateODCommand(mp4, odc);
						ChainAddEntry(au->commands, odc);
					}
				}
			}
			M4_DeleteSample(&samp);
			if (e) goto exit;
		}
		OD_DeleteDescriptor((Descriptor **) &esd);
	}

exit:
	BIFS_DeleteDecoder(bdec);
	OD_DeleteCodec(oddec);
	if (esd) OD_DeleteDescriptor((Descriptor **) &esd);
	return e;
}


M4StreamContext *M4SM_GetRootBIFS(M4SceneManager *ctx)
{
	M4StreamContext *sc;
	u32 i;
	for (i=0; i<ChainGetCount(ctx->streams); i++) {
		sc = ChainGetEntry(ctx->streams, i);
		if (sc->streamType==M4ST_BIFS) return sc;
	}
	return NULL;
}

M4Err M4SM_ImportSRT(M4SceneManager *ctx, ESDescriptor *src, MuxInfoDescriptor *mux)
{
	M4Err e;
	SFNode *text, *font;
	M4StreamContext *srt;
	FILE *srt_in;
	FieldInfo string, style;
	u32 sh, sm, ss, sms, eh, em, es, ems, start, end;
	M4AUContext *au;
	SGCommand *com;
	SFString *sfstr;
	CommandFieldInfo *inf;
	Bool italic, underlined, bold;
	u32 state, curLine, line, i, len;
	char szLine[2048], szText[2048], *ptr;

	M4StreamContext *sc = M4SM_GetRootBIFS(ctx);
	if (!sc || !sc->scene_context) {
		fprintf(stdout, "Error importing SRT: Cannot locate base scene\n");
		return M4BadParam;
	}
	if (!mux->textNode) {
		fprintf(stdout, "Error importing SRT: Target text node unspecified\n");
		return M4BadParam;
	}
	text = SG_FindNodeByName(sc->scene_context, mux->textNode);
	if (!text) {
		fprintf(stdout, "Error importing SRT: Cannot find target text node %s\n", mux->textNode);
		return M4BadParam;
	}
	if (Node_GetFieldByName(text, "string", &string) != M4OK) {
		fprintf(stdout, "Error importing SRT: Target text node %s doesn't look like text\n", mux->textNode);
		return M4BadParam;
	}

	font = NULL;
	if (mux->fontNode) {
		font = SG_FindNodeByName(sc->scene_context, mux->fontNode);
		if (!font) {
			fprintf(stdout, "Error importing SRT: Cannot find target font node %s\n", mux->fontNode);
			return M4BadParam;
		}
		if (Node_GetFieldByName(font, "style", &style) != M4OK) {
			fprintf(stdout, "Error importing SRT: Target font node %s doesn't look like font\n", mux->fontNode);
			return M4BadParam;
		}
	}

	srt_in = fopen(mux->file_name, "rt");
	if (!srt_in) {
		fprintf(stdout, "Cannot open input SRT %s\n", mux->file_name);
		return M4URLNotFound;
	}

	srt = M4SM_NewStream(ctx, src->ESID, M4ST_BIFS, 1);
	if (!srt) return M4OutOfMem;
	srt->scene_context = sc->scene_context;

	if (!src->slConfig) src->slConfig = (SLConfigDescriptor *) OD_NewDescriptor(SLConfigDescriptor_Tag);
	src->slConfig->timestampResolution = 1000;
	if (!src->decoderConfig) src->decoderConfig = (DecoderConfigDescriptor *) OD_NewDescriptor(DecoderConfigDescriptor_Tag);
	src->decoderConfig->streamType = M4ST_BIFS;
	src->decoderConfig->objectTypeIndication = 1;

	e = M4OK;
	state = end = 0;
	curLine = 0;
	au = NULL;
	com = NULL;
	italic = underlined = bold = 0;
	inf = NULL;

	while (!feof(srt_in)) {
		fgets(szLine, 2048, srt_in);

		while ( (szLine[strlen(szLine)-1]=='\n') || (szLine[strlen(szLine)-1]=='\r') || (szLine[strlen(szLine)-1]=='\t') )
			szLine[strlen(szLine)-1] = 0;

		if (!strlen(szLine)) {
			state = 0;
			if (au) {
				/*if italic or underscore do it*/
				if (font && (italic || underlined || bold)) {
					com = SG_NewCommand(SG_FieldReplace);
					com->node = font;
					Node_Register(font, NULL);
					inf = SG_NewFieldCommand(com);
					inf->fieldIndex = style.allIndex;
					inf->fieldType = style.fieldType;
					sfstr = inf->field_ptr = SG_NewFieldPointer(style.fieldType);
					if (bold && italic && underlined) sfstr->buffer = strdup("BOLDITALIC UNDERLINED");
					else if (italic && underlined) sfstr->buffer = strdup("ITALIC UNDERLINED");
					else if (bold && underlined) sfstr->buffer = strdup("BOLD UNDERLINED");
					else if (underlined) sfstr->buffer = strdup("UNDERLINED");
					else if (bold && italic) sfstr->buffer = strdup("BOLDITALIC");
					else if (bold) sfstr->buffer = strdup("BOLD");
					else sfstr->buffer = strdup("ITALIC");
					ChainAddEntry(au->commands, com);
				}

				au = M4SM_NewAU(srt, end, 0, 1);
				com = SG_NewCommand(SG_FieldReplace);
				com->node = text;
				Node_Register(text, NULL);
				inf = SG_NewFieldCommand(com);
				inf->fieldIndex = string.allIndex;
				inf->fieldType = string.fieldType;
				inf->field_ptr = SG_NewFieldPointer(string.fieldType);
				ChainAddEntry(au->commands, com);
				/*reset font styles so that all AUs are true random access*/
				if (font) {
					com = SG_NewCommand(SG_FieldReplace);
					com->node = font;
					Node_Register(font, NULL);
					inf = SG_NewFieldCommand(com);
					inf->fieldIndex = style.allIndex;
					inf->fieldType = style.fieldType;
					inf->field_ptr = SG_NewFieldPointer(style.fieldType);
					ChainAddEntry(au->commands, com);
				}
				au = NULL;
			}
			inf = NULL;
			continue;
		}

		switch (state) {
		case 0:
			if (sscanf(szLine, "%d", &line) != 1) {
				fprintf(stdout, "Bad SRT format\n");
				e = M4CorruptedData;
				goto exit;
			}
			if (line != curLine + 1) {
				fprintf(stdout, "Error importing SRT frame (previous %d, current %d)\n", curLine, line);
				e = M4CorruptedData;
				goto exit;
			}
			curLine = line;
			state = 1;
			break;
		case 1:
			if (sscanf(szLine, "%d:%d:%d,%d --> %d:%d:%d,%d", &sh, &sm, &ss, &sms, &eh, &em, &es, &ems) != 8) {
				fprintf(stdout, "Error importing SRT frame %d\n", curLine);
				e = M4CorruptedData;
				goto exit;
			}
			start = (3600*sh + 60*sm + ss)*1000 + sms;
			end = (3600*eh + 60*em + es)*1000 + ems;
			/*make stream start at 0 by inserting a fake AU*/
			if ((curLine==1) && start>0) {
				au = M4SM_NewAU(srt, 0, 0, 1);
				com = SG_NewCommand(SG_FieldReplace);
				com->node = text;
				Node_Register(text, NULL);
				inf = SG_NewFieldCommand(com);
				inf->fieldIndex = string.allIndex;
				inf->fieldType = string.fieldType;
				inf->field_ptr = SG_NewFieldPointer(string.fieldType);
				ChainAddEntry(au->commands, com);
			}

			au = M4SM_NewAU(srt, start, 0, 1);
			com = NULL;
			state = 2;			
			italic = underlined = bold = 0;
			break;

		default:
			ptr = szLine;
			/*FIXME - other styles posssibles ??*/
			while (1) {
				if (!strnicmp(ptr, "<i>", 3)) {
					italic = 1;
					ptr += 3;
				}
				else if (!strnicmp(ptr, "<u>", 3)) {
					underlined = 1;
					ptr += 3;
				}
				else if (!strnicmp(ptr, "<b>", 3)) {
					bold = 1;
					ptr += 3;
				}
				else
					break;
			}
			/*if style remove end markers*/
			while ((strlen(ptr)>4) && (ptr[strlen(ptr) - 4] == '<') && (ptr[strlen(ptr) - 1] == '>')) {
				ptr[strlen(ptr) - 4] = 0;
			}

			if (!com) {
				com = SG_NewCommand(SG_FieldReplace);
				com->node = text;
				Node_Register(text, NULL);
				inf = SG_NewFieldCommand(com);
				inf->fieldIndex = string.allIndex;
				inf->fieldType = string.fieldType;
				inf->field_ptr = SG_NewFieldPointer(string.fieldType);
				ChainAddEntry(au->commands, com);
			}
			assert(inf);
			MFField_Append(inf->field_ptr, FT_MFString, (void **) &sfstr);
			len = 0;
			for (i=0; i<strlen(ptr); i++) {
				if (ptr[i] & 0x80) {
					szText[len] = 0xc0 | ( (ptr[i] >> 6) & 0x3 );
					len++;
					ptr[i] &= 0xbf;
				}
				szText[len] = ptr[i];
				len++;
			}
			szText[len] = 0;
			sfstr->buffer = strdup(szText);
			break;
		}
	}

exit:
	if (e) M4SM_RemoveStream(ctx, srt);
	fclose(srt_in);
	return e;
}
