/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2003 
 *					All rights reserved
 *
 *  This file is part of GPAC / ISO Media File Format sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#include <gpac/intern/m4_isomedia_dev.h>

void amr3_del(Atom *s)
{
	AMRSampleEntryAtom *ptr = (AMRSampleEntryAtom *)s;
	if (ptr == NULL) return;
	if (ptr->amr_info) DelAtom((Atom *)ptr->amr_info);

	free(ptr);
}


M4Err amr3_Read(Atom *s, BitStream *bs, u64 *read)
{
	M4Err e;
	u64 sub_read;
	AMRSampleEntryAtom *ptr = (AMRSampleEntryAtom *)s;
	if (ptr == NULL) return M4BadParam;

	ReadAudioSampleEntry((AudioSampleEntryAtom*)s, bs, read);

	e = ParseAtom((Atom **)&ptr->amr_info, bs, &sub_read);
	if (e) return e;
	*read += sub_read;
	if (*read != s->size) return M4ReadAtomFailed;
	return M4OK;
}

Atom *amr3_New()
{
	AMRSampleEntryAtom *tmp = (AMRSampleEntryAtom *) malloc(sizeof(AMRSampleEntryAtom));
	if (tmp == NULL) return NULL;
	memset(tmp, 0, sizeof(AMRSampleEntryAtom));
	InitAudioSampleEntry((AudioSampleEntryAtom*)tmp);
	return (Atom *)tmp;
}

//from here, for write/edit versions
#ifndef M4_READ_ONLY

M4Err amr3_Write(Atom *s, BitStream *bs)
{
	M4Err e;
	AMRSampleEntryAtom *ptr = (AMRSampleEntryAtom *)s;
	e = Atom_Write(s, bs);
	if (e) return e;

	WriteAudioSampleEntry((AudioSampleEntryAtom*)s, bs);
	return WriteAtom((Atom *)ptr->amr_info, bs);
}

M4Err amr3_Size(Atom *s)
{
	M4Err e;
	AMRSampleEntryAtom *ptr = (AMRSampleEntryAtom *)s;
	e = Atom_Size(s);
	if (e) return e;

	SizeAudioSampleEntry((AudioSampleEntryAtom*)s);

	e = SizeAtom((Atom *)ptr->amr_info);
	if (e) return e;
	ptr->size += ptr->amr_info->size;
	return M4OK;
}

#endif //M4_READ_ONLY


void d263_del(Atom *s)
{
	H263ConfigAtom *ptr = (H263ConfigAtom *)s;
	if (ptr == NULL) return;
	free(ptr);
}


M4Err d263_Read(Atom *s, BitStream *bs, u64 *read)
{
	H263ConfigAtom *ptr = (H263ConfigAtom *)s;
	if (ptr == NULL) return M4BadParam;

	ptr->vendor = BS_ReadInt(bs, 32);
	ptr->decoder_version = BS_ReadInt(bs, 8);
	ptr->Level = BS_ReadInt(bs, 8);
	ptr->Profile = BS_ReadInt(bs, 8);

	*read += 7;
	if (*read != ptr->size) return M4ReadAtomFailed;
	return M4OK;
}

Atom *d263_New()
{
	H263ConfigAtom *tmp = (H263ConfigAtom *) malloc(sizeof(H263ConfigAtom));
	if (tmp == NULL) return NULL;
	memset(tmp, 0, sizeof(H263ConfigAtom));
	tmp->type = H263ConfigAtomType;
	return (Atom *)tmp;
}

//from here, for write/edit versions
#ifndef M4_READ_ONLY

M4Err d263_Write(Atom *s, BitStream *bs)
{
	M4Err e;
	H263ConfigAtom *ptr = (H263ConfigAtom *)s;
	e = Atom_Write(s, bs);
	if (e) return e;

	BS_WriteInt(bs, ptr->vendor, 32);
	BS_WriteInt(bs, ptr->decoder_version, 8);
	BS_WriteInt(bs, ptr->Level, 8);
	BS_WriteInt(bs, ptr->Profile, 8);
	return M4OK;
}

M4Err d263_Size(Atom *s)
{
	M4Err e;
	e = Atom_Size(s);
	if (e) return e;
	s->size += 7;
	return M4OK;
}

#endif //M4_READ_ONLY


void damr_del(Atom *s)
{
	AMRConfigAtom *ptr = (AMRConfigAtom *)s;
	if (ptr == NULL) return;
	free(ptr);
}


M4Err damr_Read(Atom *s, BitStream *bs, u64 *read)
{
	AMRConfigAtom *ptr = (AMRConfigAtom *)s;
	if (ptr == NULL) return M4BadParam;

	ptr->vendor = BS_ReadInt(bs, 32);
	ptr->decoder_version = BS_ReadInt(bs, 8);
	ptr->mode_set = BS_ReadInt(bs, 16);
	ptr->mode_change_period = BS_ReadInt(bs, 8);
	ptr->frames_per_sample = BS_ReadInt(bs, 8);
	*read += 9;
	if (*read != ptr->size) return M4ReadAtomFailed;
	return M4OK;
}

Atom *damr_New()
{
	AMRConfigAtom *tmp = (AMRConfigAtom *) malloc(sizeof(AMRConfigAtom));
	if (tmp == NULL) return NULL;
	memset(tmp, 0, sizeof(AMRConfigAtom));
	tmp->type = AMRConfigAtomType;
	return (Atom *)tmp;
}

//from here, for write/edit versions
#ifndef M4_READ_ONLY

M4Err damr_Write(Atom *s, BitStream *bs)
{
	M4Err e;
	AMRConfigAtom *ptr = (AMRConfigAtom *)s;
	e = Atom_Write(s, bs);
	if (e) return e;

	BS_WriteInt(bs, ptr->vendor, 32);
	BS_WriteInt(bs, ptr->decoder_version, 8);
	BS_WriteInt(bs, ptr->mode_set, 16);
	BS_WriteInt(bs, ptr->mode_change_period, 8);
	BS_WriteInt(bs, ptr->frames_per_sample, 8);
	return M4OK;
}

M4Err damr_Size(Atom *s)
{
	M4Err e;
	AMRConfigAtom *ptr = (AMRConfigAtom *)s;
	e = Atom_Size(s);
	if (e) return e;
	ptr->size += 9;
	return M4OK;
}

#endif //M4_READ_ONLY


void h263_del(Atom *s)
{
	H263SampleEntryAtom *ptr = (H263SampleEntryAtom *)s;
	if (ptr == NULL) return;

	if (ptr->h263_config) DelAtom((Atom *)ptr->h263_config);
	free(ptr);
}


M4Err h263_Read(Atom *s, BitStream *bs, u64 *read)
{
	M4Err e;
	u64 sub_read;
	H263SampleEntryAtom *ptr = (H263SampleEntryAtom *)s;
	if (ptr == NULL) return M4BadParam;

	ReadVideoSampleEntry((VisualSampleEntryAtom *)ptr, bs, read);

	e = ParseAtom((Atom **)&ptr->h263_config, bs, &sub_read);
	if (e) return e;
	*read += sub_read;
	if (*read != ptr->size) return M4ReadAtomFailed;
	return M4OK;
}

Atom *h263_New()
{
	H263SampleEntryAtom *tmp = (H263SampleEntryAtom *) malloc(sizeof(H263SampleEntryAtom));
	if (tmp == NULL) return NULL;
	memset(tmp, 0, sizeof(H263SampleEntryAtom));

	InitVideoSampleEntry((VisualSampleEntryAtom *)tmp);

	tmp->type = H263SampleEntryAtomType;
	return (Atom *)tmp;
}


//from here, for write/edit versions
#ifndef M4_READ_ONLY

M4Err h263_Write(Atom *s, BitStream *bs)
{
	M4Err e;
	H263SampleEntryAtom *ptr = (H263SampleEntryAtom *)s;
	e = Atom_Write(s, bs);
	if (e) return e;

	WriteVideoSampleEntry((VisualSampleEntryAtom *)s, bs);

	e = WriteAtom((Atom *)ptr->h263_config, bs);
	if (e) return e;
	return M4OK;
}

M4Err h263_Size(Atom *s)
{
	M4Err e;
	H263SampleEntryAtom *ptr = (H263SampleEntryAtom *)s;
	e = Atom_Size(s);
	if (e) return e;

	SizeVideoSampleEntry((VisualSampleEntryAtom *)s);

	e = SizeAtom((Atom *)ptr->h263_config);
	if (e) return e;
	ptr->size += ptr->h263_config->size;
	return M4OK;
}

#endif //M4_READ_ONLY
