/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2004
 *					All rights reserved
 *
 *  This file is part of GPAC / Osmo4 wxWidgets GUI
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 *		
 */


#include "wxOsmo4.h"
#include "wxGPACControl.h"
#include "fileprops.h"
#include "wx/image.h"

IMPLEMENT_APP(wxOsmo4App)

#include "osmo4.xpm"
#include <wx/dnd.h>
#include <wx/filename.h>


#include "osmo4_play.xpm"
#include "osmo4_pause.xpm"
#include "osmo4_stop.xpm"

wxString get_pref_browser(LPINIFILE cfg)
{
	char *sOpt = IF_GetKey(cfg, "General", "Browser");
	if (sOpt) return sOpt;
#ifdef __WXMAC__
	return "safari";
#else
#ifdef WIN32
	return "explorer.exe";
#else
	return "mozilla";
#endif
#endif
}


IMPLEMENT_DYNAMIC_CLASS(wxGPACEvent, wxEvent )

wxGPACEvent::wxGPACEvent(wxWindow* win)
{
    SetEventType(GPAC_EVENT);
    SetEventObject(win);
	m4evt.type = 0;
	to_url = "";
}
wxEvent *wxGPACEvent::Clone() const
{
	wxGPACEvent *evt = new wxGPACEvent((wxWindow *) m_eventObject);
	evt->to_url = to_url;
	evt->m4evt = m4evt;
	return evt;
}


Bool GPAC_EventProc(void *ptr, M4Event *evt)
{
	wxCommandEvent event;
	wxOsmo4Frame *app = (wxOsmo4Frame *)ptr;

	switch (evt->type) {
	case M4E_DURATION:
		app->m_duration = (u32) (evt->duration.duration*1000);
		break;
	case M4E_MESSAGE:
	{
		Bool main_service = 0;
		if (!evt->message.service && !strcmp(evt->message.service, app->the_url.c_str())) main_service = 1;
		if (!evt->message.message) return 0;

		if (evt->message.error) app->SetStatus(wxString::Format("%s %s", evt->message.message, M4ErrToString(evt->message.error)) );
		else if (!app->m_console_off) {
			if (strstr(evt->message.message, "100 %")) {
				app->SetStatus("");
			} else {
				app->SetStatus(wxString::Format("%s", evt->message.message) );
			}
		}

		/*log*/
		if (evt->message.error) 
			::wxLogMessage("%s %s (%s)", evt->message.message, M4ErrToString(evt->message.error), main_service ? evt->message.service : "main service");
		else
			::wxLogMessage("%s (%s)", evt->message.message, main_service ? evt->message.service : "main service");
	}
		break;
	
	case M4E_VKEYDOWN:
		if (!(evt->key.key_states & M4KM_ALT)) return 0;
		if (app->m_duration>=2000) {
			s32 res;
			switch (evt->key.m4_vk_code) {
			case M4VK_LEFT:
				res = M4T_GetCurrentTimeInMS(app->m_term) - 5*app->m_duration/100;
				if (res<0) res=0;
				app->CheckResume();
				M4T_PlayFromTime(app->m_term, res);
				break;
			case M4VK_RIGHT:
				res = M4T_GetCurrentTimeInMS(app->m_term) + 5*app->m_duration/100;
				if ((u32) res>=app->m_duration) res = 0;
				app->CheckResume();
				M4T_PlayFromTime(app->m_term, res);
				break;
			case M4VK_DOWN:
				res = M4T_GetCurrentTimeInMS(app->m_term) - 60000;
				if (res<0) res=0;
				app->CheckResume();
				M4T_PlayFromTime(app->m_term, res);
				break;
			case M4VK_UP:
				res = M4T_GetCurrentTimeInMS(app->m_term) + 60000;
				if ((u32) res>=app->m_duration) res = 0;
				app->CheckResume();
				M4T_PlayFromTime(app->m_term, res);
				break;
			}
		}
		break;

	/*we use CTRL and not ALT for keys, since windows shortcuts keypressed with ALT*/
	case M4E_KEYDOWN:
	{
		wxGPACEvent wxevt(app);
		wxevt.m4evt = *evt;
		app->AddPendingEvent(wxevt);
	}
		break;

	case M4E_CONNECT:
	{
		wxGPACEvent wxevt(app);
		wxevt.m4evt.type = M4E_CONNECT;
		wxevt.m4evt.connect.is_connected = evt->connect.is_connected;
		if (!evt->connect.is_connected) app->m_duration = 0;
		app->AddPendingEvent(wxevt);
	}
		break;
	case M4E_QUIT:
	{
		wxGPACEvent wxevt(app);
		wxevt.m4evt.type = M4E_QUIT;
		app->AddPendingEvent(wxevt);
	}
		break;

	case M4E_NAVIGATE:
	{
		wxGPACEvent wxevt(app);
		wxevt.to_url = evt->navigate.to_url;
		wxevt.m4evt.type = evt->type;
		app->AddPendingEvent(wxevt);
	}
		return 1;
	}
	return 0;
}



bool wxOsmo4App::OnInit()
{
	wxFrame *frame = new wxOsmo4Frame();
	frame->Show(TRUE);
	SetTopWindow(frame);
	return true;	
}


class myDropfiles : public wxFileDropTarget
{
public:
	myDropfiles() : wxFileDropTarget() {}
	virtual bool OnDropFiles(wxCoord x, wxCoord y, const wxArrayString& filenames);
	wxOsmo4Frame *m_pMain;
};

bool myDropfiles::OnDropFiles(wxCoord x, wxCoord y, const wxArrayString& filenames)
{
	if (filenames.GetCount()>1) return FALSE;
	m_pMain->OpenFile(filenames.Item(0));
	return TRUE;
}

bool GPACLogs::OnFrameClose(wxFrame *frame)
{
	Show(FALSE);
	return 0;
}

Bool wxOsmo4Frame::LoadTerminal()
{
	m_term = NULL;
	memset(&m_client, 0, sizeof(M4User));

	/*locate exec dir for cfg file*/
    wxPathList pathList;
	wxString currentDir(wxGetCwd());
    wxString abs_gpac_path = "";
	const char *gpac_cfg;

	::wxLogMessage("Looking for GPAC configuration file");

#if defined(__WXMAC__) && !defined(__DARWIN__)
    // On Mac, the current directory is the relevant one when the application starts.
    abs_gpac_path = wxGetCwd();
	gpac_cfg = "GPAC.cfg";
#else

#ifdef WIN32
	wxOsmo4App &app = wxGetApp();
	gpac_cfg = "GPAC.cfg";
	/*locate exe*/
    if (wxIsAbsolutePath(app.argv[0])) {
        abs_gpac_path = wxPathOnly(app.argv[0]);
		m_pExePath = abs_gpac_path;
	} else {
		if (currentDir.Last() != wxFILE_SEP_PATH) currentDir += wxFILE_SEP_PATH;
		abs_gpac_path = currentDir + app.argv[0];
		if (wxFileExists(abs_gpac_path)) {
			abs_gpac_path = wxPathOnly(abs_gpac_path);
			m_pExePath = abs_gpac_path;
		} else {
			abs_gpac_path = "";
			pathList.AddEnvList(wxT("PATH"));
			abs_gpac_path = pathList.FindAbsoluteValidPath(app.argv[0]);
			if (!abs_gpac_path.IsEmpty()) {
				abs_gpac_path = wxPathOnly(abs_gpac_path);
				m_pExePath = abs_gpac_path;
			} else {
				/*ask user*/
				wxDirDialog dlg(NULL, "Locate GPAC config file directory");
				if ( dlg.ShowModal() != wxID_OK ) return 0;
				abs_gpac_path = dlg.GetPath();
			}
		}
	}
#else
	gpac_cfg = ".gpacrc";
	char *cfg_dir = getenv("HOME");
	if (cfg_dir) {
		abs_gpac_path = cfg_dir;
	} else {
		/*ask user*/
		wxDirDialog dlg(NULL, "Locate GPAC config file directory");
		if ( dlg.ShowModal() != wxID_OK ) return 0;
		abs_gpac_path = dlg.GetPath();
	}

#endif

#endif

	/*load config*/
	m_client.config = NewIniFile(abs_gpac_path.c_str(), gpac_cfg);

	if (!m_client.config) {
		unsigned char config_file[MAX_FILE_PATH];
		strcpy((char *) config_file, (const char *) abs_gpac_path.c_str());
		if (config_file[strlen((char *) config_file)-1] != M4_PATH_SEPARATOR) {
		  char szSep[2];
		  szSep[0] = M4_PATH_SEPARATOR;
		  szSep[1] = 0;
		  strcat((char *) config_file, (const char *)szSep);
		}
		strcat((char *) config_file, gpac_cfg);
		FILE *ft = fopen((const char *) config_file, "wt");
		if (!ft) {
			wxMessageDialog(NULL, "Cannot create blank config file", "Init error", wxOK).ShowModal();
			return 0;
		}
		fclose(ft);
		m_client.config = NewIniFile(abs_gpac_path.c_str(), gpac_cfg);
		if (!m_client.config) {
			wxMessageDialog(NULL, "Cannot open GPAC configuration file", "Init error", wxOK);
			return 0;
		}
	}
	::wxLogMessage("GPAC configuration file opened - looking for plugins");
	/*get plugins*/
	const char *str = IF_GetKey(m_client.config, "General", "PluginsDirectory");
	Bool first_launch = 0;
	if (!str) {
	  first_launch = 1;
#ifdef M4_PLUGIN_PATH
		str = M4_PLUGIN_PATH;
#else
		str = abs_gpac_path.c_str();
#endif
	}
	m_client.plugins = NewPluginManager((const unsigned char *) str, m_client.config);

	/*initial launch*/
	if (first_launch || !PM_GetPluginsCount(m_client.plugins)) {
		char *sOpt;
		wxDirDialog dlg(NULL, "Locate GPAC plugins directory");
		if  (!PM_GetPluginsCount(m_client.plugins)) {
		  PM_Delete(m_client.plugins);
		  m_client.plugins = NULL;
			if ( dlg.ShowModal() != wxID_OK ) return false;
			str = dlg.GetPath().c_str();;
	
			m_client.plugins = NewPluginManager((const unsigned char *) str, m_client.config);
			if (!m_client.plugins || !PM_GetPluginsCount(m_client.plugins) ) {
				wxMessageDialog(NULL, "Cannot find any plugins for GPAC", "Init error", wxOK);
				IF_Delete(m_client.config);
				return 0;
			}		
		}

		IF_SetKey(m_client.config, "General", "PluginsDirectory", (const char *) str);

		/*setup UDP traffic autodetect*/
		IF_SetKey(m_client.config, "Network", "AutoReconfigUDP", "yes");
		IF_SetKey(m_client.config, "Network", "UDPNotAvailable", "no");
		IF_SetKey(m_client.config, "Network", "UDPTimeout", "10000");
		IF_SetKey(m_client.config, "Network", "BufferLength", "3000");

		/*check audio config on windows, force config*/
		sOpt = IF_GetKey(m_client.config, "Audio", "ForceConfig");
		if (!sOpt) {
			IF_SetKey(m_client.config, "Audio", "ForceConfig", "yes");
			IF_SetKey(m_client.config, "Audio", "NumBuffers", "8");
			IF_SetKey(m_client.config, "Audio", "BuffersPerSecond", "16");
		}

#ifdef WIN32
		sOpt = IF_GetKey(m_client.config, "Render2D", "GraphicsDriver");
		if (!sOpt) IF_SetKey(m_client.config, "Render2D", "GraphicsDriver", "gdip_rend");
		sOpt = IF_GetKey(m_client.config, "Downloader", "CacheDirectory");
		if (!sOpt) {
			unsigned char str_path[MAX_PATH];
			sprintf((char *) str_path, "%scache", abs_gpac_path.c_str());
			IF_SetKey(m_client.config, "Downloader", "CacheDirectory", (const char *) str_path);
		}
		/*by default use GDIplus, much faster than freetype on font loading*/
		IF_SetKey(m_client.config, "FontEngine", "DriverName", "gdip_rend");
		IF_SetKey(m_client.config, "Video", "DriverName", "DirectX Video Output");
#else
		wxDirDialog dlg3(NULL, "Please specify a cache directory for GPAC");
		dlg3.SetPath("/tmp");
		if ( dlg3.ShowModal() == wxID_OK ) 
			IF_SetKey(m_client.config, "Downloader", "CacheDirectory", (const char *) dlg3.GetPath().c_str() );

		wxDirDialog dlg2(NULL, "Please locate a TrueType font repository on your system for text support");
		dlg2.SetPath("/usr/share/fonts/truetype");
		if ( dlg2.ShowModal() == wxID_OK ) 
			IF_SetKey(m_client.config, "FontEngine", "FontDirectory", (const char *) dlg2.GetPath().c_str() );

		IF_SetKey(m_client.config, "Video", "DriverName", "SDL Video Output");
#endif
	}	
	if (! PM_GetPluginsCount(m_client.plugins) ) {
		wxMessageDialog(NULL, "No plugins available - system cannot work", "Fatal Error", wxOK).ShowModal();
		PM_Delete(m_client.plugins);
		IF_Delete(m_client.config);
		return 0;
	}

	::wxLogMessage("%d plugins found:", PM_GetPluginsCount(m_client.plugins));
	for (u32 i=0; i<PM_GetPluginsCount(m_client.plugins); i++) {
		::wxLogMessage("\t%s", PM_GetFileName(m_client.plugins, i));
	}

	::wxLogMessage("Starting MPEG-4 Terminal");
	/*now load terminal*/
	m_client.opaque = this;
	m_client.EventProc = GPAC_EventProc;
	m_term = NewMPEG4Term(&m_client);
	if (!m_term) {
		wxMessageDialog(NULL, "Fatal Error", "Cannot load MPEG-4 Terminal", wxOK).ShowModal();
		return 0;
	} else {
		::wxLogMessage("MPEG-4 Terminal started - using %s", IF_GetKey(m_client.config, "Rendering", "RendererName"));
	}
	return 1;
}


void wxOsmo4Frame::CheckResume()
{
	if (m_paused) M4T_Pause(m_term, 0);
	m_paused = 0;
}

wxOsmo4Frame::wxOsmo4Frame():
    wxFrame((wxFrame *) NULL, -1, "Osmo4 - GPAC", wxPoint(-1, -1), wxSize(275, 95), 
		wxDEFAULT_FRAME_STYLE & ~(wxMAXIMIZE_BOX | wxRESIZE_BORDER))
{
	int ws[3];
	SetIcon(wxIcon(osmo4));


	myDropfiles *droptarget = new myDropfiles();
	droptarget->m_pMain = this;
	SetDropTarget(droptarget);
	m_pLogs = new GPACLogs(this);
	m_bGrabbed = 0;

	/*new menu bar*/
	wxMenuBar *b = new wxMenuBar();
	/*file*/
	wxMenu *menu = new wxMenu();
	menu->Append(FILE_OPEN, "&Open File\tCtrl+O", "Open local presentation");
	menu->Append(FILE_OPEN_URL, "&Open URL\tCtrl+U", "Open remote presentation");
	menu->Append(FILE_RELOAD, "&Reload File\tCtrl+R", "Reload presentation");
	menu->AppendSeparator();
	menu->Append(FILE_PLAY, "&Play/Pause\tCtrl+P", "Play/Pause presentation");
	menu->Append(FILE_STEP, "&Frame Step\tCtrl+S", "Step one frame into presentation");
	menu->AppendSeparator();
	menu->Append(FILE_PROPERTIES, "&Properties\tCtrl+I", "Show presentation properties");
	menu->Enable(FILE_PROPERTIES, 0);
	menu->AppendSeparator();
	menu->Append(FILE_QUIT, "E&xit", "Quit the application");
	b->Append(menu, "&File");
	/*view*/
	menu = new wxMenu();
	menu->Append(VIEW_LOGS, "&Logs", "View GPAC logs");
	menu->AppendSeparator();
	menu->Append(VIEW_FULLSCREEN, "&Fullscreen", "Toggles Fullscreen",wxITEM_CHECK);
	menu->Append(VIEW_ORIGINAL, "&Original Size", "Restore original size");
	wxMenu *smenu = new wxMenu();
	smenu->Append(VIEW_AR_KEEP, "Keep Original\tCtrl+1", "Keep original aspect ratio", wxITEM_CHECK);
	smenu->Append(VIEW_AR_FILL, "Fill Screen\tCtrl+2", "Stretch presentation to fill screen", wxITEM_CHECK);
	smenu->Append(VIEW_AR_43, "Ratio 4/3\tCtrl+3", "Force aspect ratio to 4/3", wxITEM_CHECK);
	smenu->Append(VIEW_AR_169, "Ratio 16/9\tCtrl+4", "Force aspect ratio to 16/9", wxITEM_CHECK);
	menu->Append(0, "&Aspect Ratio", smenu);
	smenu->Check(VIEW_AR_KEEP, 1);
	menu->AppendSeparator();
	menu->Append(VIEW_OPTIONS, "&Options", "View Options");
	b->Append(menu, "&View");

	menu = new wxMenu();
	menu->Append(APP_SHORTCUTS, "&Shortcuts", "Show keyboard shortcuts");
	menu->Append(APP_ABOUT, "&About", "Display information and copyright");
	b->Append(menu, "&?");

	SetMenuBar(b);

    m_pStatusbar = CreateStatusBar(1, 0, -1, "statusBar");
	ws[0] = 50;
	ws[1] = 60;
	ws[2] = -1;
	m_pStatusbar->SetFieldsCount(3, ws);

	SetStatusBarPane(2);

	m_pTimer = new wxTimer();
	m_pTimer->SetOwner(this, ID_CTRL_TIMER);
	m_bGrabbed = 0;

	m_pPlayBMP = new wxBitmap(osmo4_play);
	m_pPauseBMP = new wxBitmap(osmo4_pause);
	m_pStopBMP = new wxBitmap(osmo4_stop);
	m_pPlay = new wxBitmapButton(this, ID_CTRL_PLAY, *m_pPlayBMP , wxPoint(2 ,0), wxSize(20, 30), wxNO_3D);
	m_pPlay->SetBackgroundColour(wxColour("LIGHT GREY"));
	m_pPause = new wxBitmapButton(this, ID_CTRL_PAUSE, *m_pPauseBMP , wxPoint(2 ,0), wxSize(20, 30), wxNO_3D);
	m_pPause->SetBackgroundColour(wxColour("LIGHT GREY"));

	m_pStop = new wxBitmapButton(this, ID_CTRL_STOP, *m_pStopBMP , wxPoint(24 ,0), wxSize(20, 30), wxNO_3D);
	m_pStop->SetBackgroundColour(wxColour("LIGHT GREY"));
	m_pProg = new wxSlider(this, ID_CTRL_SLIDE, 0, 0, 1000, wxPoint(46, 0), wxSize(220, 30), wxSL_HORIZONTAL | wxSUNKEN_BORDER);

	m_pPause->Hide();
	m_pStop->Hide();
	m_pPlay->Hide();
	m_pProg->Hide();


	SetBackgroundColour(wxColour("LIGHT GREY"));

	m_connected = 0;
	if (!LoadTerminal()) {
		Close(TRUE);
		return;
	}

	char *sOpt = IF_GetKey(m_client.config, "General", "ConsoleOff");
	m_console_off = (sOpt && !stricmp(sOpt, "yes")) ? 1 : 0;
	sOpt = IF_GetKey(m_client.config, "General", "Loop");
	m_loop = (sOpt && !stricmp(sOpt, "yes")) ? 1 : 0;
	sOpt = IF_GetKey(m_client.config, "General", "StopAtEnd");
	m_stop_at_end = (sOpt && !stricmp(sOpt, "yes")) ? 1 : 0;
	
	
	Raise();
	m_pStatusbar->SetStatusText("Ready", 2);
	m_LastStatusTime = 0;

	wxOsmo4App &app = wxGetApp();
	if (app.argc>1) OpenFile(wxString(app.argv[1]));

}

wxOsmo4Frame::~wxOsmo4Frame()
{
	if (m_term) M4T_Delete(m_term);
	if (m_client.plugins) PM_Delete(m_client.plugins);
	if (m_client.config) IF_Delete(m_client.config);

	delete m_pTimer;
	delete m_pPlayBMP;
	delete m_pPauseBMP;
	delete m_pStopBMP;
}


BEGIN_EVENT_TABLE(wxOsmo4Frame, wxFrame)
	EVT_MENU(FILE_OPEN, wxOsmo4Frame::OnFileOpen)
	EVT_MENU(FILE_OPEN_URL, wxOsmo4Frame::OnFileOpenURL)
	EVT_MENU(FILE_RELOAD, wxOsmo4Frame::OnFileReload)
	EVT_MENU(FILE_PLAY, wxOsmo4Frame::OnFilePlay)
	EVT_MENU(FILE_STEP, wxOsmo4Frame::OnFileStep)
	EVT_MENU(FILE_PROPERTIES, wxOsmo4Frame::OnFileProperties)
	EVT_MENU(FILE_QUIT, wxOsmo4Frame::OnFileQuit)
	EVT_MENU(VIEW_FULLSCREEN, wxOsmo4Frame::OnFullScreen)
	EVT_MENU(VIEW_OPTIONS, wxOsmo4Frame::OnOptions)
	EVT_MENU(VIEW_AR_KEEP, wxOsmo4Frame::OnViewARKeep)
	EVT_MENU(VIEW_AR_FILL, wxOsmo4Frame::OnViewARFill)
	EVT_MENU(VIEW_AR_169, wxOsmo4Frame::OnViewAR169)
	EVT_MENU(VIEW_AR_43, wxOsmo4Frame::OnViewAR43)
	EVT_MENU(VIEW_ORIGINAL, wxOsmo4Frame::OnViewOriginal)

	EVT_MENU(APP_SHORTCUTS, wxOsmo4Frame::OnShortcuts)
	EVT_MENU(APP_ABOUT, wxOsmo4Frame::OnAbout)
	EVT_GPACEVENT(wxOsmo4Frame::OnGPACEvent)
	EVT_TIMER(ID_CTRL_TIMER, wxOsmo4Frame::OnTimer)
    EVT_BUTTON(ID_CTRL_PLAY, wxOsmo4Frame::OnStart)
    EVT_BUTTON(ID_CTRL_PAUSE, wxOsmo4Frame::OnPause)
    EVT_BUTTON(ID_CTRL_STOP, wxOsmo4Frame::OnStop)
	EVT_COMMAND_SCROLL(ID_CTRL_SLIDE, wxOsmo4Frame::OnSlide)
	EVT_MENU(VIEW_LOGS, wxOsmo4Frame::OnLogs)

	EVT_UPDATE_UI(FILE_PROPERTIES, wxOsmo4Frame::OnUpdateNeedsConnect)  
	EVT_UPDATE_UI(FILE_RELOAD, wxOsmo4Frame::OnUpdateNeedsConnect)  
	EVT_UPDATE_UI(FILE_PLAY, wxOsmo4Frame::OnUpdateNeedsConnect)  
	EVT_UPDATE_UI(FILE_STEP, wxOsmo4Frame::OnUpdateNeedsConnect)  
	EVT_UPDATE_UI(VIEW_ORIGINAL, wxOsmo4Frame::OnUpdateNeedsConnect)  
	EVT_UPDATE_UI(VIEW_FULLSCREEN, wxOsmo4Frame::OnUpdateFullScreen)  
	EVT_UPDATE_UI(VIEW_AR_KEEP, wxOsmo4Frame::OnUpdateAR)
	EVT_UPDATE_UI(VIEW_AR_FILL, wxOsmo4Frame::OnUpdateAR)
	EVT_UPDATE_UI(VIEW_AR_169, wxOsmo4Frame::OnUpdateAR)
	EVT_UPDATE_UI(VIEW_AR_43, wxOsmo4Frame::OnUpdateAR)
END_EVENT_TABLE()

/*open file dlg*/
class OpenFileDlg : public wxDialog {
public:
    OpenFileDlg(wxWindow *parent);
	wxString m_urlVal;
private:
    wxButton *m_go;
    wxComboBox *m_url;
	wxOsmo4Frame *m_app;
	void OnGo(wxCommandEvent& event);
    DECLARE_EVENT_TABLE()
};

BEGIN_EVENT_TABLE(OpenFileDlg, wxDialog)
    EVT_BUTTON(ID_URL_GO, OpenFileDlg::OnGo)
END_EVENT_TABLE()

OpenFileDlg::OpenFileDlg(wxWindow *parent)
             : wxDialog(parent, -1, wxString(_T("Enter remote presentation location")))
{
	SetSize(410, 50);
	Centre();
    m_url = new wxComboBox(this, -1, _T(""), wxDefaultPosition, wxDefaultSize, 0, NULL, wxCB_DROPDOWN);
	m_url->SetSize(0, 2, 360, 18, wxSIZE_AUTO);
    m_go = new wxButton(this, ID_URL_GO, _T("Go !"));
	m_go->SetSize(364, 2, 30, 18, wxSIZE_AUTO);
	m_urlVal = "";

	m_app = (wxOsmo4Frame *)parent;

	char *sOpt;
	char filename[1024];
	u32 i=0;

	while (1) {
		sprintf(filename, "last_file_%d", i);
		sOpt = IF_GetKey(m_app->m_client.config, "General", filename);
		if (!sOpt) break;
		m_url->Append(sOpt);
		i++;
	}
}

void OpenFileDlg::OnGo(wxCommandEvent& event)
{
	m_urlVal = m_url->GetValue();
	char *sOpt;
	char filename[1024];
	u32 i=0;

	while (1) {
		sprintf(filename, "last_file_%d", i);
		sOpt = IF_GetKey(m_app->m_client.config, "General", filename);
		if (!sOpt) break;
		if (!strcmp(sOpt, m_urlVal.c_str())) {
			EndModal(wxID_OK);
			return;
		}
		i++;
	}
	/*add it*/
	if (i<10) {
		IF_SetKey(m_app->m_client.config, "General", filename, m_urlVal.c_str());
	} else {
		IF_SetKey(m_app->m_client.config, "General", "last_file_10", m_urlVal.c_str());
	}
	EndModal(wxID_OK);
}
/*end open file dlg*/

void wxOsmo4Frame::OnFileOpen(wxCommandEvent & WXUNUSED(event))
{
	wxFileDialog dlg(this);
	u32 keyCount, i;
	wxString sFiles;

	sFiles = "All Files (*.*)|*.*|";
	keyCount = IF_GetKeyCount(m_client.config, "FileAssociations");
	for (i=0; i<keyCount; i++) {
		const char *sKey;
		sKey = IF_GetKeyName(m_client.config, "FileAssociations", i);
		if (!sKey) continue;
		wxString sOpt = IF_GetKey(m_client.config, "FileAssociations", sKey);
		while (1) {
			wxString ext = sOpt.BeforeFirst(' ');
			if (sFiles.Find(ext.c_str()) < 0) 
				sFiles += wxString::Format("%s Files (*.%s)|*.%s|", ext.c_str(), ext.c_str(), ext.c_str());

			if (sOpt==ext) break;
			wxString rem = wxString::Format("%s ", ext.c_str());;
			sOpt.Replace(rem.c_str(), "", TRUE);
		}
	}
	dlg.SetWildcard(sFiles);
	if (dlg.ShowModal()==wxID_OK) OpenFile(dlg.GetPath());
}

void wxOsmo4Frame::OnFileOpenURL(wxCommandEvent & WXUNUSED(event))
{
	OpenFileDlg dlg(this);
	if (dlg.ShowModal()==wxID_OK) OpenFile(dlg.m_urlVal);	
}

void wxOsmo4Frame::OnFileProperties(wxCommandEvent & WXUNUSED(event))
{
	wxFileProps dlg(this);
	dlg.SetIcon(wxIcon(osmo4));
	dlg.ShowModal();
}

void wxOsmo4Frame::OnFileReload(wxCommandEvent & WXUNUSED(event))
{
	OpenFile(the_url);
}

void wxOsmo4Frame::OnFileQuit(wxCommandEvent & WXUNUSED(event))
{
	Close(FALSE);
}

void wxOsmo4Frame::OnViewOriginal(wxCommandEvent & WXUNUSED(event))
{
	M4T_SetOption(m_term, M4O_OriginalView, 1);
}

void wxOsmo4Frame::OnOptions(wxCommandEvent & WXUNUSED(event))
{
	wxGPACControl dlg(this);
	dlg.SetIcon(wxIcon(osmo4));
	dlg.ShowModal();
}

void wxOsmo4Frame::OpenFile(wxString target)
{
	CheckResume();
	if (m_connected) M4T_CloseURL(m_term);
	m_connected = 0;
	the_url = target;
	M4T_ConnectURL(m_term, the_url.c_str());
}

void wxOsmo4Frame::OnLogs(wxCommandEvent & WXUNUSED(event))
{
	m_pLogs->Show();
}

void wxOsmo4Frame::OnUpdateNeedsConnect(wxUpdateUIEvent &event)
{
	event.Enable(m_connected ? 1 : 0);
}

void wxOsmo4Frame::OnUpdateFullScreen(wxUpdateUIEvent &event)
{
	if (m_connected) {
		event.Enable(1);
		event.Check(M4T_GetOption(m_term, M4O_Fullscreen) ? 1 : 0);
	} else {
		event.Enable(0);
	}
}

void wxOsmo4Frame::OnFullScreen(wxCommandEvent & WXUNUSED(event))
{
	Bool isFS = M4T_GetOption(m_term, M4O_Fullscreen) ? 1 : 0;
	M4T_SetOption(m_term, M4O_Fullscreen, isFS ? 0 : 1);
}

void wxOsmo4Frame::OnViewARKeep(wxCommandEvent & WXUNUSED(event))
{
	M4T_SetOption(m_term, M4O_AspectRatio, M4_AR_Keep);
}
void wxOsmo4Frame::OnViewARFill(wxCommandEvent & WXUNUSED(event))
{
	M4T_SetOption(m_term, M4O_AspectRatio, M4_AR_None);
}
void wxOsmo4Frame::OnViewAR169(wxCommandEvent & WXUNUSED(event))
{
	M4T_SetOption(m_term, M4O_AspectRatio, M4_AR_16_9);
}
void wxOsmo4Frame::OnViewAR43(wxCommandEvent & WXUNUSED(event))
{
	M4T_SetOption(m_term, M4O_AspectRatio, M4_AR_4_3);
}

void wxOsmo4Frame::OnUpdateAR(wxUpdateUIEvent &event)
{
	if (!m_connected) {
		event.Enable(0);
		return;
	}
	event.Enable(1);
	u32 val = M4T_GetOption(m_term, M4O_AspectRatio);
	if ((event.GetId() == VIEW_AR_FILL) && (val==M4_AR_None))
		event.Check(1);
	else if ((event.GetId() == VIEW_AR_KEEP) && (val==M4_AR_Keep)) 
		event.Check(1);
	else if ((event.GetId() == VIEW_AR_169) && (val==M4_AR_16_9)) 
		event.Check(1);
	else if ((event.GetId() == VIEW_AR_43) && (val==M4_AR_4_3)) 
		event.Check(1);
	else event.Check(0);
}

void wxOsmo4Frame::OnShortcuts(wxCommandEvent & WXUNUSED(event))
{
	wxMessageDialog dlg(this,
		"Shortcuts with focus on main frame:\n"
		"Open File: Ctrl + O\n"
		"Show File Information: Ctrl + I\n"
		"Reload File: Ctrl + R\n"
		"Pause/Resume File: Ctrl + P\n"
		"Step by Step: Ctrl + S\n"
		"Fullscreen On/Off: Alt + Return\n"
		"Aspect Ratio Normal: Ctrl + 1\n"
		"Aspect Ratio Fill: Ctrl + 2\n"
		"Aspect Ratio 4/3: Ctrl + 3\n"
		"Aspect Ratio 16/9: Ctrl + 4\n"
		"\n"		
		"Shortcuts with focus on video frame:\n"
		"Seek +5% into presentation: Alt + right arrow\n"		
		"Seek -5% into presentation: Alt + left arrow\n"		
		"Seek +1min into presentation: Alt + up arrow\n"
		"Seek -1min into presentation: Alt + down arrow\n"
		
		, "Shortcuts Available on Osmo4 - GPAC"
		, wxOK | wxICON_INFORMATION);

	dlg.SetIcon(wxIcon(osmo4));
	dlg.ShowModal();
}

/*open file dlg*/
class AboutDlg : public wxDialog {
public:
    AboutDlg(wxWindow *parent);

private:
    wxStaticText *m_info;
    wxButton *m_close;
	void OnClose(wxCommandEvent& event);
    DECLARE_EVENT_TABLE()
};

BEGIN_EVENT_TABLE(AboutDlg, wxDialog)
    EVT_BUTTON(ID_ABOUT_CLOSE, AboutDlg::OnClose)
END_EVENT_TABLE()

AboutDlg::AboutDlg(wxWindow *parent)
             : wxDialog(parent, -1, wxString(_T("About Osmo4 - GPAC")))
{
	SetSize(340, 220);
	Centre();
	m_info = new wxStaticText(this, -1, "http://gpac.sourceforge.net", wxPoint(2, 2), wxSize(330, 165), wxALIGN_CENTRE | wxST_NO_AUTORESIZE );
    m_close = new wxButton(this, ID_ABOUT_CLOSE, _T("Close"), wxPoint(2, 170), wxSize(330, 20));

	SetIcon(wxIcon(osmo4));
	m_info->SetLabel(
		"GPAC MPEG-4 SDK\n"
		"This program is free software and may be distributed\naccording to the terms of the GNU General Public License"
		"\n\n"
		"GPAC (C) 2000 - 2004 by Jean Le Feuvre\nAll Rights Reserved"
		"\nhttp://gpac.sourceforge.net"
		"\n\n"
		"ECMAScript support through Mozilla SpiderMonkey\nhttp://www.mozilla.org/js\n"
		);
}
void AboutDlg::OnClose(wxCommandEvent& WXUNUSED(event))
{
	Close(FALSE);
}

void wxOsmo4Frame::OnAbout(wxCommandEvent & WXUNUSED(event))
{
	AboutDlg dlg(this);
	dlg.ShowModal();
}

void wxOsmo4Frame::OnGPACEvent(wxGPACEvent &event)
{
	wxCommandEvent evt;

	switch (event.m4evt.type) {
	case M4E_NAVIGATE:
	{
		wxString ext = wxFileName(event.to_url).GetExt();
		ext.LowerCase();

		if ((ext==wxString("mp4")) || (ext==wxString("avi")) || (ext==wxString("mp3")) || (ext==wxString("mpeg"))
			|| (ext==wxString("mpg")) || (ext==wxString("3gp")) || (ext==wxString("jpg")) || (ext==wxString("jpeg"))
			|| (ext==wxString("png")) 
			) {
			char *str = URL_Concatenate(the_url, event.to_url);
			if (str) {
				OpenFile(str);
				free(str);
			}
		} else {
			wxString cmd = get_pref_browser(m_client.config);
			cmd += " ";
			cmd += event.to_url;
			wxExecute(cmd);
		}
	}
		break;
	case M4E_QUIT:
		Close(TRUE);
		break;
	case M4E_CONNECT:
		Connect(event.m4evt.connect.is_connected);
		break;
	case M4E_KEYDOWN:
		if (!(event.m4evt.key.key_states & M4KM_CTRL)) return;
		switch (event.m4evt.key.virtual_code) {
		case 'R':
		case 'r':
			M4T_SetOption(m_term, M4O_ForceRedraw, 1);
			break;
		case 'P':
		case 'p':
			OnFilePlay(evt);
			break;
		case 's':
		case 'S':
			OnFileStep(evt);
			break;
		}
	}
}


static wxString format_time(u32 duration, u32 timescale)
{
	u32 h, m, s;
	Float time = duration;
	time /= timescale;
	time *= 1000;
	h = (u32) (time / 1000 / 3600);
	m = (u32) (time / 1000 / 60 - h*60);
	s = (u32) (time / 1000 - h*3600 - m*60);
	return wxString::Format("%02d:%02d:%02d", h, m, s);
}

void wxOsmo4Frame::SetStatus(wxString str)
{
	m_pStatusbar->SetStatusText(str, 2); 
	m_LastStatusTime = M4_GetSysClock();
}

void wxOsmo4Frame::OnTimer(wxTimerEvent& WXUNUSED(event))
{

	u32 now;
	if (m_LastStatusTime) {
		now = M4_GetSysClock();
		if (now > 1000+m_LastStatusTime) {
			m_LastStatusTime = 0;
			m_pStatusbar->SetStatusText("Ready", 2); 
		}
	}

	now = M4T_GetCurrentTimeInMS(m_term);
	if (!now) return;

	if (m_duration && !m_bGrabbed) {
		wxString str;
		if ((now >= m_duration + 500) && M4T_GetOption(m_term, M4O_IsOver)) {
			if (m_loop) {
				M4T_PlayFromTime(m_term, 0);
			} else if (m_stop_at_end && !m_paused) {
				Stop();
			}
		} else {
			Double val = now * 1000;
			val /= m_duration;
			m_pProg->SetValue((val<=1000) ? (u32) val : 1000);

			if (0) {
				str = format_time(m_duration-now, 1000);
			} else {
				str = format_time(now, 1000);
			}
			m_pStatusbar->SetStatusText(str);
			str = wxString::Format("FPS %.2f", M4T_GetCurrentFPS(m_term, 0));
			m_pStatusbar->SetStatusText(str, 1);
		}
	}
}

void wxOsmo4Frame::Connect(Bool bOn) 
{
	if (bOn) {
		m_pTimer->Start(100, 0);
		m_pPlay->Hide();
		m_pPause->Show();
		m_connected = 1;
		m_paused = 0;
		m_bToReset = 0;
		m_pStop->Show();
		m_pProg->Show();
	} else {
		if (!m_connected) {
			m_pTimer->Stop();
			m_pPlay->Show();
			m_pPause->Hide();
			m_pStop->Hide();
			m_pProg->Hide();
		}
	}
}

void wxOsmo4Frame::OnStop(wxCommandEvent &WXUNUSED(event))
{
	Stop();
}
void wxOsmo4Frame::OnPause(wxCommandEvent &WXUNUSED(event))
{
	if (m_bToReset) return;
	/*paused, do step*/
	if (m_paused) {
		/*step*/
		M4T_Pause(m_term, 2);
		m_pTimer->Start(100, 1);
	} else {
		M4T_Pause(m_term, 1);
		m_paused = 1;
		m_pTimer->Stop();
		m_pPlay->Show();
		m_pPause->Hide();

	}
}
void wxOsmo4Frame::OnStart(wxCommandEvent &WXUNUSED(event))
{
	if (!m_paused) return;
	if (m_bToReset) M4T_PlayFromTime(m_term, 0);
	M4T_Pause(m_term, 0);
	m_bToReset = m_paused = 0;
	m_pTimer->Start(100, 0);
	m_pPlay->Hide();
	m_pPause->Show();
}

void wxOsmo4Frame::Stop()
{
	if (!m_paused) {
		M4T_Pause(m_term, 1);
		m_paused = 1;
	}
	m_bToReset = 1;
	m_pTimer->Stop();
	m_pProg->SetValue(0);
	m_pPlay->Show();
	m_pPause->Hide();
}

void wxOsmo4Frame::OnSlide(wxScrollEvent &event)
{
	if (!m_duration) return;
	s32 type = event.GetEventType();
	if (type== wxEVT_SCROLL_THUMBTRACK) {
	  m_bGrabbed = 1;
	  Double now = (Double) event.GetPosition();
	  now /= 1000;
	  now *= m_duration;
	  wxString str = format_time((u32) (now), 1000);
	  m_pStatusbar->SetStatusText(str);
	  }
	else if (m_bGrabbed){
	  m_bGrabbed = 0;
	  Double res = (Double) m_pProg->GetValue();
	  res /= 1000;
	  res *= m_duration;
	  if (m_paused) {
		M4T_Pause(m_term, 0);
		m_bToReset = m_paused = 0;
		m_pTimer->Start(100, 0);
	  }
	  M4T_PlayFromTime(m_term, (u32) res);
	}
}

void wxOsmo4Frame::OnRelease(wxScrollEvent &WXUNUSED(event))
{
	m_bGrabbed = 0;
	Double res = (Double) m_pProg->GetValue();
	res /= 1000;
	res *= m_duration;

	if (m_paused) {
		M4T_Pause(m_term, 0);
		m_bToReset = m_paused = 0;
		m_pTimer->Start(100, 0);
	}
	M4T_PlayFromTime(m_term, (u32) res);
}

void wxOsmo4Frame::ReloadTerminal()
{
	Bool reconnect = m_connected;
	::wxLogMessage("Reloading MPEG-4 Terminal");

	M4T_Delete(m_term);
	m_term = NewMPEG4Term(&m_client);
	if (!m_term) {
		wxMessageDialog(this, "Fatal Error !!", "Couldn't change renderer", wxOK);
		Destroy();
		return;
	}
	if (reconnect) M4T_ConnectURL(m_term, the_url.c_str());
	::wxLogMessage("MPEG-4 Terminal reloaded");
}


void wxOsmo4Frame::OnFilePlay(wxCommandEvent & WXUNUSED(event))
{
	wxCommandEvent evt;
	if (m_paused) OnStart(evt);
	else OnPause(evt);
}

void wxOsmo4Frame::OnFileStep(wxCommandEvent & WXUNUSED(event))
{
	wxCommandEvent evt;
	if (!m_paused) OnPause(evt);
	else M4T_Pause(m_term, 2);
}
