/*
 * DirectoryStorage.java
 *
 * Created on December 19, 2003, 11:39 PM
 */

package com.sun.enterprise.config.backup.pluggable.impl;

import java.io.*;
import java.util.*;

import com.sun.enterprise.util.io.FileUtils;
import com.sun.enterprise.config.backup.pluggable.BackupStorage;
import com.sun.enterprise.config.backup.utils.FactoryHelper;
import com.sun.enterprise.config.backup.utils.LoggerHelper;
import com.sun.enterprise.config.backup.BackupException;
import com.sun.enterprise.config.backup.DefaultConstants;
import com.sun.enterprise.config.backup.status.Status;

/**
 *
 * @author  bnevins
 */
abstract class GenericStorage implements BackupStorage
{
	abstract long copyFiles(File fromDir, File toDir) throws BackupException;
	
	///////////////////////////////////////////////////////////////////////////
	
	public long backup(String[] srcDirs, String targetString) throws BackupException
	{
		File target = new File(targetString);
		
		// make sure there ARE source dirs
		if(srcDirs == null || srcDirs.length == 0)
			throw new BackupException("No source directories specified", "");
		
		// make sure they point to bonafide directories
		for (int i=0;i<srcDirs.length;i++)
			if(!FileUtils.safeIsDirectory(srcDirs[i]))
				throw new BackupException("A specified source directory is NOT a directory: " + srcDirs[i], "");
		
		// create the target directory if it doesn't exist
		if(!FileUtils.safeIsDirectory(target))
			target.mkdirs();

		// Now make sure the target directory exists
		if(!FileUtils.safeIsDirectory(target))
			throw new BackupException("The target directory is NOT a directory: " + target.getPath(), "");

		if(!target.canWrite())
			throw new BackupException("The target directory is read-only: " + target.getPath(), "");

		int totalSizeBackedUp = 0;
		for (int i=0;i<srcDirs.length;i++)
		{
			// Note: say srcDirs[i] is C:/domains/domain1 and target is C:/backups/backup12345678
			// then what we really want to do is this:
			// mkdir C:/backups/backup12345678/domain1
			// xcopy C:/domains/domain1/*.*  C:/backups/backup12345678/domain1 /s

			try
			{
				File srcFile = new File(srcDirs[i]);
				File currentTarget = new File(target, srcFile.getName());
				totalSizeBackedUp += copyFiles(srcFile, currentTarget);
			}
			catch(Exception ioe)
			{
				throw new BackupException("backup error", "Exception copying "
				+ srcDirs[i] + " to " + target.getPath(), ioe);
			}
		}
		return totalSizeBackedUp;
	}

	///////////////////////////////////////////////////////////////////////////
	
	public long restore(File file) throws BackupException
	{
		if(file == null || !file.exists() || !file.isDirectory())
			return -1;
		
		File[] childrenFiles = getBUSubdirs(file);
		
		String[] targetDirs = FactoryHelper.getEnv().getDirectoriesToBackup();
		
		if(targetDirs == null || targetDirs.length == 0)
			return -1;
		
		if(targetDirs.length == 1 && childrenFiles.length == 1)
			return restoreSingle(childrenFiles[0], new File(targetDirs[0]));
		else if(targetDirs.length == 1)
			return restoreMultiToSingle(childrenFiles, new File(targetDirs[0]));
		else
			return restoreMultiToMulti(childrenFiles, targetDirs);
	}

	///////////////////////////////////////////////////////////////////////////
	
	private String getTargetDirToRestore(String[] targetDirs, File child)
	{
		for(int i = 0; i < targetDirs.length;i++)
		{
			File f = new File(targetDirs[i]);
			if(f.getName().equals(child.getName()))
			{
				return f.getParent();
			}
		}
		LoggerHelper.warning("no_target_directory_to_restore " + child);
		return null;
	}

	///////////////////////////////////////////////////////////////////////////
	
	public void createStatusFile(String target, Status status)
	{
		if (status == null) return;
		
		File f = new File(target, FactoryHelper.getEnv().getStatusInfoFileName());
		FileOutputStream fos = null;
		try
		{
			fos = new FileOutputStream(f);
			fos.write(status.toString().getBytes());
		} catch(Exception e)
		{
			//ignore
		} finally
		{
			try
			{
				fos.close();
			} catch(Exception ex)
			{}
		}
	}

	///////////////////////////////////////////////////////////////////////////
	
	public void deleteTarget()
	{
		String[] targetDirs = FactoryHelper.getEnv().getDirectoriesToBackup();
		
		for(int i = 0; i < targetDirs.length; i++)
		{
			FileUtils.whack(new File(targetDirs[i]));
		}
	}

	///////////////////////////////////////////////////////////////////////////
	
	private File[] pruneNonDirs(File[] files)
	{
		// bnevins -- prune out regular files (like the status.txt file)
		if(files == null)
			return new File[0];
		
		ArrayList list = new ArrayList(files.length);
		
		for(int i = 0; i < files.length; i++)
			if(files[i].isDirectory())
				list.add(files[i]);
		
		File[] ret = new File[list.size()];
		
		if(list.size() == 0)
			return ret;
		
		return (File[])(list.toArray(ret));
	}

	///////////////////////////////////////////////////////////////////////////
	
	private long restoreSingle(File from, File to) throws BackupException
	{
		if(from.getName().equals(to.getName()))
		{
			to = to.getParentFile();
		}
		
		try
		{
			FileUtils.copyTree(from, to);
		}
		catch(IOException ioe)
		{
			throw new BackupException("xx", "IOException copying tree -- from: " + from + ", To: " + to, ioe);
		}
		
		return 100;
	}

	///////////////////////////////////////////////////////////////////////////
	
	private long restoreMultiToSingle(File[] from, File to) throws BackupException
	{
		// by definition -- put all the bu dirs under one dir
		try
		{
			for(int i = 0; i < from.length; i++)
			{
				FileUtils.copyTree(from[i], to);
			}
			
			return 100;
		}
		catch(IOException ioe)
		{
			throw new BackupException("xx", "IOException copying tree -- from: " + from + ", To: " + to, ioe);
		}
	}

	///////////////////////////////////////////////////////////////////////////
	
	private long restoreMultiToMulti(File[] from, String[] toStrings) throws BackupException
	{
		// They all have to match their last directory name or
		// it's an error...bnevins
		File[] to = new File[toStrings.length];
		
		for(int i = 0; i < toStrings.length; i++)
			to[i] = new File(toStrings[i]);
		
		if(to.length != from.length)
			throw new BackupException("restoreMultiToMulti", "Number of backup-dirs (" + from.length + " doesn't match number of target dirs (" + to.length + ")");
		
		int numMatches = 0;
		
		// inefficient to do this twice -- but very time-consuming to do neatly...
		for(int i = 0; i < to.length; i++)
		{
			String toName = to[i].getName();
			
			for(int j = 0; j < from.length; j++)
			{
				if(toName.equals(from[j].getName()))
				{
					++numMatches;
					break;
				}
			}
		}
		
		if(numMatches != to.length)
			throw new BackupException("restoreMultiToMulti", "backup-dirs don't match up with target dirs (" + to.length + ")");
		
		try
		{
			for(int i = 0; i < to.length; i++)
			{
				String toName = to[i].getName();
				
				for(int j = 0; j < from.length; j++)
				{
					if(toName.equals(from[j].getName()))
					{
						FileUtils.copyTree(from[j], to[i].getParentFile());
						break;
					}
				}
			}
			return 100;
		}
		catch(IOException ioe)
		{
			throw new BackupException("xxx", "IOException copying tree -- from: " + from + ", To: " + to, ioe);
		}
	}

	///////////////////////////////////////////////////////////////////////////
	
	private File[] getBUSubdirs(File root) throws BackupException
	{
		File[] files = root.listFiles();
		files = pruneNonDirs(files);
		
		if(files.length == 0)
			throw new BackupException("getBUSubDirs", "Nothing in backup dir: " + root);
		
		return files;
	}
}
