/*
 * MaxBackupMonitor.java
 *
 * Created on December 21, 2003, 8:29 PM
 */

package com.sun.enterprise.config.backup;

import java.io.IOException;
import java.io.File;
import java.util.Arrays;
import java.util.Collections;
import java.util.ArrayList;
import java.util.List;
import java.io.FilenameFilter;
import com.sun.enterprise.config.backup.utils.LoggerHelper;
import com.sun.enterprise.config.backup.utils.LocalStringsHelper;
import com.sun.enterprise.config.backup.pluggable.EnvironmentFactory;
import com.sun.enterprise.config.backup.pluggable.BackupEnvironment;
import com.sun.enterprise.config.backup.utils.FactoryHelper;
import com.sun.enterprise.config.backup.phase.RemovePhases;


/**
 * Anything extraneous in the monitored directory will
 * not be touched
 * @author  sridatta
 */
public abstract class StorageMonitor {
    
    /** Creates a new instance of MaxBackupMonitor */
    public StorageMonitor() {
    }
    
    /**
     * checks if number of backups exceed max.
     *
     * Does not throw any exception. Best effort solution.
     * Not synchronized. Hence, the number of backups
     * can potentially be greater once in a while
     * if so, deletes the oldest ones
     */
    public void run() {
        try {
            File[] f = getFiles();
            int num = getNumberOfBackups(f);
            int max = getMaxBackups();
            
            if(num <= max) {
                LoggerHelper.fine("no_backup_file_to_delete");
                return; //nothing to do
            }
            
            deleteBackups(f, num-max);
        } catch(Exception e) {
            //log and ignore
            LoggerHelper.info("error_monitoring_max_backups", e);
        }
    }
    
    public File getLatestFile() throws BackupException {
        File[] f = getFiles();
         if(f == null) {
            LoggerHelper.fine("no_backup_file_found");
            return null;
        }
        Arrays.sort(f, Collections.reverseOrder());
        return f[0];
    }
    
    protected File[] getFiles() throws BackupException {
        String dir = getMonitoredDirectory();
        File f = new File(dir);
        FilenameFilter filter = getFileNameFilter();
        return f.listFiles(filter);
    }
    
    private int getNumberOfBackups(File[] f) {
        if (f == null) return 0;
        
        return f.length;
    }
    
    
    private void deleteBackups(File[] f, int number) 
                     throws IOException, BackupException {
        
        if(f == null || number == 0) {
            LoggerHelper.fine("no_backup_file_to_delete");
            return;
        }
        
        /*File[] actualList = getActualList(f);
        if(actualList == null) {
            LoggerHelper.fine("no_backup_file_to_delete");
            return;
        }
         */
        Arrays.sort(f);
        deleteFiles(f, number);
    }
   
    /**
     * return -1 if incorrect file
     */
    /*
    private File[] getActualList(File[] f) 
                        throws BackupInitException {

        ArrayList arr = new ArrayList();
        
        for(int i = 0; i < f.length; i++) {
            String name = f[i].getName();
            String[] parts =
            name.split(DefaultConstants.INTRA_FILENAME_SEPARATOR);
            
            if (!getFileNamePrefix().equals(parts[0]))
                continue;
            
            try {
                new Long(parts[1]).longValue();
                arr.add(f[i]);
            } catch(Exception e) {
                continue;
            }
        }
        return convertToFileArr(arr);
    }
        
     */
    private File[] convertToFileArr(ArrayList arr) {
        if(arr==null || arr.size() == 0) return null;
        File[] res = new File[arr.size()];
        for(int i=0;i<arr.size();i++) {
            res[i] = (File) arr.get(i);
        }
        return res;
    }
    
    private void deleteFiles(File[] delete, int number) {
        if(delete == null) return;
        int max;
        
        if(number > delete.length) 
            max = delete.length;
        else 
            max = number;
        
        for(int i=0;i<max;i++) {
            try {
                RemovePhases rp = new RemovePhases();
                rp.deleteBackup(delete[i], true);
            } catch(Exception e) {
                LoggerHelper.info("error_deleting_file" + delete[i].getName(), e);
            }
        }
    }
    
    protected BackupEnvironment getEnv() {
        return getFactory().getBackupEnvironment();
    }
    
    protected EnvironmentFactory getFactory() {
        try {
            return EnvironmentFactory.getEnvironmentFactory();
        } catch(Exception e) {
            throw new BackupAbortException(
                "error_getting_environment_factory",
                LocalStringsHelper.getString("error_getting_environment_factory"),
                e);
        }
    }
    
    protected abstract FilenameFilter getFileNameFilter();
    protected abstract String getMonitoredDirectory();
    protected abstract int getMaxBackups();
    protected abstract String getFileNamePrefix();
}
