/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the "License").  You may not use this file except 
 * in compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt or 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html. 
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * HEADER in each file and include the License file at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt.  If applicable, 
 * add the following below this CDDL HEADER, with the 
 * fields enclosed by brackets "[]" replaced with your 
 * own identifying information: Portions Copyright [yyyy] 
 * [name of copyright owner]
 */
// Copyright (c) 1998, 2006, Oracle. All rights reserved.  
package oracle.toplink.essentials.internal.helper;

import java.security.AccessController;
import java.security.PrivilegedActionException;

import oracle.toplink.essentials.Version;
import oracle.toplink.essentials.exceptions.ValidationException;
import oracle.toplink.essentials.internal.security.PrivilegedAccessHelper;
import oracle.toplink.essentials.internal.security.PrivilegedNewInstanceFromClass;
import oracle.toplink.essentials.internal.security.PrivilegedClassForName;

/**
 *  INTERNAL:
 *  JavaPlatform abstracts the version of the JDK we are using.  It allows any operation
 *  which is dependant on JDK version to be called from a single place and then delegates
 *  the call to its JDKPlatform
 *  @see JDPlatform
 *  @author Tom Ware
 */
public class JavaPlatform {
    private static JDKPlatform platform = null;

    // 3 possible states are required for conforming like
    public static final int FALSE = 0;
    public static final int TRUE = 1;
    public static final int UNDEFINED = 2;

    /**
     *  INTERNAL:
     *  Get the version of JDK being used from the Version class.
     *  @return JDKPlatform a platform appropriate for the version of JDK being used.
     */
    private static JDKPlatform getPlatform() {
        if (platform == null) {
            if (Version.isJDK15()) {
                try {
                    Class platformClass = null;
                    // use class.forName() to avoid loading the JDK 1.5 class unless it is needed.
                    if (PrivilegedAccessHelper.shouldUsePrivilegedAccess()){
                        try {
                            platformClass = (Class)AccessController.doPrivileged(new PrivilegedClassForName("oracle.toplink.essentials.internal.helper.JDK15Platform"));
                        } catch (PrivilegedActionException exception) {
                        }
                    } else {
                        platformClass = oracle.toplink.essentials.internal.security.PrivilegedAccessHelper.getClassForName("oracle.toplink.essentials.internal.helper.JDK15Platform");
                    }                  
                    if (PrivilegedAccessHelper.shouldUsePrivilegedAccess()){
                        try {
                            platform = (JDKPlatform)AccessController.doPrivileged(new PrivilegedNewInstanceFromClass(platformClass));
                        } catch (PrivilegedActionException exception) {
                        }
                    } else {
                        platform = (JDKPlatform)PrivilegedAccessHelper.newInstanceFromClass(platformClass);
                    }      
                } catch (Exception exception) {
                }
            }
            if (platform == null) {
                platform = new JDK15Platform();
            }
        }
        return platform;
    }

    /**
     *  INTERNAL:
     *  Conform an expression which uses the operator "like" for an in-memory query
     *  @return int
     *  int FALSE = 0 - if the expression does not conform
     *  int TRUE = 1 - if the expression does conform
     *  int UNDEFINED = 2 - if it cannot be determined if the expression conforms
     */
    public static int conformLike(Object left, Object right) {
        return getPlatform().conformLike(left, right);
    }

    /**
     * INTERNAL:
     * Get the milliseconds from a Calendar.
     * @param calendar the instance of calendar to get the millis from
     * @return long the number of millis
     */
    public static long getTimeInMillis(java.util.Calendar calendar) {
        return getPlatform().getTimeInMillis(calendar);
    }

    /**
     * INTERNAL:
     * Get the Map to store the query cache in
     */
    public static java.util.Map getQueryCacheMap() {
        return getPlatform().getQueryCacheMap();
    }

    /**
     * INTERNAL:
     * Set the milliseconds for a Calendar.
     */
    public static void setTimeInMillis(java.util.Calendar calendar, long millis) {
        getPlatform().setTimeInMillis(calendar, millis);
    }

    /**
     *  INTERNAL:
     *  Set the cause of an exception.  This is useful for JDK 1.4 exception chaining
     *  @param java.lang.Throwable the exception to set the cause for
     *  @param java.lang.Throwable the cause of this exception
     */
    public static void setExceptionCause(Throwable exception, Throwable cause) {
        getPlatform().setExceptionCause(exception, cause);
    }

    /**
     * INTERNAL
     * return a boolean which determines where TopLink should include the TopLink-stored
     * Internal exception in it's stack trace.  For JDK 1.4 VMs with exception chaining
     * the Internal exception can be redundant and confusing.
     * @return boolean
     */
    public static boolean shouldPrintInternalException() {
        return getPlatform().shouldPrintInternalException();
    }
}
