/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the "License").  You may not use this file except 
 * in compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt or 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html. 
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * HEADER in each file and include the License file at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt.  If applicable, 
 * add the following below this CDDL HEADER, with the 
 * fields enclosed by brackets "[]" replaced with your 
 * own identifying information: Portions Copyright [yyyy] 
 * [name of copyright owner]
 */
// Copyright (c) 1998, 2006, Oracle. All rights reserved.  
package oracle.toplink.essentials.internal.ejb.cmp3.metadata.accessors;

import java.lang.reflect.AnnotatedElement;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.persistence.AssociationOverride;
import javax.persistence.AssociationOverrides;
import javax.persistence.AttributeOverride;
import javax.persistence.AttributeOverrides;
import javax.persistence.DiscriminatorColumn;
import javax.persistence.DiscriminatorValue;
import javax.persistence.Entity;
import javax.persistence.EntityListeners;
import javax.persistence.ExcludeDefaultListeners;
import javax.persistence.ExcludeSuperclassListeners;
import javax.persistence.IdClass;
import javax.persistence.Inheritance;
import javax.persistence.InheritanceType;
import javax.persistence.JoinColumn;
import javax.persistence.MappedSuperclass;
import javax.persistence.NamedNativeQueries;
import javax.persistence.NamedNativeQuery;
import javax.persistence.NamedQueries;
import javax.persistence.NamedQuery;
import javax.persistence.PostLoad;
import javax.persistence.PostPersist;
import javax.persistence.PostRemove;
import javax.persistence.PostUpdate;
import javax.persistence.PrePersist;
import javax.persistence.PreRemove;
import javax.persistence.PreUpdate;
import javax.persistence.SecondaryTable;
import javax.persistence.SecondaryTables;
import javax.persistence.SqlResultSetMapping;
import javax.persistence.SqlResultSetMappings;
import javax.persistence.Table;
import javax.persistence.Transient;

import oracle.toplink.essentials.descriptors.ClassDescriptor;

// WIP shouldn't be here
import oracle.toplink.essentials.exceptions.ValidationException;

import oracle.toplink.essentials.internal.ejb.cmp3.metadata.accessors.MetadataAccessor;

import oracle.toplink.essentials.internal.ejb.cmp3.metadata.accessors.objects.MetadataAccessibleObject;
import oracle.toplink.essentials.internal.ejb.cmp3.metadata.accessors.objects.MetadataClass;
import oracle.toplink.essentials.internal.ejb.cmp3.metadata.accessors.objects.MetadataField;
import oracle.toplink.essentials.internal.ejb.cmp3.metadata.accessors.objects.MetadataMethod;

import oracle.toplink.essentials.internal.ejb.cmp3.metadata.columns.MetadataColumn;
import oracle.toplink.essentials.internal.ejb.cmp3.metadata.columns.MetadataDiscriminatorColumn;
import oracle.toplink.essentials.internal.ejb.cmp3.metadata.columns.MetadataJoinColumn;
import oracle.toplink.essentials.internal.ejb.cmp3.metadata.columns.MetadataJoinColumns;
import oracle.toplink.essentials.internal.ejb.cmp3.metadata.columns.MetadataPrimaryKeyJoinColumn;
import oracle.toplink.essentials.internal.ejb.cmp3.metadata.columns.MetadataPrimaryKeyJoinColumns;

import oracle.toplink.essentials.internal.ejb.cmp3.metadata.listeners.MetadataEntityClassListener;
import oracle.toplink.essentials.internal.ejb.cmp3.metadata.listeners.MetadataEntityListener;

import oracle.toplink.essentials.internal.ejb.cmp3.metadata.MetadataHelper;
import oracle.toplink.essentials.internal.ejb.cmp3.metadata.MetadataConstants;
import oracle.toplink.essentials.internal.ejb.cmp3.metadata.MetadataProcessor;
import oracle.toplink.essentials.internal.ejb.cmp3.metadata.MetadataDescriptor;

import oracle.toplink.essentials.internal.ejb.cmp3.metadata.queries.MetadataEntityResult;
import oracle.toplink.essentials.internal.ejb.cmp3.metadata.queries.MetadataFieldResult;
import oracle.toplink.essentials.internal.ejb.cmp3.metadata.queries.MetadataNamedNativeQuery;
import oracle.toplink.essentials.internal.ejb.cmp3.metadata.queries.MetadataNamedQuery;
import oracle.toplink.essentials.internal.ejb.cmp3.metadata.queries.MetadataSQLResultSetMapping;

import oracle.toplink.essentials.internal.ejb.cmp3.metadata.tables.MetadataSecondaryTable;
import oracle.toplink.essentials.internal.ejb.cmp3.metadata.tables.MetadataTable;

import oracle.toplink.essentials.internal.ejb.cmp3.xml.accessors.XMLMappedSuperclassAccessor;
import oracle.toplink.essentials.internal.ejb.cmp3.xml.listeners.XMLEntityListener;
import oracle.toplink.essentials.internal.ejb.cmp3.xml.XMLConstants;
import oracle.toplink.essentials.internal.ejb.cmp3.xml.XMLHelper;

import oracle.toplink.essentials.internal.helper.DatabaseField;
import oracle.toplink.essentials.internal.helper.DatabaseTable;
import oracle.toplink.essentials.internal.helper.Helper;

import oracle.toplink.essentials.queryframework.ColumnResult;
import oracle.toplink.essentials.queryframework.EntityResult;
import oracle.toplink.essentials.queryframework.FieldResult;

import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * A class accessor.
 * 
 * @author Guy Pelletier
 * @since TopLink EJB 3.0 Reference Implementation
 */
public class ClassAccessor extends NonRelationshipAccessor {
    protected Class m_inheritanceRootClass;
    protected Boolean m_isInheritanceSubclass;
    protected List<ClassAccessor> m_mappedSuperclasses;
    
    /**
     * INTERNAL:
     */
    public ClassAccessor(MetadataAccessibleObject accessibleObject, MetadataProcessor processor, MetadataDescriptor descriptor) {
        super(accessibleObject, processor, descriptor);
    }
    
    /**
     * INTERNAL:
	 * Add multiple fields to the descriptor. Called from either @Inheritance 
     * or @SecondaryTable context.
	 */
    protected void addMultipleTableKeyFields(MetadataPrimaryKeyJoinColumns primaryKeyJoinColumns, String PK_CTX, String FK_CTX) {
        // ProcessPrimaryKeyJoinColumns will validate the primary key join
        // columns passed in and will return a list of 
        // MetadataPrimaryKeyJoinColumn.
        for (MetadataPrimaryKeyJoinColumn primaryKeyJoinColumn : processPrimaryKeyJoinColumns(primaryKeyJoinColumns)) {
            // In an inheritance case this call will return the pk field on the
            // root class of the inheritance hierarchy. Otherwise in a secondary
            // table case it's the primary key field name off our own descriptor.
            String defaultPKFieldName = m_descriptor.getPrimaryKeyFieldName();

            DatabaseField pkField = primaryKeyJoinColumn.getPrimaryKeyField();
            pkField.setName(getName(pkField, defaultPKFieldName, PK_CTX));

            DatabaseField fkField = primaryKeyJoinColumn.getForeignKeyField();
            fkField.setName(getName(fkField, pkField.getName(), FK_CTX));
        
            if (fkField.getName().equals(pkField.getName())) {
                // Add a multiple table primary key to the descriptor.
                m_descriptor.addMultipleTablePrimaryKeyField(pkField, fkField);
            } else {
                // Add a multiple table foreign key to the descriptor.
                m_descriptor.addMultipleTableForeignKeyField(pkField, fkField);
            }
        }
    }
    
    /**
     * INTERNAL:
     * Create and return the appropriate accessor based on the accessible 
     * object given.
     */
    protected MetadataAccessor buildAccessor(MetadataAccessibleObject accessibleObject) {
        MetadataAccessor accessor = m_descriptor.getAccessorFor(accessibleObject.getAttributeName());
        
        if (accessor == null) {
            if (MetadataHelper.isEmbeddedId(accessibleObject, m_descriptor)) {
                return new EmbeddedAccessor(accessibleObject, this, true);
            } else if (MetadataHelper.isEmbedded(accessibleObject, m_descriptor)) {
                return new EmbeddedAccessor(accessibleObject, this, false);
            } else if (MetadataHelper.isManyToMany(accessibleObject, m_descriptor)) {
                return new ManyToManyAccessor(accessibleObject, this);
            } else if (MetadataHelper.isManyToOne(accessibleObject, m_descriptor)) {
                return new ManyToOneAccessor(accessibleObject, this);
            } else if (MetadataHelper.isOneToMany(accessibleObject, m_logger, m_descriptor)) {
                return new OneToManyAccessor(accessibleObject, this);
            } else if (MetadataHelper.isOneToOne(accessibleObject, m_project, m_logger, m_descriptor)) {
                return new OneToOneAccessor(accessibleObject, this);
            } else {
                // Default case (everything else currently falls into this)
                return new BasicAccessor(accessibleObject, this);
            }
        } else {
            return accessor;
        }
    }
    
    /**
     * INTERNAL:
     * Clear the mapped supeclasses. Do this after the class loader has changed.
     * The list of mapped superclasses is lazily initialized.
     */
    public void clearMappedSuperclasses() {
        m_mappedSuperclasses = null;     
    }
    
    /**
     * INTERNAL: (Overridden in XMLClassAccessor)
     * Return the discriminator value for this accessor.
     */
    public String getDiscriminatorValue() {
        DiscriminatorValue discriminatorValue = getAnnotation(DiscriminatorValue.class);
        
        if (discriminatorValue == null) {
            return null;
        } else {
            return discriminatorValue.value();
        }
    }
    
    /**
     * INTERNAL: (Overridden in XMLClassAccessor)
     * Return the name of this entity class.
     */
    public String getEntityName() {
        Entity entity = getAnnotation(Entity.class);
        return (entity == null) ? "" : entity.name();
    }
    
    /**
     * INTERNAL: (Overridden in XMLClassAccessor)
     */
    protected Class getIdClass() {
    	IdClass idClass = getAnnotation(IdClass.class);
        return (idClass == null) ? null : idClass.value();
    }
    
    /**
     * INTERNAL:
     * You should call isInheritanceSubclass before calling this method,
     * otherwise you risk extra processing trying to figure out a root class.
     * That is, the class traversal will be done everytime you call this method
     * on a class that is actually not an inheritance subclass.
     */
	public Class getInheritanceRootClass() {
        if (m_inheritanceRootClass == null) {
            Class lastParent = null;
            Class parent = getJavaClass().getSuperclass();
        
            while (parent != Object.class) {
                if (hasInheritanceTag(parent) || m_project.containsDescriptor(parent)) {
                    lastParent = parent;
                }
                
                parent = parent.getSuperclass();
            }
        
            // Finally set whatever we found as the inheritance root class. 
            // Which may be null.
            m_inheritanceRootClass = lastParent;
        }
  
        return m_inheritanceRootClass;
    }
    
    /**
     * INTERNAL: 
     * Store the descriptor metadata for the root of our inheritance hierarchy.
     */
    public MetadataDescriptor getInheritanceRootDescriptor() {
        return m_project.getDescriptor(getInheritanceRootClass());
    }
    
    /**
     * INTERNAL: (Overridden in XMLClassAccessor)
     * Return the inheritance strategy. This method should only be called
     * on the root of the inheritance hierarchy.
     */
    protected String getInheritanceStrategy() {
        Inheritance inheritance = getAnnotation(Inheritance.class);
        
        if (inheritance == null) {
            return "";
        } else {
            return inheritance.strategy().name();   
        }
    }
    
    /**
     * INTERNAL: (OVERRIDE)
     * Return the java class that defines this accessor. It may be an
     * entity, embeddable or mapped superclass.
     */
    public Class getJavaClass() {
        return (Class) getAnnotatedElement();
    }
    
    /**
     * INTERNAL:
     * Build a list of classes that are decorated with a @MappedSuperclass.
     */
	public List<ClassAccessor> getMappedSuperclasses() {
        if (m_mappedSuperclasses == null) {
            m_mappedSuperclasses = new ArrayList<ClassAccessor>();
            
            Class parent = getJavaClass().getSuperclass();
        
            while (parent != Object.class) {
                if (m_project.hasMappedSuperclass(parent)) {
                    Node node = m_project.getMappedSuperclassNode(parent);
                    XMLHelper helper = m_project.getMappedSuperclassHelper(parent);                    
                    m_mappedSuperclasses.add(new XMLMappedSuperclassAccessor(new MetadataClass(parent), node, helper, m_processor, m_descriptor));
                } else if (isAnnotationPresent(MappedSuperclass.class, parent)) {
                    m_mappedSuperclasses.add(new MappedSuperclassAccessor(new MetadataClass(parent), m_processor, m_descriptor));
                }
                
                parent = parent.getSuperclass();
            }
        }
        
        return m_mappedSuperclasses;
    }
    
    /**
     * INTERNAL: (Overridden in XMLClassAccessor)
     */
    protected boolean hasEntityTag(Class cls) {
        return isAnnotationPresent(Entity.class, cls);
    }
    
    /**
     * INTERNAL: (Overridden in XMLClassAccessor)
     * Return true if this class has an @Inheritance.
     */
    protected boolean hasInheritanceTag(Class entityClass) {
        return isAnnotationPresent(Inheritance.class, entityClass);
    }
    
    /**
     * INTERNAL:
     */
     public boolean isClass() {
     	return true;
     }
     
    /**
     * INTERNAL:
     */
    public boolean isInheritanceSubclass() {
        if (m_isInheritanceSubclass == null) {
            m_isInheritanceSubclass = new Boolean(getInheritanceRootClass() != null);
        }
        
        return m_isInheritanceSubclass;
    }
     
     /**
     * INTERNAL:
     */
    protected boolean isTransient(AnnotatedElement annotatedElement, int modifier) {
        if (isAnnotationPresent(Transient.class, annotatedElement)) {
            if (MetadataHelper.getDeclaredAnnotationsCount(annotatedElement, m_descriptor) > 1) {
                m_validator.throwMappingAnnotationsAppliedToTransientAttribute(annotatedElement);
            }
            
            return true;
        } else if (Modifier.isTransient(modifier)) {
            if (MetadataHelper.getDeclaredAnnotationsCount(annotatedElement, m_descriptor) > 0) {
                m_validator.throwMappingAnnotationsAppliedToTransientAttribute(annotatedElement);
            }
            
            return true;
        }
        
        return false;
    }

    /**
     * INTERNAL:
     * Return true is this annotated element is not marked transient, static or 
     * abstract.
     */
    protected boolean isValidPersistenceElement(AnnotatedElement annotatedElement, int modifiers) {
        return ! (isTransient(annotatedElement, modifiers) || Modifier.isStatic(modifiers) || Modifier.isAbstract(modifiers));
    }
    
    /**
     * INTERNAL:
     * Check to see if this is a valid field to process for persistence. It is 
     * valid if it is not static, transient or has a @Transient specified.
     */
    protected boolean isValidPersistenceField(Field field) {
        return (isValidPersistenceElement(field, field.getModifiers()));
    }
    
    /**
     * INTERNAL:
     * Check to see if this is a valid method to process for persistence. It is 
     * valid if it is not static, transient or has a @Transient specified.
     */
    protected boolean isValidPersistenceMethod(Method method) {
        // Ignore methods marked transient, static or abstract.
        if (isValidPersistenceElement(method, method.getModifiers())) {
            // Look for methods that begin with "get" or "is", ignore all others.
            String methodName = method.getName();
            if (MetadataHelper.isValidPersistenceMethodName(methodName)) {
                // Ignore get methods with parameters.
                if (method.getParameterTypes().length > 0) {
                    return false;
                }
            
                Method setMethod = MetadataHelper.getSetMethod(method, getJavaClass());
            
                if (setMethod == null) {
                    if (MetadataHelper.getDeclaredAnnotationsCount(method, m_descriptor) > 0) {
                        // We decorated the property with annotations, but have 
                        // no corresponding setter property.
                        m_validator.throwNoCorrespondingSetterMethodDefined(getJavaClass(), method);
                    }
                    
                    // WIP
                    //m_logger.logWarningMessage(m_logger.IGNORE_GET_METHOD, m_descriptor, method);
                } else {
                    return true;
                }
            }
        }
        
        return false;
    }
    
    /**
     * INTERNAL:
     * Process the items of interest on an entity class. The order of 
     * processing is important, care must be taken if changes must be made.
     * 
     * Classes without an @Entity are ignored if no metadata complete flag
     * is set.
     */
    public void process() {
        if (hasEntityTag(getJavaClass())) {
            // This accessor represents an @Entity class.
            
            // Set the ignore flags for items that are already defined.
            m_descriptor.setIgnoreFlags();
            
            // Process the @Entity.
            processEntity();
            
            // Process the common class level attributes that an entity or
            // mapped superclass may define.
            processClass();
            
            // Process the @Table and @Inheritance.
            processTableAndInheritance();
                
            // Process the @MappedSuperclass(es).
            processMappedSuperclasses();
                    
            // Process the accessors on this entity.
            processAccessors();
            
            // Validate we found a primary key.
            validatePrimaryKey();
                    
            // Process the @SecondaryTable(s).
            processSecondaryTables();
        } else {
            // This accessor represents an @Embeddable class
            m_descriptor.setDescriptorIsEmbeddable();
            
            // Process the accessors on this embeddable.
            processAccessors();
        }
    }
    
    /**
     * INTERNAL:
     * Process an accessor method or field. Relationship accessors will be 
     * stored for later processing.
     */
    protected void processAccessor(MetadataAccessor accessor) {
        if (! accessor.isProcessed()) {
            // Store the accessor for later retrieval.
            m_descriptor.addAccessor(accessor);
        
            if (accessor.isRelationship()) {
                // Store the relationship accessors for later processing.
                m_project.addRelationshipDescriptor(m_descriptor);
            } else {
                accessor.process();
                accessor.setIsProcessed();
            }
        }
    }
    
    /**
     * INTERNAL:
     * Create mappings from the fields directly.
     */
    protected void processAccessorFields() {
        for (Field field : MetadataHelper.getFields(getJavaClass())) {
            if (isValidPersistenceField(field)) {
                processAccessor(buildAccessor(new MetadataField(field)));
            }
        }
    }
    
    /**
     * INTERNAL:
     * Create mappings via the class properties.
     */
    protected void processAccessorMethods() {
        for (Method method : MetadataHelper.getDeclaredMethods(getJavaClass())) {
            if (isValidPersistenceMethod(method)) {
                processAccessor(buildAccessor(new MetadataMethod(method)));
            }
        }
    }
    
    /**
     * INTERNAL:
     * Process the accessors for the given class.
     */
    protected void processAccessors() {
        // Process the fields or methods on the class.
        if (m_descriptor.usesPropertyAccess()) {
            processAccessorMethods();
        } else {
            processAccessorFields();
        }
    }
    
    /**
     * INTERNAL:
     * Process an @AssociationOverride for an Entity (or MappedSuperclass) 
     * that inherits from a MappedSuperclass.
     */
    protected void processAssociationOverride(String attributeName, MetadataJoinColumns joinColumns) {
        // Add association overrides from XML as we find them.
        if (joinColumns.loadedFromXML()) {
            // WIP - what about any defaulting?
            m_descriptor.addAssociationOverride(attributeName, joinColumns);
        } else {
            // Association override from annotations should not override those 
            // loaded from XML.
            MetadataJoinColumns existingJoinColumns = m_descriptor.getAssociationOverrideFor(attributeName);
            
            if (existingJoinColumns == null || ! existingJoinColumns.loadedFromXML()) {
                // WIP - what about any defaulting?
                m_descriptor.addAssociationOverride(attributeName, joinColumns);
            } else {
                // WIP should log a warning.
            }
        }
    }
    
    /**
     * INTERNAL: (Overriden in XMLClassAccessor)
     * Process an @AssociationOverrides for an Entity (or MappedSuperclass) 
     * that inherits from a MappedSuperclass.
     * 
     * It will also look for an @AssociationOverride.
     */
    protected void processAssociationOverrides() {
        // Look for an @AssociationOverrides.
        AssociationOverrides associationOverrides = getAnnotation(AssociationOverrides.class);
        if (associationOverrides != null) {
            for (AssociationOverride associationOverride : associationOverrides.value()) {
                processAssociationOverride(associationOverride.name(), new MetadataJoinColumns(associationOverride.joinColumns()));
            }
        }
        
        // Look for an @AssociationOverride.
        AssociationOverride associationOverride = getAnnotation(AssociationOverride.class);
        if (associationOverride != null) {
            processAssociationOverride(associationOverride.name(), new MetadataJoinColumns(associationOverride.joinColumns()));
        }
    }
    
    /**
     * INTERNAL:
     * Process the @AttributeOverrides and @AttributeOverride for an Entity (or 
     * MappedSuperclass) that inherits from a MappedSuperclass.
     */
    protected void processAttributeOverride(MetadataColumn column) {
        String attributeName = column.getAttributeName();
        
        // Add attribute overrides from XML as we find them.
        if (column.loadedFromXML()) {
            m_descriptor.addAttributeOverride(column);
        } else {
            // Attribute overrides from annotations should not override
            // those loaded from XML.
            MetadataColumn existingColumn = m_descriptor.getAttributeOverrideFor(attributeName);
            
            if (existingColumn == null || ! existingColumn.loadedFromXML()) {
                m_descriptor.addAttributeOverride(column);
            } else {
                // WIP should log a warning.
            }
        }
    }
    
    /**
     * INTERNAL: (Overridden in XMLClassAccessor)
     * Process the @AttributeOverrides and @AttributeOverride for an Entity (or 
     * MappedSuperclass) that inherits from a MappedSuperclass.
     */
    protected void processAttributeOverrides() {
        // Look for an @AttributeOverrides.
        AttributeOverrides attributeOverrides = getAnnotation(AttributeOverrides.class);	
        if (attributeOverrides != null) {
            for (AttributeOverride attributeOverride : attributeOverrides.value()) {
                processAttributeOverride(new MetadataColumn(attributeOverride, getAnnotatedElement()));
            }
        }
        
        // Look for an @AttributeOverride.
        AttributeOverride attributeOverride = getAnnotation(AttributeOverride.class);
        if (attributeOverride != null) {
            processAttributeOverride(new MetadataColumn(attributeOverride, getAnnotatedElement()));
        }
    }
    
    /**
     * INTERNAL:
     * Process the array of methods for lifecyle callback events and set them
     * on the given event listener.
     */
    protected void processCallbackMethods(Method[] candidateMethods, MetadataEntityListener listener) {
        for (Method method : candidateMethods) {
            if (isAnnotationPresent(PostLoad.class, method)) {
                setPostLoad(method, listener);
            }
            
            if (isAnnotationPresent(PostPersist.class, method)) {
                setPostPersist(method, listener);
            }
            
            if (isAnnotationPresent(PostRemove.class, method)) {
                setPostRemove(method, listener);
            }
            
            if (isAnnotationPresent(PostUpdate.class, method)) {
                setPostUpdate(method, listener);
            }
            
            if (isAnnotationPresent(PrePersist.class, method)) {
                setPrePersist(method, listener);
            }
            
            if (isAnnotationPresent(PreRemove.class, method)) {
                setPreRemove(method, listener);
            }
            
            if (isAnnotationPresent(PreUpdate.class, method)) {
                setPreUpdate(method, listener);
            }
        }
    }
    
    /**
     * INTERNAL:
     * Process the items of interest on an entity or mapped superclass class. 
     */
    protected void processClass() {
        // Process the @AttributeOverrides and @AttributeOverride.
        processAttributeOverrides();
                    
        // Process the @AssociationOverrides and @AssociationOverride.
        processAssociationOverrides();
        
        // Process the @NamedQueries and @NamedQuery.
        processNamedQueries();
                    
        // Process the @NamedNativeQueries and @NamedNativeQuery.
        processNamedNativeQueries();
                    
        // Process the @SqlRessultSetMapping.
        processSqlResultSetMappings();
                    
        // Process the @TableGenerator.
        processTableGenerator();
            
        // Process the @SequenceGenerator
        processSequenceGenerator();
                    
        // Process the @IdClass (pkClass).
        processIdClass();
        
        // Process the @ExcludeDefaultListeners.
        processExcludeDefaultListeners();
        
        // Process the @ExcludeSuperclassListeners.
        processExcludeSuperclassListeners();
    }

    /**
     * INTERNAL:
     * Process the default listeners defined in XML. This method will process 
     * the class for additional lifecycle callback methods that are decorated 
     * with annotations.
     * 
     * NOTE: We add the default listeners regardless if the exclude default 
     * listeners flag is set. This allows the user to change the exlcude flag 
     * at runtime and have the default listeners available to them.
     */
    protected void processDefaultListeners(ClassLoader loader) {
        Map<XMLHelper, NodeList> defaultListeners = m_project.getDefaultListeners();
        
        for (XMLHelper helper : defaultListeners.keySet()) {
            // Update the class loader.
            helper.setLoader(loader);
            NodeList nodes = defaultListeners.get(helper);
            
            for (int i = 0; i < nodes.getLength(); i++) {
                Node node = nodes.item(i);
                
                // Build an xml entity listener.
                XMLEntityListener listener = new XMLEntityListener(helper.getClassForNode(node), getJavaClass());
                
                // Process the lifecycle callback events from XML.
                Method[] candidateMethods = MetadataHelper.getCandidateCallbackMethodsForDefaultListener(listener);
                processLifecycleEvents(listener, node, helper, candidateMethods);
                
                // Process the candidate callback methods on this listener for
                // additional callback methods decorated with annotations.
                processCallbackMethods(candidateMethods, listener);
        
                // Add the listener to the descriptor.
                m_descriptor.addDefaultEventListener(listener);
            }
		}
    }
    
    /**
     * INTERNAL:
     * Process a @DiscriminatorColumn (if there is one, otherwise default) to 
     * set this classes indication field name for inheritance.
     */
    protected void processDiscriminatorColumn() {
        DiscriminatorColumn discriminatorColumn = getAnnotation(DiscriminatorColumn.class);
        processDiscriminatorColumn(new MetadataDiscriminatorColumn(discriminatorColumn));
    }
    
    /**
     * INTERNAL:
	 * Process a discriminator column to set this class indicatior field name 
     * for inheritance.
	 */
	protected void processDiscriminatorColumn(MetadataDiscriminatorColumn discriminatorColumn) {
        DatabaseField field = new DatabaseField();

        field.setName(getName(discriminatorColumn.getName(), MetadataDiscriminatorColumn.DEFAULT_NAME, m_logger.DISCRIMINATOR_COLUMN));
        field.setLength(discriminatorColumn.getLength());
        field.setTableName(m_descriptor.getPrimaryTableName());
        field.setColumnDefinition(discriminatorColumn.getColumnDefinition());
        field.setType(MetadataHelper.getDiscriminatorType(discriminatorColumn.getDiscriminatorType()));
        
        // Set the class indicator field on the inheritance policy.
        m_descriptor.setClassIndicatorField(field);
    }
	
    /**
     * INTERNAL:
	 * Process a discriminator value to set the class indicator on the root 
     * descriptor of the inheritance hierarchy. 
     * 
     * If there is no discriminator value, the class indicator defaults to 
     * the class name.
	 */
	protected void processDiscriminatorValue() {
        if (! Modifier.isAbstract(getJavaClass().getModifiers())) {
            // Add the indicator to the inheritance root class' descriptor. The
            // default is the short class name.
            String discriminatorValue = getDiscriminatorValue();
            
            if (discriminatorValue == null) {
                // WIP - log a warning ...
                m_descriptor.addClassIndicator(getJavaClass(), Helper.getShortClassName(getJavaClassName()));    
            } else {
                m_descriptor.addClassIndicator(getJavaClass(), discriminatorValue);    
            }
        }
    }
    
    /**
	 * INTERNAL:
	 * Process an entity.
	 */
	protected void processEntity() {
        // Don't override existing alias.
        if (m_descriptor.getAlias().equals("")) {
            String alias = getEntityName();
            
            if (alias.equals("")) {
                alias = Helper.getShortClassName(getJavaClassName());
                m_logger.logConfigMessage(m_logger.ALIAS, m_descriptor, alias);
            }

            // Verify that the alias is not a duplicate.
            ClassDescriptor existingDescriptor = getProject().getSession().getProject().getDescriptorForAlias(alias);
            if (existingDescriptor != null) {
                m_validator.throwNonUniqueEntityName(existingDescriptor.getJavaClassName(), m_descriptor.getJavaClassName(), alias);
            }

            // Set the alias on the descriptor and add it to the project.
            m_descriptor.setAlias(alias);
            getProject().getSession().getProject().addAlias(alias, m_descriptor.getDescriptor());
        }
	}
    
    /**
     * INTERNAL: (Overridden in XMLCLassAccessor)
     * Process the entity class for lifecycle callback event methods.
     */
    public MetadataEntityListener processEntityEventListener(ClassLoader loader) {
        MetadataEntityClassListener listener = new MetadataEntityClassListener(getJavaClass());
            
        // Check the entity class for lifecycle callback annotations.
        processCallbackMethods(MetadataHelper.getCandidateCallbackMethodsForEntityClass(getJavaClass()), listener);
        
        return listener;
    }
    
    /**
     * INTERNAL: (Overridden in XMLClassAccessor)
     * Process the @EntityListeners for this class accessor.
     */
    public void processEntityListeners(Class entityClass, ClassLoader loader) {
        EntityListeners entityListeners = getAnnotation(EntityListeners.class);
        
        if (entityListeners != null) {
            for (Class entityListener : entityListeners.value()) {
                MetadataEntityListener listener = new MetadataEntityListener(entityListener, entityClass);
                
                // Process the candidate callback methods for this listener ...
                processCallbackMethods(MetadataHelper.getCandidateCallbackMethodsForEntityListener(listener), listener);
                
                // Add the entity listener to the descriptor event manager.    
                m_descriptor.addEntityListenerEventListener(listener);
            }
        }
    }
    
    /**
     * INTERNAL: (Overridden in XMLClassAccessor)
     * Process the @ExcludeDefaultListeners if one is specified (taking
     * metadata-complete into consideration).
     */
    protected void processExcludeDefaultListeners() {
        // Don't overrite a true flag that could be set from a subclass
        // that already exlcuded them.
        if (isAnnotationPresent(ExcludeDefaultListeners.class)) {
            m_descriptor.setExcludeDefaultListeners(true);
        }
    }
    
    /**
     * INTERNAL: (Overridden in XMLClassAccessor)
     * Process the @ExcludeSuperclassListeners if one is specified (taking
     * metadata-complete into consideration).
     */
    protected void processExcludeSuperclassListeners() {
        // Don't overrite a true flag that could be set from a subclass
        // that already exlcuded them.
        if (isAnnotationPresent(ExcludeSuperclassListeners.class)) {
            m_descriptor.setExcludeSuperclassListeners(true);
        }
    }
    
    /**
     * INTERNAL:
     * Process an @IdClass or id-class element. It is used to specify composite 
     * primary keys. The primary keys will be processed and stored from the PK 
     * class so that they may be validated against the fields or properties of 
     * the entity bean. The access type of a primary key class is determined by 
     * the access type of the entity for which it is the primary key.
     * 
     * NOTE: the class passed in may be a mapped-superclass or entity.
     */
    protected void processIdClass() {
        Class idClass = getIdClass();
        
        if (idClass != null) {
            if (m_descriptor.ignoreIDs()) {
                m_logger.logWarningMessage(m_logger.IGNORE_ID_CLASS, m_descriptor, idClass);
            } else {
                m_descriptor.setPKClass(idClass);
                
                if (m_descriptor.usesPropertyAccess()) {
                    for (Method method : MetadataHelper.getDeclaredMethods(idClass)) {
                        String methodName = method.getName();
                
                        if (MetadataHelper.isValidPersistenceMethodName(methodName)) {
                            m_descriptor.addPKClassId(MetadataHelper.getAttributeNameFromMethodName(methodName), MetadataHelper.getGenericReturnType(method));
                        }
                    }
                } else {
                    for (Field field : MetadataHelper.getFields(idClass)) {
                        m_descriptor.addPKClassId(field.getName(), MetadataHelper.getGenericType(field));
                    }
                }   
            }
        }
    }
    
    /**
     * INTERNAL:
     * Process the @Inheritance or inheritance tag if there is one. One may or
     * may not be specified for the entity class that is the root of the entity 
     * class hierarchy.
     */
    protected void processInheritanceRoot() {
        if (m_descriptor.ignoreInheritance()) {
            m_logger.logWarningMessage(m_logger.IGNORE_INHERITANCE, m_descriptor);
        } else {
            // Get the inheritance strategy and store it on the descriptor.
            String inheritanceStrategy = getInheritanceStrategy();
            if (inheritanceStrategy.equals("")) {
                // WIP - should log a message that we are defaulting ...
                inheritanceStrategy = InheritanceType.SINGLE_TABLE.name();
            }

            m_descriptor.setInheritanceStrategy(inheritanceStrategy);
                
            // Process the discriminator column metadata.
            processDiscriminatorColumn();
                
            // Process the discriminator value metadata.
            processDiscriminatorValue();
        }
    }
    
    /**
     * INTERNAL:
     * Process an inheritance subclass.
     */
    public void processInheritanceSubclass() {
        if (m_descriptor.ignoreInheritance()) {
            m_logger.logWarningMessage(m_logger.IGNORE_INHERITANCE, m_descriptor);
        } else {
            // Log a warning if we are ignoring an entity that has an inheritance tag.
            if (hasInheritanceTag(getJavaClass())) {
                m_logger.logWarningMessage(m_logger.IGNORE_INHERITANCE, m_descriptor);
            }
            
            // Set the parent class on the source descriptor.
            setInheritanceParentClass();
                
            // Get the inheritance root descriptor metadata.
            MetadataDescriptor rootDescriptor = getInheritanceRootDescriptor();
                
            // Inheritance.stategy() = SINGLE_TABLE, set the flag.
            if (rootDescriptor.usesSingleTableInheritanceStrategy()) {
                m_descriptor.setSingleTableInheritanceStrategy();
            } else {
                // Inheritance.stategy() = JOINED, then we need to look for 
                // primary key join column(s) and add multiple table key fields.
                MetadataPrimaryKeyJoinColumns primaryKeyJoinColumns = getPrimaryKeyJoinColumns(rootDescriptor.getPrimaryTableName(), m_descriptor.getPrimaryTableName());
                addMultipleTableKeyFields(primaryKeyJoinColumns, m_logger.INHERITANCE_PK_COLUMN, m_logger.INHERITANCE_FK_COLUMN);
            }    
            
            // Process the discriminator value.
            processDiscriminatorValue();
            
            // If the root descriptor has an id class, we need to set the same 
            // id class on our descriptor.
            if (rootDescriptor.hasCompositePrimaryKey()) {
                m_descriptor.setPKClass(rootDescriptor.getPKClassName());
            }
        }
    }
    
    /**
     * INTERNAL:
     * Process and array of @JoinColumn into a list of metadata join column.
     */
    protected List<MetadataJoinColumn> processJoinColumns(JoinColumn[] joinColumns) {
        ArrayList<MetadataJoinColumn> list = new ArrayList<MetadataJoinColumn>();
        
        for (JoinColumn joinColumn : joinColumns) {
            list.add(new MetadataJoinColumn(joinColumn));
        }
        
        return list;
    }
    
    /**
     * INTERNAL:
     * Process the XML lifecycle event for the given listener.
     */
    protected void processLifecycleEvent(MetadataEntityListener listener, Node node, String event, XMLHelper helper, Method[] candidateMethods) {
        Node eventNode = helper.getNode(node, event);
        
        if (eventNode != null) {
            String methodName = helper.getNodeValue(eventNode, XMLConstants.ATT_METHOD_NAME);
            Method method = MetadataHelper.getMethodForName(candidateMethods, methodName);

            if (method == null) {
                // WIP - should be on the validator.
                throw ValidationException.invalidCallbackMethod(listener.getListenerClass(), methodName);
            } else if (event.equals(XMLConstants.PRE_PERSIST)) {
                setPrePersist(method, listener);
            } else if (event.equals(XMLConstants.POST_PERSIST)) {
                setPostPersist(method, listener);
            } else if (event.equals(XMLConstants.PRE_REMOVE)) {
                setPreRemove(method, listener);
            } else if (event.equals(XMLConstants.POST_REMOVE)) {
                setPostRemove(method, listener);
            } else if (event.equals(XMLConstants.PRE_UPDATE)) {
                setPreUpdate(method, listener);
            } else if (event.equals(XMLConstants.POST_UPDATE)) {
                setPostUpdate(method, listener);
            } else if (event.equals(XMLConstants.POST_LOAD)) {
                setPostLoad(method, listener);
            }
        }
    }
    
    /**
     * INTERNAL:
     * Process the XML lifecycle events for the given listener.
     */
    protected void processLifecycleEvents(MetadataEntityListener listener, Node node, XMLHelper helper, Method[] candidateMethods) {
        processLifecycleEvent(listener, node, XMLConstants.PRE_PERSIST, helper, candidateMethods);
        processLifecycleEvent(listener, node, XMLConstants.POST_PERSIST, helper, candidateMethods);
        processLifecycleEvent(listener, node, XMLConstants.PRE_REMOVE, helper, candidateMethods);
        processLifecycleEvent(listener, node, XMLConstants.POST_REMOVE, helper, candidateMethods);
        processLifecycleEvent(listener, node, XMLConstants.PRE_UPDATE, helper, candidateMethods);
        processLifecycleEvent(listener, node, XMLConstants.POST_UPDATE, helper, candidateMethods);
        processLifecycleEvent(listener, node, XMLConstants.POST_LOAD, helper, candidateMethods);
    }
    
    /**
     * INTERNAL:
     * Process the listeners for this class.
     */
    public void processListeners(ClassLoader loader) {
        // Step 1 - process the default listeners.
        processDefaultListeners(loader);

        // Step 2 - process the entity listeners that are defined on the entity 
        // class and mapped superclasses (taking metadata-complete into 
        // consideration). Go through the mapped superclasses first, top -> down 
        // only if the exclude superclass listeners flag is not set.    
        if (! m_descriptor.excludeSuperclassListeners()) {
            List<ClassAccessor> mappedSuperclasses = getMappedSuperclasses();
            int mappedSuperclassesSize = mappedSuperclasses.size();
            
            for (int i = mappedSuperclassesSize - 1; i >= 0; i--) {
                mappedSuperclasses.get(i).processEntityListeners(getJavaClass(), loader);
            }
        }
        
        processEntityListeners(getJavaClass(), loader); 
                
        // Step 3 - process the entity class for lifecycle callback methods. Go
        // through the mapped superclasses as well.
        MetadataEntityListener listener = processEntityEventListener(loader);
        
        if (! m_descriptor.excludeSuperclassListeners()) {
            for (ClassAccessor mappedSuperclass : getMappedSuperclasses()) {
                mappedSuperclass.processMappedSuperclassEventListener(listener, getJavaClass(), loader);
            }
        }
        
        // Add the listener only if we actually found callback methods.
        if (listener.hasCallbackMethods()) {
            m_descriptor.setEntityEventListener(listener);
        }
    }

    /**
     * INTERNAL:
     * Process the @MappedSuperclass(es) if there are any. There may be
     * several MappedSuperclasses for any given Entity.
     */
    protected void processMappedSuperclass() {
        // Process the common class level attributes that an entity or
        // mapped superclass may define.
        processClass();
            
        // Process the accessors from the mapped superclass.
        processAccessors();
    }
    
    /**
     * INTERNAL:
     * Process the @MappedSuperclass(es) if there are any. There may be
     * several MappedSuperclasses for any given Entity.
     */
    protected void processMappedSuperclasses() {
        for (ClassAccessor mappedSuperclass : getMappedSuperclasses()) {
            mappedSuperclass.process();
        }
    }
    
    /**
     * INTERNAL: (Overridden in XMLCLassAccessor)
     * Process the mapped superclass class for lifecycle callback event methods.
     */
    public void processMappedSuperclassEventListener(MetadataEntityListener listener, Class entityClass, ClassLoader loader) {
        // Check the mapped superclass for lifecycle callback annotations.
        processCallbackMethods(MetadataHelper.getCandidateCallbackMethodsForMappedSuperclass(getJavaClass(), entityClass), listener);
    }
    
    /**
     * INTERNAL:
     * Process a @NamedNativeQueries. The method will also look for 
     * a @NamedNativeQuery. This method currently only stores the queries if 
     * there are some. The actually query processing isn't done till 
     * addNamedQueriesToSession is called.
     */
    protected void processNamedNativeQueries() {
        // Look for a @NamedNativeQueries.
        NamedNativeQueries namedNativeQueries = getAnnotation(NamedNativeQueries.class);
        if (namedNativeQueries != null) {
            for (NamedNativeQuery namedNativeQuery : namedNativeQueries.value()) {
                processNamedNativeQuery(new MetadataNamedNativeQuery(namedNativeQuery));
            }
        }
        
        // Look for a @NamedNativeQuery.
        NamedNativeQuery namedNativeQuery = getAnnotation(NamedNativeQuery.class);
        if (namedNativeQuery != null) {
            processNamedNativeQuery(new MetadataNamedNativeQuery(namedNativeQuery));
        }
    }
    
    /**
     * INTERNAL:
     * Process a MetadataNamedNativeQuery. The actually query processing isn't 
     * done till addNamedQueriesToSession is called.
     */
    protected void processNamedNativeQuery(MetadataNamedNativeQuery namedNativeQuery) {
        if (m_project.hasNamedNativeQuery(namedNativeQuery.getName())) {
            MetadataNamedNativeQuery existingNamedNativeQuery = m_project.getNamedNativeQuery(namedNativeQuery.getName());
            
            if (existingNamedNativeQuery.loadedFromAnnotations() && namedNativeQuery.loadedFromXML()) {
                // Override the existing query.
                m_project.addNamedNativeQuery(namedNativeQuery);
            } else {
                // Ignore the query and log a message.
                m_logger.logWarningMessage(m_logger.IGNORE_QUERY, m_descriptor, namedNativeQuery.getName());
            }
        } else {
            m_project.addNamedNativeQuery(namedNativeQuery);
        }
    }
    
    /**
     * INTERNAL:
     * Process a @NamedQueries. The method will also look for a @NamedQuery.
     * This method currently only stores the queries if there are some. The
     * actually query processing isn't done till addNamedQueriesToSession is
     * called.
     */
    protected void processNamedQueries() {
        // Look for a @NamedQueries.
        NamedQueries namedQueries = getAnnotation(NamedQueries.class);
        
        if (namedQueries != null) {
            for (NamedQuery namedQuery : namedQueries.value()) {
                processNamedQuery(new MetadataNamedQuery(namedQuery));
            }
        }
        
        // Look for a @NamedQuery.
        NamedQuery namedQuery = getAnnotation(NamedQuery.class);
        
        if (namedQuery != null) {
            processNamedQuery(new MetadataNamedQuery(namedQuery));
        }
    }
    
    /**
     * INTERNAL:
     * Add a metadata named query to the project. The actually query processing 
     * isn't done till addNamedQueriesToSession is called.
     */
    protected void processNamedQuery(MetadataNamedQuery namedQuery) {
        if (m_project.hasNamedQuery(namedQuery.getName())) {
            MetadataNamedQuery existingNamedQuery = m_project.getNamedQuery(namedQuery.getName());
            
            if (existingNamedQuery.loadedFromAnnotations() && namedQuery.loadedFromXML()) {
                // Override the existing query.
                m_project.addNamedQuery(namedQuery);
            } else {
                // Ignore the query and log a message.
                m_logger.logWarningMessage(m_logger.IGNORE_QUERY, m_descriptor, namedQuery.getName());
            }
        } else {
            m_project.addNamedQuery(namedQuery);
        }
    }
    
    /**
     * INTERNAL:
     * Process a MetadataSecondaryTable. Do all the table name defaulting and 
     * set the correct, fully qualified name on the TopLink DatabaseTable.
     */
    protected void processSecondaryTable(MetadataSecondaryTable secondaryTable) {
        // Catalog could be "", need to check for an XML default.
        // WIP - build m_logger.SECONDARY_TABLE_CATALOG
        String catalog = getName(secondaryTable.getCatalog(), m_descriptor.getCatalog(), m_logger.TABLE_CATALOG);
        
        // Schema could be "", need to check for an XML default.
        // WIP - build m_logger.SECONDARY_TABLE_SCHEMA
        String schema = getName(secondaryTable.getSchema(), m_descriptor.getSchema(), m_logger.TABLE_SCHEMA);
		
        // Build a fully qualified name and set it on the secondary table.
        secondaryTable.setName(MetadataHelper.getFullyQualifiedTableName(secondaryTable.getName(), catalog, schema));
        
        // Add the table to the descriptor.
        m_descriptor.addTable(secondaryTable.getDatabaseTable());
        
        // Get the primary key join column(s) and add the multiple table key fields.
        MetadataPrimaryKeyJoinColumns primaryKeyJoinColumns = secondaryTable.getPrimaryKeyJoinColumns(m_descriptor.getPrimaryTableName());
        addMultipleTableKeyFields(primaryKeyJoinColumns, m_logger.SECONDARY_TABLE_PK_COLUMN, m_logger.SECONDARY_TABLE_FK_COLUMN);
    }
    
    /**
     * INTERNAL: (Overridden in XMLClassAccessor)
     * Process a @SecondaryTables. If one isn't found, try a @SecondaryTable.
     * WIP - If the @SecondaryTable does not define the pkJoinColumns(), we
     * could look for PrimaryKeyJoinColumns on the class itself. This is not
     * mandatory through.
     */
    protected void processSecondaryTables() {
        if (m_descriptor.ignoreTables()) {
            m_logger.logWarningMessage(m_logger.IGNORE_TABLE, getJavaClass());
        } else {
            // Look for a SecondaryTables annotation.
            SecondaryTables secondaryTables = getAnnotation(SecondaryTables.class);
            if (secondaryTables != null) {
                for (SecondaryTable secondaryTable : secondaryTables.value()) {
                    processSecondaryTable(new MetadataSecondaryTable(secondaryTable));
                }
            } else {
                // Look for a SecondaryTable annotation
                SecondaryTable secondaryTable = getAnnotation(SecondaryTable.class);
                if (secondaryTable != null) {
                    processSecondaryTable(new MetadataSecondaryTable(secondaryTable));
                }
            }
        }
    } 
    
    /**
     * INTERNAL:
     * Process an sql result set mapping metadata into a TopLink 
     * SqlResultSetMapping and store it on the session.
     */
    protected void processSqlResultSetMapping(MetadataSQLResultSetMapping sqlResultSetMapping) {        
        // Initialize a new SqlResultSetMapping (with the metadata name)
        oracle.toplink.essentials.queryframework.SQLResultSetMapping mapping = new oracle.toplink.essentials.queryframework.SQLResultSetMapping(sqlResultSetMapping.getName());
        
        // Process the entity results.
        for (MetadataEntityResult eResult : sqlResultSetMapping.getEntityResults()) {
            EntityResult entityResult = new EntityResult(eResult.getEntityClass().getName());
        
            // Process the field results.
            for (MetadataFieldResult fResult : eResult.getFieldResults()) {
                entityResult.addFieldResult(new FieldResult(fResult.getName(), fResult.getColumn()));
            }
        
            // Process the discriminator value;
            entityResult.setDiscriminatorColumn(eResult.getDiscriminatorColumn());
        
            // Add the result to the SqlResultSetMapping.
            mapping.addResult(entityResult);
        }
        
        // Process the column results.
        for (String columnResult : sqlResultSetMapping.getColumnResults()) {
            mapping.addResult(new ColumnResult(columnResult));
        }
            
        getProject().getSession().getProject().addSQLResultSetMapping(mapping);
    }
    
    /**
     * INTERNAL:
     * Process a @SqlResultSetMappings.
     */
    protected void processSqlResultSetMappings() {
        // Look for a @SqlResultSetMappings.
        SqlResultSetMappings sqlResultSetMappings = getAnnotation(SqlResultSetMappings.class);

        if (sqlResultSetMappings != null) {
            for (SqlResultSetMapping sqlResultSetMapping : sqlResultSetMappings.value()) {
                processSqlResultSetMapping(new MetadataSQLResultSetMapping(sqlResultSetMapping));
            }
        } else {
            // Look for a @SqlResultSetMapping.
            SqlResultSetMapping sqlResultSetMapping = getAnnotation(SqlResultSetMapping.class);
            
            if (sqlResultSetMapping != null) {
                processSqlResultSetMapping(new MetadataSQLResultSetMapping(sqlResultSetMapping));
            }
        }
    } 

    /**
     * INTERNAL: (Overridden in XMLClassAccessor)
	 * Process a @Table annotation.
	 */
	protected void processTable() {
        if (m_descriptor.ignoreTables()) {
            m_logger.logWarningMessage(m_logger.IGNORE_TABLE, getJavaClass());
        } else {
            Table table = getAnnotation(Table.class);
            processTable(new MetadataTable(table));
        }
    }
    
    /**
     * INTERNAL:
	 * Process a MetadataTable. Do all the table name defaulting and set the
     * correct, fully qualified name on the TopLink DatabaseTable.
	 */
    protected void processTable(MetadataTable table) {
        // Name could be "", need to check against the default name.
		String name = getName(table.getName(), m_descriptor.getDefaultTableName(), m_logger.TABLE_NAME);

        // Catalog could be "", need to check for an XML default.
        String catalog = getName(table.getCatalog(), m_descriptor.getCatalog(), m_logger.TABLE_CATALOG);
        
        // Schema could be "", need to check for an XML default.
        String schema = getName(table.getSchema(), m_descriptor.getSchema(), m_logger.TABLE_SCHEMA);
		
        // Build a fully qualified name and set it on the table.
        table.setName(MetadataHelper.getFullyQualifiedTableName(name, catalog, schema));

        // Set the table on the descriptor.
        m_descriptor.setPrimaryTable(table.getDatabaseTable());
    }
    
    /**
     * INTERNAL:
     * Process any inheritance specifics. This method will fast track any root 
     * inheritance processing, be it specified or defaulted.
     */
    protected void processTableAndInheritance() {
        // If we are an inheritance subclass, ensure our root is processed 
        // first since it has information its subclasses depend on.
		if (isInheritanceSubclass()) {
            MetadataDescriptor rootDescriptor = getInheritanceRootDescriptor();
            
            // Process the root class accesor if it hasn't already been done.
            ClassAccessor rootAccessor = rootDescriptor.getClassAccessor();
            if (rootAccessor == null) {
                rootAccessor = processAccessor(rootDescriptor);
            }
            
            // If the root class inheritance specifics haven't been processed
            // yet, then do them now.
            if (! rootDescriptor.hasInheritance()) {
                rootAccessor.processInheritanceRoot();
            }
                
            // Process the @Table if there is one. Must be processed before
            // processing the inheritance subclass metadata.
            if (! rootDescriptor.usesSingleTableInheritanceStrategy()) {
                processTable();    
            }
            
            // Process our inheritance specifics.
            processInheritanceSubclass();
		} else {
            // Process the @Table if there is one.
            processTable();
        }
    }
    
    /**
     * INTERNAL:
     * Should only be called from those classes that are in fact part of an 
     * inheritance hierarchy. It figures out the parent and then sets it.
     */
    protected void setInheritanceParentClass() {
        Class parent = getJavaClass().getSuperclass();
        
        while (parent != Object.class) {
            if (hasEntityTag(parent) || m_project.containsDescriptor(parent)) {
                break;
            }
            
            parent = parent.getSuperclass();
        }
        
        m_descriptor.setParentClass(parent);
    }
    
    /**
     * INTERNAL:
     * Set the post load event method on the listener.
     */
    protected void setPostLoad(Method method, MetadataEntityListener listener) {
        listener.setPostBuildMethod(method);
        listener.setPostCloneMethod(method);
        listener.setPostRefreshMethod(method);
    }
    
    /**
     * INTERNAL:
     * Set the post persist event method on the listener.
     */
    protected void setPostPersist(Method method, MetadataEntityListener listener) {
        listener.setPostInsertMethod(method); 
    }
    
    /**
     * INTERNAL:
     * Set the post remove event method on the listener.
     */
    protected void setPostRemove(Method method,  MetadataEntityListener listener) {
        listener.setPostDeleteMethod(method);
    }
    
    /**
     * INTERNAL:
     * * Set the post update event method on the listener.
     */
    protected void setPostUpdate(Method method,  MetadataEntityListener listener) {
        listener.setPostUpdateMethod(method);
    }
            
    /**
     * INTERNAL:
     * Set the pre persist event method on the listener.
     */
    protected void setPrePersist(Method method,  MetadataEntityListener listener) {
        listener.setPrePersistMethod(method);
    }
    
    /**
     * INTERNAL:
     * Set the pre remove event method on the listener.
     */
    protected void setPreRemove(Method method,  MetadataEntityListener listener) {
        listener.setPreRemoveMethod(method);
    }
    
    /**
     * INTERNAL:
     * Set the pre update event method on the listener.
     */
    protected void setPreUpdate(Method method,  MetadataEntityListener listener) {
        listener.setPreUpdateWithChangesMethod(method);
    }
    
    /**
     * INTERNAL:
     * Call this method after a primary key should have been found.
     */
    protected void validatePrimaryKey() {
        // If this descriptor has a composite primary key, check that all 
        // our composite primary key attributes were validated. 
        if (m_descriptor.hasCompositePrimaryKey()) {
            if (m_descriptor.pkClassWasNotValidated()) {
                m_validator.throwInvalidCompositePKSpecification(getJavaClass(), m_descriptor.getPKClassName());
            }
        } else {
            // Descriptor has a single primary key. Validate an id 
            // attribute was found, unless we are an inheritance subclass
            // or an aggregate descriptor.
            if (! m_descriptor.hasPrimaryKeyFields() && ! isInheritanceSubclass()) {
                m_validator.throwNoPrimaryKeyAnnotationsFound(getJavaClass());
            }
        }  
    }
}
