/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.tools.verifier.ejb;

import java.io.File;
import java.io.IOException;

import com.sun.enterprise.deployment.Descriptor;
import com.sun.enterprise.deployment.EjbBundleDescriptor;
import com.sun.enterprise.tools.verifier.BaseVerifier;
import com.sun.enterprise.tools.verifier.FrameworkContext;
import com.sun.enterprise.tools.verifier.SpecVersionMapper;
import com.sun.enterprise.tools.verifier.apiscan.classfile.ClassFileLoaderFactory;
import com.sun.enterprise.tools.verifier.apiscan.packaging.ClassPathBuilder;
import com.sun.enterprise.tools.verifier.apiscan.stdapis.EjbClosureCompiler;
import com.sun.enterprise.util.io.FileUtils;

/**
 * The class handles the EJB Verification.
 *
 * @author Vikas Awasthi
 */
public class EjbVerifier extends BaseVerifier {

    private EjbBundleDescriptor ejbd = null;
    private String classPath;//this is lazily populated in getClassPath()
    private boolean isASMode = false;

    public EjbVerifier(FrameworkContext frameworkContext,
                       EjbBundleDescriptor ejbd) {
        this.frameworkContext = frameworkContext;
        this.ejbd = ejbd;
        this.isASMode = !frameworkContext.isPortabilityMode();
    }

    public void verify() throws Exception {
        if (areTestsNotRequired(frameworkContext.isEjb()) &&
                areTestsNotRequired(frameworkContext.isWebServices()) &&
                areTestsNotRequired(frameworkContext.isWebServicesClient()) &&
                areTestsNotRequired(frameworkContext.isPersistenceUnits()))
            return;

        preVerification();
        createClosureCompiler();//this can be moved up to base verifier in future.
        verify(ejbd, new EjbCheckMgrImpl(frameworkContext));
    }

    public Descriptor getDescriptor() {
        return ejbd;
    }

    /**
     * Create the classloader from the extracted directory
     */
    protected ClassLoader createClassLoader()
            throws IOException {
        return ejbd.getClassLoader();
    }

    protected String getArchiveUri() {
        return FileUtils.makeFriendlyFileName(ejbd.getModuleDescriptor().getArchiveUri());
    }

    protected String[] getDDString() {
        String dd[] = {"META-INF/sun-ejb-jar.xml", "META-INF/ejb-jar.xml", // NOI18N
                       "META-INF/webservices.xml"}; // NOI18N
        return dd;
    }

    /**
     * Creates and returns the class path associated with the ejb archive.
     * Uses the exploded location of the archive for generating the classpath.
     *
     * @return entire classpath string
     * @throws IOException
     */
    protected String getClassPath() throws IOException {
        if (classPath != null) return classPath;

        if(isASMode)
            return (classPath = getClassPath(frameworkContext.getClassPath()));

        String cp;
        if (!ejbd.getModuleDescriptor().isStandalone()) {
            //take the cp from the enclosing ear file
            String ear_uri = frameworkContext.getExplodedArchivePath();
            File ear = new File(ear_uri);
            assert(ear.isDirectory());
            String earCP = ClassPathBuilder.buildClassPathForEar(ear);
            String libdir = ejbd.getApplication().getLibraryDirectory();
            if (libdir!=null) {
                earCP = getLibdirClasspath(ear_uri, libdir) + earCP;
            }
            String module_uri = ejbd.getModuleDescriptor().getArchiveUri();//this is a relative path
            File module = new File(module_uri);
            assert(module.isFile() && !module.isAbsolute());
            // exploder creates the directory replacing all dots by '_'
            File explodedModuleDir = new File(ear_uri,
                    FileUtils.makeFriendlyFileName(module_uri));
            String moduleCP = ClassPathBuilder.buildClassPathForJar(
                    explodedModuleDir);
            cp = moduleCP + File.pathSeparator + earCP;
        } else {
            String module_uri = frameworkContext.getExplodedArchivePath();//this is an absolute path
            File module = new File(module_uri);
            assert(module.isDirectory() && module.isAbsolute());
            cp = ClassPathBuilder.buildClassPathForJar(module);
        }
        return (classPath = cp);
    }

    /**
     * creates the ClosureCompiler for the ejb module and sets it to the
     * verifier context. This is used to compute the closure on the classes used
     * in the ejb archive.
     *
     * @throws IOException
     */
    protected void createClosureCompiler() throws IOException {
        String specVer = SpecVersionMapper.getEJBVersion(
                frameworkContext.getJavaEEVersion());
        Object arg = (isASMode)?ejbd.getClassLoader():(Object)getClassPath();
        EjbClosureCompiler cc = new EjbClosureCompiler(specVer,
                ClassFileLoaderFactory.newInstance(new Object[]{arg}));
        context.setClosureCompiler(cc);
    }
}
