/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.web.connector.grizzly;

import java.io.IOException;
import java.io.InterruptedIOException;
import java.net.BindException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketException;
import java.util.Iterator;
import java.util.Set;
import java.util.logging.Logger;
import java.util.logging.Level;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.nio.channels.CancelledKeyException;
import java.nio.channels.Selector;
import java.nio.channels.SelectionKey;
import java.nio.channels.ServerSocketChannel;
import java.nio.channels.SocketChannel;

import java.security.AccessControlException;

import org.apache.coyote.Adapter;
import org.apache.coyote.RequestGroupInfo;
import org.apache.tomcat.util.net.ServerSocketFactory;
import org.apache.tomcat.util.net.SSLImplementation;

import com.sun.org.apache.commons.modeler.Registry;
import javax.management.ObjectName;
import javax.management.MBeanServer;
import javax.management.MBeanRegistration;

import com.sun.enterprise.web.connector.grizzly.algorithms.NoParsingAlgorithm;

/**
 * This class implement an NIO socket HTTP Listener. This class 
 * supports three stagegy:
 *
 * Mode Blocking: This mode uses NIO blocking mode, and doesn't uses any of the 
 *                java.nio.* classes.
 *
 *
 * Mode Non-Blocking: This mode uses NIO non blocking mode and read the entire 
 *         request stream before processing the request. The stragegy used is 
 *         to find the content-lenght header and buffer bytes until the end of 
 *         the stream is read.
 *
 * @author Jean-Francois Arcand
 */
public class SelectorThread extends Thread implements MBeanRegistration{
    
    private int serverTimeout = Constants.DEFAULT_SERVER_SOCKET_TIMEOUT;

    private InetAddress inet;
    protected int port;

    // SSL required stuff.
    private ServerSocketFactory factory;
    private ServerSocket serverSocket;
    protected SSLImplementation sslImplementation = null;

    
    /**
     * The <code>ServerSocketChannel</code> used in blocking mode.
     */
    private ServerSocketChannel serverSocketChannel;
    
    protected volatile boolean running = false;
    private volatile boolean paused = false;
    private boolean initialized = false;
    private boolean reinitializing = false;
    // ----------------------------------------------------- JMX Support ---/
    protected String domain;
    protected ObjectName oname;
    protected ObjectName globalRequestProcessorName;
    private ObjectName keepAliveMbeanName;
    private ObjectName pwcConnectionQueueMbeanName;
    private ObjectName pwcFileCacheMbeanName;
    protected MBeanServer mserver;
    protected ObjectName processorWorkerThreadName;


    // ------------------------------------------------------Socket setting --/

    protected boolean tcpNoDelay=false;
    
    
    protected int linger=100;
    
    
    protected int socketTimeout=-1;
    
    
    protected int maxKeepAliveRequests = Constants.DEFAULT_MAX_KEEP_ALIVE;
        
    // ------------------------------------------------------ Properties----/
    
    /**
     * Default HTTP header buffer size.
     */
    protected int maxHttpHeaderSize = Constants.DEFAULT_HEADER_SIZE;


    /**
     * Number of polled <code>Read*Task</code> instance.
     */
    protected int minReadQueueLength = 10;


    /**
     * Number of polled <code>ProcessorTask</code> instance.
     */
    protected int minProcessorQueueLength = 10;


    /**
     * The <code>Selector</code> used by the connector.
     */
    protected Selector selector;


    /**
     * Associated adapter.
     */
    protected Adapter adapter = null;
    
    
    /**
     * Is SSL enabled
     */
    private boolean secure = false;

    
    /**
     * The queue shared by this thread and code>ReadTask</code>
     */ 
    protected Pipeline readPipeline;
    

    /**
     * The queue shared by this thread and the code>ProcessorTask</code>.
     */ 
    protected Pipeline processorPipeline;
    
  
    /**
     * Placeholder for <code>Pipeline</code> statistic.
     */
    protected PipelineStatistic pipelineStat;
    
    /**
     * The default <code>Pipeline</code> used.
     */
    protected String pipelineClassName = 
        com.sun.enterprise.web.connector.grizzly.
            LinkedListPipeline.class.getName();
    
    /**
     * Maximum number of <code>WorkerThread</code>
     */
    protected int maxProcessorWorkerThreads = 20; // By default
    
    
    /**
     * Maximum number of <code>ReadWorkerThread</code>
     */
    protected int maxReadWorkerThreads = -1; // By default

    
    /**
     * Minimum numbers of <code>WorkerThread</code> created
     */
    protected int minWorkerThreads = 5;
    

    /**
     * Minimum numbers of <code>WorkerThread</code> 
     * before creating new thread.
     * <implementation-note>
     * Not used in 9.x
     * </implementation-note>
     */
    protected int minSpareThreads = 2;

    
    /**
     * The number used when increamenting the <code>Pipeline</code> 
     * thread pool.
     */
    protected int threadsIncrement = 1;
    
    
    /**
     * The timeout used by the thread when processing a request.
     */
    protected int threadsTimeout = Constants.DEFAULT_TIMEOUT;


    /**
     * Are we using NIO non bloking mode
     */
    protected boolean useNioNonBlocking = true;

    
    /**
     * Is the <code>ByteBuffer</code> used by the <code>ReadTask</code> use
     * direct <code>ByteBuffer</code> or not.
     */
    protected boolean useDirectByteBuffer = false;
    
  
    /**
     * Monitoring object used to store information.
     */
    private RequestGroupInfo globalRequestProcessor= new RequestGroupInfo();
    
    
    /**
     * Keep-alive stats
     */
    private KeepAliveStats keepAliveStats = new KeepAliveStats();


    /**
     * If <code>true</code>, display the NIO configuration information.
     */
    protected boolean displayConfiguration = false;
    
    
    /**
     * Is monitoring already started.
     */
    private boolean isMonitoringEnabled = false;
    

    /**
     * The current number of simulatenous connection.
     */
    protected int currentConnectionNumber;


    /**
     * Is this Selector currently in Wating mode?
     */
    protected volatile boolean isWaiting = false;
    

    /**
     * The input request buffer size.
     */
    protected int requestBufferSize = Constants.DEFAULT_REQUEST_BUFFER_SIZE;
    
    
    /**
     * Create view <code>ByteBuffer</code> from another <code>ByteBuffer</code>
     */
    protected boolean useByteBufferView = false;
    

    /*
     * Number of seconds before idle keep-alive connections expire
     */
    private int keepAliveTimeoutInSeconds = Constants.DEFAULT_TIMEOUT;

    
    /**
     * Number of seconds before idle keep-alive connections expire
     */
    private int kaTimeout = Constants.DEFAULT_TIMEOUT * 1000;
    
    
    /**
     * Recycle the <code>Task</code> after running them
     */
    protected boolean recycleTasks = Constants.DEFAULT_RECYCLE;
    
    
    /**
     * The <code>Selector</code> timeout value. By default, it is set to 60000
     * miliseconds (as in the j2se 1.5 ORB).
     */
    protected static int selectorTimeout = 1000;


    /**
     * Maximum pending connection before refusing requests.
     */
    protected int maxQueueSizeInBytes = Constants.DEFAULT_QUEUE_SIZE;


    /**
     * The <code>Algorithm</code> used to predict the end of the NIO stream
     */
    protected Class algorithmClass;
    
    
    /**
     * The <code>Algorithm</code> used to parse the NIO stream.
     */
    protected String algorithmClassName = DEFAULT_ALGORITHM;
    
    
    /**
     * The default NIO stream algorithm.
     */
    public final static String DEFAULT_ALGORITHM =
        com.sun.enterprise.web.connector.grizzly.algorithms.
            NoParsingAlgorithm.class.getName();

    
    /**
     * Server socket backlog.
     */
    protected int ssBackLog = 4096;
    
    
    /**
     * Next time the exprireKeys() will delete keys.
     */    
    private long nextKeysExpiration = 0;
    
    
    /**
     * The default response-type
     */
    protected String defaultResponseType = 
            org.apache.coyote.tomcat5.Constants.DEFAULT_RESPONSE_TYPE;


    /**
     * The forced response-type
     */
    protected String forcedResponseType = 
            org.apache.coyote.tomcat5.Constants.DEFAULT_RESPONSE_TYPE;
    
    
    /**
     * The root folder where application are deployed
     */
    protected static String rootFolder = "";
    
    // ----------------------------------------------------- Collections --//
    
    
    /**
     * List of <code>SelectionKey</code> event to register next time the 
     * <code>Selector</code> wakeups. This is needed since there a bug
     * in j2se 1.4.x that prevent registering selector event if the call
     * is done on another thread.
     */
    private ConcurrentLinkedQueue<SelectionKey> keysToEnable =
        new ConcurrentLinkedQueue<SelectionKey>();
         
    
    // ---------------------------------------------------- Object pools --//


    /**
     * <code>ConcurrentLinkedQueue</code> used as an object pool.
     * If the list becomes empty, new <code>ProcessorTask</code> will be
     * automatically added to the list.
     */
    protected ConcurrentLinkedQueue<ProcessorTask> processorTasks =
        new ConcurrentLinkedQueue<ProcessorTask>();
                                                                             
    /**
     * <code>ConcurrentLinkedQueue</code> used as an object pool.
     * If the list becomes empty, new <code>ReadTask</code> will be
     * automatically added to the list.
     */
    protected ConcurrentLinkedQueue<ReadTask> readTasks =
        new ConcurrentLinkedQueue<ReadTask>();

    
    /**
     * List of active <code>ProcessorTask</code>.
     */
    protected ConcurrentLinkedQueue<ProcessorTask> activeProcessorTasks =
        new ConcurrentLinkedQueue<ProcessorTask>();
    
    // -----------------------------------------  Multi-Selector supports --//

    /**
     * The number of <code>SelectorReadThread</code>
     */
    protected int selectorReadThreadsCount = 0;

    
    /**
     * The <code>Selector</code> used to register OP_READ
     */    
    protected SelectorReadThread[] readThreads;
    
    
    /**
     * The current <code>readThreads</code> used to process OP_READ.
     */
    int curReadThread;

    
    /**
     * The logger used by the grizzly classes.
     */
    protected static Logger logger = Logger.getLogger("GRIZZLY");
    
    
    // -----------------------------------------  Keep-Alive subsystems --//
    
     
    /**
     * Keep-Alive subsystem. If a client opens a socket but never close it,
     * the <code>SelectionKey</code> will stay forever in the 
     * <code>Selector</code> keys, and this will eventualy produce a 
     * memory leak.
     */
    protected KeepAlivePipeline keepAlivePipeline;
    
    
    // ------------------------------------------------- FileCache support --//
   
    
    /**
     * The FileCacheFactory associated with this Selector
     */ 
    protected FileCacheFactory fileCacheFactory;
    
        /**
     * Timeout before remove the static resource from the cache.
     */
    protected int secondsMaxAge = -1;
    
    
    /**
     * The maximum entries in the <code>fileCache</code>
     */
    protected int maxCacheEntries = 1024;
    
 
    /**
     * The maximum size of a cached resources.
     */
    protected long minEntrySize = 2048;
            
               
    /**
     * The maximum size of a cached resources.
     */
    protected long maxEntrySize = 537600;
    
    
    /**
     * The maximum cached bytes
     */
    protected long maxLargeFileCacheSize = 10485760;
 
    
    /**
     * The maximum cached bytes
     */
    protected long maxSmallFileCacheSize = 1048576;
    
    
    /**
     * Is the FileCache enabled.
     */
    protected boolean isFileCacheEnabled = true;
    
    
    /**
     * Is the large FileCache enabled.
     */
    protected boolean isLargeFileCacheEnabled = true;    
    
    // --------------------------------------------- Asynch supports -----//
    
    /**
     * Is asynchronous mode enabled?
     */
    protected boolean asyncExecution = false;
    
    
    /**
     * When the asynchronous mode is enabled, the execution of this object
     * will be delegated to the <code>AsyncHandler</code>
     */
    protected AsyncHandler asyncHandler;
    
    
    // ---------------------------------------------------- Constructor --//
    
    
    /**
     * Create the <code>Selector</code> object. Each instance of this class
     * will listen to a specific port.
     */
    public SelectorThread(){
    }
    
    // ------------------------------------------------------ Selector hook --/
    
    
   /**
     * Enable all registered interestOps. Due a a NIO bug, all interestOps
     * invokation needs to occurs on the same thread as the selector thread.
     */
    public void enableSelectionKeys(){
        SelectionKey selectionKey;
        int size = keysToEnable.size();
        long currentTime = (Long)System.currentTimeMillis();
        for (int i=0; i < size; i++) {
            selectionKey = keysToEnable.poll();

            selectionKey.interestOps(
                    selectionKey.interestOps() | SelectionKey.OP_READ);

            selectionKey.attach(currentTime);
            keepAlivePipeline.trap(selectionKey);   
        } 
    } 
    
    
    /**
     * Register a <code>SelectionKey</code> to this <code>Selector</code>
     * running of this thread.
     */
    public void registerKey(SelectionKey key){
        if ( key == null ) return;
        
        if (  keepAlivePipeline.dropConnection() ) {
            cancelKey(key);
            return;
        }
        
        // add SelectionKey & Op to list of Ops to enable
        keysToEnable.add(key);
        // tell the Selector Thread there's some ops to enable
        selector.wakeup();
        // wakeup() will force the SelectorThread to bail out
        // of select() to process your registered request
    } 

   // -------------------------------------------------------------- Init // 


    /**
     * initialized the endpoint by creating the <code>ServerScoketChannel</code>
     * and by initializing the server socket.
     */
    public void initEndpoint() throws IOException, InstantiationException {
        SelectorThreadConfig.configure(this);
        
        initFileCacheFactory();
        initPipeline();
        initAlgorithm();
        initMonitoringLevel();
        
        setName("SelectorThread-" + port);
        
        try{
                    
            // SSL is not yet supported by the VM
            if (secure){
                useNioNonBlocking = false; 
            } 
            
            if ( useNioNonBlocking ){
                // Create the socket listener
                serverSocketChannel = ServerSocketChannel.open();
                selector = Selector.open();

                serverSocket = serverSocketChannel.socket();
                serverSocket.setReuseAddress(true);
                if ( inet == null)
                    serverSocket.bind(new InetSocketAddress(port),ssBackLog);
                else
                    serverSocket.bind(new InetSocketAddress(inet,port),ssBackLog);

                serverSocketChannel.configureBlocking(false);
                serverSocketChannel.register(selector, SelectionKey.OP_ACCEPT);
            } else {
                if (inet == null) {
                    serverSocket = factory.createSocket(port,ssBackLog);
                } else {
                    serverSocket = factory.createSocket(port,ssBackLog,inet);
                }
                serverSocket.setReuseAddress(true);
            }
        } catch (SocketException ex){
            throw new BindException(ex.getMessage() + ": " + port);
        }
        
        serverSocket.setSoTimeout(serverTimeout);
        
        if (useNioNonBlocking){
            if ( selectorReadThreadsCount > 1 ){
                readThreads = new SelectorReadThread[selectorReadThreadsCount];
                initSelectorReadThread();
            } else {
                initProcessorTask(maxProcessorWorkerThreads);
                initReadTask(minReadQueueLength);                
            }
            SelectorFactory.maxSelectors = maxProcessorWorkerThreads;
        } else {
            initReadBlockingTask(maxProcessorWorkerThreads);
        }
        
        initialized = true;   
        
        if ( useNioNonBlocking){
            logger.log(Level.FINE,"Initializing Grizzly Non-Blocking Mode");                     
        } else if ( !useNioNonBlocking ){
            logger.log(Level.FINE,"Initializing Grizzly Blocking Mode");            
        } 
        
    }
     
    
    /**
     * Create a new <code>Pipeline</code> instance using the 
     * <code>pipelineClassName</code> value.
     */
    protected Pipeline newPipeline(int maxThreads,
                                   int minThreads,
                                   String name, 
                                   int port,
                                   int priority){
        
        Class className = null;                               
        Pipeline pipeline = null;                               
        try{                              
            className = Class.forName(pipelineClassName);
            pipeline = (Pipeline)className.newInstance();
        } catch (ClassNotFoundException ex){
            logger.log(Level.WARNING,
                       "Unable to load Pipeline: " + pipelineClassName);
            pipeline = new LinkedListPipeline();
        } catch (InstantiationException ex){
            logger.log(Level.WARNING,
                       "Unable to instantiate Pipeline: "
                       + pipelineClassName);
            pipeline = new LinkedListPipeline();
        } catch (IllegalAccessException ex){
            logger.log(Level.WARNING,
                       "Unable to instantiate Pipeline: "
                       + pipelineClassName);
            pipeline = new LinkedListPipeline();
        }
        
        if (logger.isLoggable(Level.FINE)){
            logger.log(Level.FINE,
                       "http-listener " + port + " uses pipeline: "
                       + pipeline.getClass().getName());
        }
        
        pipeline.setMaxThreads(maxThreads);
        pipeline.setMinThreads(minThreads);    
        pipeline.setName(name);
        pipeline.setPort(port);
        pipeline.setPriority(priority);
        pipeline.setQueueSizeInBytes(maxQueueSizeInBytes);
        pipeline.setThreadsIncrement(threadsIncrement);
        pipeline.setThreadsTimeout(threadsTimeout);
        
        return pipeline;
    }
    
    
    /**
     * Initialize the fileCacheFactory associated with this instance
     */
    private void initFileCacheFactory(){
        fileCacheFactory = FileCacheFactory.getFactory(port);
        fileCacheFactory.setIsEnabled(isFileCacheEnabled);
        fileCacheFactory.setLargeFileCacheEnabled(isLargeFileCacheEnabled);
        fileCacheFactory.setSecondsMaxAge(secondsMaxAge);
        fileCacheFactory.setMaxCacheEntries(maxCacheEntries);
        fileCacheFactory.setMinEntrySize(minEntrySize);
        fileCacheFactory.setMaxEntrySize(maxEntrySize);
        fileCacheFactory.setMaxLargeCacheSize(maxLargeFileCacheSize);
        fileCacheFactory.setMaxSmallCacheSize(maxSmallFileCacheSize);         
        fileCacheFactory.setIsMonitoringEnabled(isMonitoringEnabled);
    }
       
    
    /**
     * Injects <code>PipelineStatistic</code> into every
     * <code>Pipeline</code>, for monitoring purposes.
     */
    private void enablePipelineStats(){
        pipelineStat.start();

        processorPipeline.setPipelineStatistic(pipelineStat);       
        pipelineStat.setProcessorPipeline(processorPipeline);

        if (keepAlivePipeline != null){
            keepAlivePipeline.setKeepAliveStats(keepAliveStats);
        }
    }
    

    /**
     * Removes <code>PipelineStatistic</code> from every
     * <code>Pipeline</code>, when monitoring has been turned off.
     */
    private void disablePipelineStats(){
        pipelineStat.stop();
        
        processorPipeline.setPipelineStatistic(null);
        pipelineStat.setProcessorPipeline(null);

        if (keepAlivePipeline != null){
            keepAlivePipeline.setKeepAliveStats(null);
        }

    }

    
    /**
     * Load using reflection the <code>Algorithm</code> class.
     */
    protected void initAlgorithm(){
        try{                              
            algorithmClass = Class.forName(algorithmClassName);
            logger.log(Level.FINE,
                       "Using Algorithm: " + algorithmClassName);   
        } catch (ClassNotFoundException ex){
            logger.log(Level.FINE,
                       "Unable to load Algorithm: " + algorithmClassName);        
        }  finally {
            if ( algorithmClass == null ){
                algorithmClass = NoParsingAlgorithm.class;
            }
        }
    }
    
    
    /**
     * Initialize the keep-alive mechanism.
     */
    protected void initKeepAlivePipeline(){
        keepAlivePipeline = new KeepAlivePipeline();
        keepAlivePipeline.setMaxKeepAliveRequests(maxKeepAliveRequests);
        keepAlivePipeline
            .setKeepAliveTimeoutInSeconds(keepAliveTimeoutInSeconds);
        keepAlivePipeline.setPort(port);
        keepAlivePipeline.setThreadsTimeout(threadsTimeout);

        keepAliveStats.setMaxConnections(maxKeepAliveRequests);
        keepAliveStats.setSecondsTimeouts(keepAliveTimeoutInSeconds);        
    }
    
    
    /**
     * Init the <code>Pipeline</code>s used by the <code>WorkerThread</code>s.
     */
    protected void initPipeline(){        
        initKeepAlivePipeline();
        
        processorPipeline = newPipeline(maxProcessorWorkerThreads, 
                                        minWorkerThreads, "http",
                                        port,Thread.MAX_PRIORITY);  
        processorPipeline.initPipeline();

        // SSL needs at least two threads, and so re-use the default behaviour.
        if ( secure && maxReadWorkerThreads == 0){
            maxReadWorkerThreads = -1;
            logger.log(Level.WARNING,
                       "http-listener " + port + 
                       " is security-enabled and needs at least 2 threads");
        }
        
        // Only creates the pipeline if the max > 0
        if ( maxReadWorkerThreads > 0 ){                          
            readPipeline = newPipeline(maxReadWorkerThreads, 
                                       minWorkerThreads, "read", 
                                       port,Thread.NORM_PRIORITY);
            readPipeline.initPipeline();
        } else {
            readPipeline = (maxReadWorkerThreads == 0 ? null:processorPipeline);
        }
    }
    
    
    /**
     * Create a pool of <code>ReadTask</code>
     */
    protected void initReadTask(int size){         
        ReadTask task;
        for (int i=0; i < size; i++){
            task = newReadTask();             
            readTasks.offer(task);    
        }
    }
    
    
    /**
     * Return a new <code>ReadTask</code> instance
     */
    private ReadTask newReadTask(){
        StreamAlgorithm streamAlgorithm = null;
        
        try{
            streamAlgorithm = (StreamAlgorithm)algorithmClass.newInstance();
        } catch (InstantiationException ex){
            logger.log(Level.WARNING,
                       "Unable to instantiate Algorithm: "+ algorithmClassName);
        } catch (IllegalAccessException ex){
            logger.log(Level.WARNING,
                       "Unable to instantiate Algorithm: " + algorithmClassName);
        } finally {
            if ( streamAlgorithm == null)
                streamAlgorithm = new NoParsingAlgorithm();
        }       
        streamAlgorithm.setPort(port);
        
        ReadTask task;
        if ( maxReadWorkerThreads <= 0) {
            task = new ReadTask(streamAlgorithm, useDirectByteBuffer,
                                useByteBufferView);
        } else {
            task = new AsyncReadTask(streamAlgorithm, useDirectByteBuffer,
                                      useByteBufferView);
        }

        task.setPipeline(readPipeline);  
        task.setSelectorThread(this);
        task.setRecycle(recycleTasks);
        
        return task;
    }
    
    
    /**
     * Create a pool of <code>ReadBlockingTask</code>
     */
    private void initReadBlockingTask(int size){         
        for (int i=0; i < size; i++){
            readTasks.offer(newReadBlockingTask(false));
        }
    }

    
    /**
     * Create a <code>ReadBlockingTask</code> instance used with blocking
     * socket.
     */
    private ReadBlockingTask newReadBlockingTask(boolean initialize){
                                                        
        ReadBlockingTask task = new ReadBlockingTask();
        task.setSelectorThread(this);
        
        if (maxReadWorkerThreads > 0){
            task.setPipeline(readPipeline);
        }
 
        task.setRecycle(recycleTasks);
        task.attachProcessor(newProcessorTask(false,initialize));
        task.setPipelineStatistic(pipelineStat);
 
        return task;
    }
    

    /**
     * Initialize <code>SelectorReadThread</code> used to process
     * OP_READ operations.
     */
    private void initSelectorReadThread() throws IOException, 
                                                 InstantiationException {
        for (int i = 0; i < readThreads.length; i++) {
            readThreads[i] = new SelectorReadThread();
            readThreads[i].countName = i;
            readThreads[i].setMaxThreads(maxProcessorWorkerThreads);
            readThreads[i].setBufferSize(requestBufferSize);
            readThreads[i].setMaxKeepAliveRequests(maxKeepAliveRequests);
            readThreads[i]
                    .setKeepAliveTimeoutInSeconds(keepAliveTimeoutInSeconds);
            readThreads[i].maxQueueSizeInBytes = maxQueueSizeInBytes;
            readThreads[i].fileCacheFactory = fileCacheFactory;     
            readThreads[i].maxReadWorkerThreads = maxReadWorkerThreads;
            readThreads[i].defaultResponseType = defaultResponseType;
            readThreads[i].forcedResponseType = forcedResponseType;          
            readThreads[i].minReadQueueLength = minReadQueueLength;
            readThreads[i].maxHttpHeaderSize = maxHttpHeaderSize;
            
            if ( asyncExecution ) {
                readThreads[i].asyncExecution = asyncExecution;
                readThreads[i].asyncHandler = asyncHandler;
            }
            
            readThreads[i].threadsIncrement = threadsIncrement;
            readThreads[i].setPort(port);
            readThreads[i].setAdapter(adapter);
            readThreads[i].initEndpoint();
            readThreads[i].start();
        }
        curReadThread = 0;
    }


    /**
     * Return an instance of <code>SelectorReadThread</code> to use
     * for registering OP_READ
     */
    private synchronized SelectorReadThread getSelectorReadThread() {
        if (curReadThread == readThreads.length)
            curReadThread = 0;
        return readThreads[curReadThread++];
    }
    
    
    /**
     * Create a pool of <code>ProcessorTask</code>
     */
    protected void initProcessorTask(int size){
        for (int i=0; i < size; i++){           
            processorTasks.offer(newProcessorTask(useNioNonBlocking,false));
        }
    }  


    /**
     * Initialize <code>ProcessorTask</code>
     */
    protected void rampUpProcessorTask(){
        Iterator<ProcessorTask> iterator = processorTasks.iterator();
        while (iterator.hasNext()) {
            iterator.next().initialize();
        }
    }  
    

    /**
     * Create <code>ProcessorTask</code> objects and configure it to be ready
     * to proceed request.
     */
    protected ProcessorTask newProcessorTask(boolean useNioNonBlocking,
                                             boolean initialize){     
                                                 
        ProcessorTask task = new ProcessorTask(useNioNonBlocking, initialize);
        task.setAdapter(adapter);
        task.setMaxHttpHeaderSize(maxHttpHeaderSize);
        task.setBufferSize(requestBufferSize);
        task.setSelectorThread(this);               
        task.setRecycle(recycleTasks);
        task.setDefaultResponseType(defaultResponseType);
        task.setForcedResponseType(forcedResponseType);
        
        // Asynch extentions
        if ( asyncExecution ) {
            task.setEnableAsyncExecution(asyncExecution);
            task.setAsyncHandler(asyncHandler);
        }
              
        
        if (!useNioNonBlocking){
            task.setMaxKeepAliveRequests(maxKeepAliveRequests);
        }
                        
        if (secure){
            task.setSSLImplementation(sslImplementation);  
        } 
         
        if ( keepAlivePipeline.dropConnection() ) {
            task.setDropConnection(true);
        }
        
        task.setPipeline(processorPipeline);  
        return task;        
    }
 
 
    /**
     * Return a <code>ProcessorTask</code> from the pool. If the pool is empty,
     * create a new instance.
     */
     protected ProcessorTask getProcessorTask(){
        ProcessorTask processorTask = null;
        if (recycleTasks) {
            processorTask = processorTasks.poll();
        }
        
        if (processorTask == null){
            processorTask = newProcessorTask(true, false);
        } 
        
        if ( isMonitoringEnabled() ){
           activeProcessorTasks.offer(processorTask); 
        }

        
        return processorTask;
    }
        
    
    /**
     * Return a <code>ReadTask</code> from the pool. If the pool is empty,
     * create a new instance.
     */
    protected ReadTask getReadTask(SelectionKey key) throws IOException{
        ReadTask task = null;
        if ( recycleTasks ) {
            task = readTasks.poll();
        }
        
        if (task == null){
            task = newReadTask(); 
        }           

        task.setSelectionKey(key);
        return task;
    }
    
    
    /**
     * Return a <code>ReadBlockingTask</code> from the pool. 
     * If the pool is empty,
     * create a new instance.
     */
    protected ReadBlockingTask getReadBlockingTask(Socket socket){
        ReadBlockingTask task = null;
        if (recycleTasks) {
            task = (ReadBlockingTask)readTasks.poll();
        }
               
        if (task == null){
            task = newReadBlockingTask(false);
        }   
        
        ProcessorTask processorTask = task.getProcessorTask(); 
        processorTask.setSocket(socket);
        
        return task;
    }

    
    // --------------------------------------------------------- Thread run --/
    
    
    /**
     * Start the endpoint (this)
     */
    public void run(){
        try{
            startEndpoint();
        } catch (Exception ex){
            logger.log(Level.SEVERE,"selectorThread.errorOnRequest", ex);
        }
    }

    
    // ------------------------------------------------------------Start ----/
    
    
    /**
     * Start the Acceptor Thread and wait for incoming connection, in a non
     * blocking mode.
     */
    public void startEndpoint() throws IOException, InstantiationException {
        running = true;
        
        kaTimeout = keepAliveTimeoutInSeconds * 1000;
        rampUpProcessorTask();
        registerComponents();
 
        displayConfiguration();

        startPipelines();
        
        // (1) acceptSocket only if secure or blocking
        if (secure || !useNioNonBlocking){
            startBlockingMode();
        } else if (useNioNonBlocking){
            startNonBlockingMode();
        }               
    }
    
    
    /**
     * Starts the <code>Pipeline</code> used by this <code>Selector</code>
     */
    protected void startPipelines(){
        if (readPipeline != null){
            readPipeline.startPipeline();
        }

        processorPipeline.startPipeline();        
    }

    
    /**
     * Stop the <code>Pipeline</code> used by this <code>Selector</code>
     */
    protected void stopPipelines(){
        if ( keepAlivePipeline != null )
            keepAlivePipeline.stopPipeline();        

        if (readPipeline != null){
            readPipeline.stopPipeline();
        }
        
        processorPipeline.stopPipeline();

    }
    
    
    
    /**
     * Start a blocking server <code>Socket</code>
     */
    protected void startBlockingMode(){
        Socket socket = null;
        while (running){
            socket = acceptSocket();
            if (socket == null) {
                continue;
            }

            try {
                handleConnection(socket);
            } catch (Throwable ex) {
                logger.log(Level.FINE, 
                           "selectorThread.handleConnectionException",
                           ex);
                try {
                    socket.close();                   
                } catch (IOException ioe){
                    // Do nothing
                }
                continue;
            }
        }         
    }
    
    
    /**
     * Start a non blocking <code>Selector</code> object.
     */
    protected void startNonBlockingMode(){
        while (running) {
            doSelect();
        }       
    }
    
    
    /**
     * Execute a <code>Selector.select()</code> operation.
     */
    protected void doSelect(){
        SelectionKey key = null;
        Set readyKeys;
        Iterator<SelectionKey> iterator;
        int selectorState; 

        try{
            selectorState = 0;
            enableSelectionKeys();                

            try{
                selectorState = selector.select(selectorTimeout);
            } catch (CancelledKeyException ex){
                ;
            }

            readyKeys = selector.selectedKeys();
            iterator = readyKeys.iterator();
            while (iterator.hasNext()) {
                key = iterator.next();
                iterator.remove();
                key.attach(null);
                if (key.isValid()) {
                    handleConnection(key);
                } else {
                    cancelKey(key);
                }
            }
            
            expireIdleKeys();
            
            if (selectorState <= 0){
                selector.selectedKeys().clear();
                return;
            }
        } catch (Throwable t){
            if ( key != null ){
                key.attach(null);
                key.cancel();
            }
            logger.log(Level.FINE,"selectorThread.errorOnRequest",t);
        }
    }
    
    
    /**
     * Cancel keep-alive connections.
     */
    private void expireIdleKeys(){
        if ( keepAliveTimeoutInSeconds <= 0 || !selector.isOpen()) return;
        long current = System.currentTimeMillis();

        if (current < nextKeysExpiration) {
            return;
        }
        nextKeysExpiration = current + kaTimeout;
        
        Set<SelectionKey> readyKeys = selector.keys();
        if (readyKeys.isEmpty()){
            return;
        }
        Iterator<SelectionKey> iterator = readyKeys.iterator();
        SelectionKey key;
        while (iterator.hasNext()) {
            key = iterator.next();
            if ( !key.isValid() ) {
                keepAlivePipeline.untrap(key); 
                continue;
            }            
            // Keep-alive expired
            if ( key.attachment() != null) {
                long expire = (Long) key.attachment();
                if (current - expire >= kaTimeout) {
                    cancelKey(key);
                } else if (expire + kaTimeout < nextKeysExpiration){
                    nextKeysExpiration = expire + kaTimeout;
                }
            }
        }                    
    }  
    
    
    /**
     * Handle a blocking operation on the socket.
     */
    private void handleConnection(Socket socket) throws IOException{
                
        if (isMonitoringEnabled()) {
            globalRequestProcessor.increaseCountOpenConnections();
            pipelineStat.incrementTotalAcceptCount();
        }

        setSocketOptions(socket);
        getReadBlockingTask(socket).execute();
    }
   
    
    /**
     * Handle an incoming operation on the channel. It is always an ACCEPT or
     * a READ.
     */
    protected void handleConnection(SelectionKey key) throws IOException,
                                                           InterruptedException{
                                                    
        Task task = null;
        if ((key.readyOps() & SelectionKey.OP_ACCEPT) == SelectionKey.OP_ACCEPT){
            handleAccept(key);
            return;
        } else if ((key.readyOps() & SelectionKey.OP_READ) 
                                                    == SelectionKey.OP_READ) {
            task = handleRead(key);
        } 
 
        task.execute();
    }
      
    
    /**
     * Handle OP_ACCEPT
     */
    private void handleAccept(SelectionKey key) throws IOException{
        ServerSocketChannel server = (ServerSocketChannel) key.channel();
        SocketChannel channel = server.accept();

        if (channel != null) {
            if ( selectorReadThreadsCount > 1 ) {
                SelectorReadThread srt = getSelectorReadThread();
                srt.addChannel(channel);
            } else {
                channel.configureBlocking(false);
                SelectionKey readKey = 
                        channel.register(selector, SelectionKey.OP_READ);
                setSocketOptions(((SocketChannel)readKey.channel()).socket());
            }

            if (isMonitoringEnabled()) {
                getRequestGroupInfo().increaseCountOpenConnections();
                pipelineStat.incrementTotalAcceptCount();
            }
        }
     }
    
    
    /**
     * Handle OP_READ
     */ 
    private ReadTask handleRead(SelectionKey key) throws IOException{                   
        // disable OP_READ on key before doing anything else 
        key.interestOps(key.interestOps() & (~SelectionKey.OP_READ));
        ReadTask task = getReadTask(key);  
        
        return task;
    }
    
    
    /**
     * Cancel the current <code>SelectionKey</code>
     */
    protected void cancelKey(SelectionKey key){
        if (key == null || !key.isValid()){
            return;
        }
        
        keepAlivePipeline.untrap(key);
 
        try{
            ((SocketChannel)key.channel()).socket().shutdownInput();
        } catch (IOException ex){
            ;
        }
        
        try{
            ((SocketChannel)key.channel()).socket().shutdownOutput();
        } catch (IOException ex){
            ;
        }

        try{
            ((SocketChannel)key.channel()).socket().close();
        } catch (IOException ex){
            ;
        } finally {
            try{
                key.channel().close();
            } catch (IOException ex){
                logger.log(Level.FINEST,"selectorThread.unableToCloseKey", key);
            }
            if (isMonitoringEnabled()) {
                getRequestGroupInfo().decreaseCountOpenConnections();
            }
        }
               
        // Set the attachement to null so the Selector.java handleConnection
        // stop processing this key.
        key.attach(null);
        key.cancel();
        key = null;
    }
    
    
    /**
     * Returns the <code>Task</code> object to the pool.
     */
    public void returnTask(Task task){
        // Returns the object to the pool.
        if (task != null) {
            if (task.getType() == Task.PROCESSOR_TASK){
                                
                if ( isMonitoringEnabled() ){
                   activeProcessorTasks.remove(((ProcessorTask)task));
                }  
                
                processorTasks.offer((ProcessorTask)task);
            } else if (task.getType() == Task.READ_TASK){
                readTasks.offer((ReadTask)task);               
            }
        }
    }
    

    /**
     * Wakes up the <code>Selector</code> associated with this thread.
     */
    public void wakeup(){
        selector.wakeup();
    }

    
    /**
     * Clear all cached <code>Tasks</code> 
     */
    protected void clearTasks(){
        processorTasks.clear();
        readTasks.clear();
    }

    
    /**
     * Cancel the <code>threadID</code> execution. Return <code>true</code>
     * if it is successful.
     *
     * @param id the thread name to cancel
     */
    public boolean cancelThreadExecution(long cancelThreadID){
   
        if ( selectorReadThreadsCount > 1 ){
            boolean cancelled = false;
            for (SelectorReadThread readSelector : readThreads) {
                cancelled = readSelector.cancelThreadExecution(cancelThreadID);   
                if (cancelled) return true;
            }       
            return false;
        }
                       
        if (activeProcessorTasks.size() == 0) return false;
        
        Iterator<ProcessorTask> iterator = activeProcessorTasks.iterator();
        ProcessorTask processorTask;
        long threadID;
        while( iterator.hasNext() ){
            processorTask = iterator.next();
            threadID = processorTask.getRequest().getRequestProcessor()
                .getWorkerThreadID();
            if (threadID == cancelThreadID){
                processorTask.cancelTask("Request cancelled.","500");
                logger.log(Level.WARNING,
                        "Thread Request Cancelled: " + threadID);     
                return processorTask.getPipeline().interruptThread(threadID);
            }
        }
        return false;
    }

    
    // ---------------------------------------------------Endpoint Lifecycle --/
    
    
    public void pauseEndpoint() {
        if (running && !paused) {
            paused = true;
            unlockAccept();
        }
        
        try{
            selector.close();
        } catch (IOException ex){
            ; 
        }        
               
    }

    public void resumeEndpoint() {
        if (running) {
            paused = false;
        }
    }

    public void stopEndpoint() {
        try{
            if (running) running = false;

            stopPipelines();

            try{
                if ( serverSocket != null )
                    serverSocket.close();
            } catch (Throwable ex){
                logger.log(Level.SEVERE,
                        "selectorThread.closeSocketException",ex);
            }

            try{
                if ( serverSocketChannel != null)
                    serverSocketChannel.close();
            } catch (Throwable ex){
                logger.log(Level.SEVERE,
                        "selectorThread.closeSocketException",ex);
            }

            try{
                if ( selector != null)
                    selector.close();
            } catch (Throwable ex){
                logger.log(Level.SEVERE,
                        "selectorThread.closeSocketException",ex);
            }

            clearTasks();

            unregisterComponents();
        } catch (Throwable t){
             logger.log(Level.SEVERE,"selectorThread.stopException",t);
        }
    }
    
    protected void unlockAccept() {
        
        // Not required with non blocking.
        if (secure) {
            Socket s = null;
            try {
                // Need to create a connection to unlock the accept();
                if (inet == null) {
                    s=new Socket("127.0.0.1", port );
                }else{
                    s=new Socket(inet, port );
                        // setting soLinger to a small value will help shutdown the
                        // connection quicker
                    s.setSoLinger(true, 0);
                }
            } catch(Exception e) {
                logger.log(Level.FINE,"selectorThread.unlockAcceptException" + port
                          + " " + e.toString());
            } finally {
                if (s != null) {
                    try {
                        s.close();
                    } catch (Exception e) {
                        // Ignore
                    }
                }
            }
        }
    }

    
    // ------------------------------------------------------Public methods--/
    
    public boolean getUseNioNonBlocking(){
        return useNioNonBlocking;
    }
    

    public void setMaxThreads(int maxThreads) {
        if ( maxThreads == 1 ) {
            maxProcessorWorkerThreads = 5;
        } else {
            maxProcessorWorkerThreads = maxThreads;
        }
    }

    public int getMaxThreads() {
        return maxProcessorWorkerThreads;
    }

    public void setMaxSpareThreads(int maxThreads) {
    }

    public int getMaxSpareThreads() {
        return maxProcessorWorkerThreads;
    }

    public void setMinSpareThreads(int minSpareThreads) {
        this.minSpareThreads = minSpareThreads;
    }

    public int getMinSpareThreads() {
        return  minSpareThreads;
    }


    public int getPort() {
        return port;
    }

    public void setPort(int port ) {
        this.port=port;
    }

    public InetAddress getAddress() {
        return inet;
    }

    public void setAddress(InetAddress inet) {
        this.inet=inet;
    }

    public void setServerSocketFactory(  ServerSocketFactory factory ) {
        this.factory=factory;
    }

    ServerSocketFactory getServerSocketFactory() {
        return factory;
    }


    public boolean isRunning() {
        return running;
    }
    
    public boolean isPaused() {
        return paused;
    }


    /**
     * Provides the count of request threads that are currently
     * being processed by the container
     *
     * @return The count of busy threads 
     */
    public int getCurrentBusyProcessorThreads() {
        int busy = 0;
        
        // multi selector support
        if (selectorReadThreadsCount > 1) {
            for (SelectorReadThread readSelector : readThreads) {
                busy += readSelector.getCurrentBusyProcessorThreads();   
            }

        } else {
            busy = processorPipeline.getCurrentThreadsBusy();
        }

        return busy;  
    }


    /**
     * Sets the timeout in ms of the server sockets created by this
     * server. This method allows the developer to make servers
     * more or less responsive to having their server sockets
     * shut down.
     *
     * <p>By default this value is 1000ms.
     */
    public void setServerTimeout(int timeout) {
        this.serverTimeout = timeout;
    }

    public boolean getTcpNoDelay() {
        return tcpNoDelay;
    }
    
    public void setTcpNoDelay( boolean b ) {
        tcpNoDelay=b;
    }

    public int getSoLinger() {
        return linger;
    }
    
    public void setSoLinger( int i ) {
        linger=i;
    }

    public int getSoTimeout() {
        return socketTimeout;
    }
    
    public void setSoTimeout( int i ) {
        socketTimeout=i;
    }
    
    public int getServerSoTimeout() {
        return serverTimeout;
    }  
    
    public void setServerSoTimeout( int i ) {
        serverTimeout=i;
    }
    
    public void setSecure(boolean secure){
        this.secure = secure;
    }
    
    public boolean getSecure(){
        return secure;
    }
        
    // ------------------------------------------------------ Connector Methods


    /**
     * Get the maximum pending connection this <code>Pipeline</code>
     * can handle.
     */
    public int getQueueSizeInBytes(){
        return maxQueueSizeInBytes;
    }
    
    
    
    public int getMaxKeepAliveRequests() {
        return maxKeepAliveRequests;
    }
    
    
    /** 
     * Set the maximum number of Keep-Alive requests that we will honor.
     */
    public void setMaxKeepAliveRequests(int mkar) {
        maxKeepAliveRequests = mkar;
    }
    

    /** 
     * Sets the number of seconds before a keep-alive connection that has
     * been idle times out and is closed.
     *
     * @param timeout Keep-alive timeout in number of seconds
     */    
    public void setKeepAliveTimeoutInSeconds(int timeout) {
        keepAliveTimeoutInSeconds = timeout;
        keepAliveStats.setSecondsTimeouts(timeout);
    }


    /** 
     * Gets the number of seconds before a keep-alive connection that has
     * been idle times out and is closed.
     *
     * @return Keep-alive timeout in number of seconds
     */    
    public int getKeepAliveTimeoutInSeconds() {
        return keepAliveTimeoutInSeconds;
    }


    /** 
     * Sets the number of keep-alive threads.
     *
     * @param threadCount Number of keep-alive threads
     */    
    public void setKeepAliveThreadCount(int threadCount) {
        keepAlivePipeline.setMaxThreads(threadCount);
    }


    /**
     * Return the current <code>SSLImplementation</code> this Thread
     */
    public SSLImplementation getSSLImplementation() {
        return sslImplementation;
    }
    
    
    /**
     * Set the <code>SSLImplementation</code> used by this thread.It usually
     * means HTTPS will be used.
     */
    public void setSSLImplementation( SSLImplementation sslImplementation) {
        this.sslImplementation = sslImplementation;
    }


    /**
     * Set the associated adapter.
     * 
     * @param adapter the new adapter
     */
    public void setAdapter(Adapter adapter) {
        this.adapter = adapter;
    }


    /**
     * Get the associated adapter.
     * 
     * @return the associated adapter
     */
    public Adapter getAdapter() {
        return adapter;
    }
    
    // ------------------------------------------------------- SSL supports --/
    
    protected Socket acceptSocket() {
        if( !running || serverSocket==null ) return null;

        Socket socket = null;

    	try {
            if(factory==null) {
                socket = serverSocketChannel.accept().socket();
            } else {
                socket = factory.acceptSocket(serverSocket);
            }
            if (null == socket) {
                logger.log(Level.WARNING,"selectorThread.acceptSocket");
            } else {
                if (!running) {
                    socket.close();  // rude, but unlikely!
                    socket = null;
                } else if (factory != null) {
                    factory.initSocket( socket );
                }
            }
        } catch(InterruptedIOException iioe) {
            // normal part -- should happen regularly so
            // that the endpoint can release if the server
            // is shutdown.
        } catch (AccessControlException ace) {
            // When using the Java SecurityManager this exception
            // can be thrown if you are restricting access to the
            // socket with SocketPermission's.
            // Log the unauthorized access and continue
            logger.log(Level.WARNING,"selectorThread.wrongPermission",
                       new Object[]{serverSocket,ace});
        } catch (IOException e) {

            String msg = null;

            if (running) {
                logger.log(Level.SEVERE,"selectorThread.shutdownException", 
                           new Object[]{serverSocket, e});
            }

            if (socket != null) {
                try {
                    socket.close();
                } catch(Throwable ex) {
                    logger.log(Level.SEVERE,"selectorThread.shutdownException", 
                               new Object[]{serverSocket, ex});
                }
                socket = null;
            }

            if( !running ) return null;
        } catch (Throwable t) {                                
            try{
                if (socket != null)
                    socket.close();
            } catch (IOException ex){
                ;
            }
            logger.log(Level.FINE,
                       "selectorThread.errorOnRequest",
                       t);
        }

        return socket;
    }  
    
   
    protected void setSocketOptions(Socket socket) throws SocketException {
        try{
            if(linger >= 0 ) {
            socket.setSoLinger( true, linger);
            }
        } catch (SocketException ex){
            logger.log(Level.WARNING,
                    "setSoLinger exception ",ex);
        }
        try{
        if( tcpNoDelay )
            socket.setTcpNoDelay(tcpNoDelay);
        } catch (SocketException ex){
            logger.log(Level.WARNING,
                    "setTcpNoDelay exception ",ex);
        }
        try{
        if( keepAliveTimeoutInSeconds > 0 && !useNioNonBlocking)
            socket.setSoTimeout( keepAliveTimeoutInSeconds * 1000 );
        } catch (SocketException ex){
            logger.log(Level.WARNING,
                    "setSoTimeout exception ",ex);
        }
        try{
        if ( maxReadWorkerThreads != 0)
            socket.setReuseAddress(true);
        } catch (SocketException ex){
            logger.log(Level.WARNING,
                    "setReuseAddress exception ",ex);
    }
    }
    
    // ------------------------------- JMX and Monnitoring support --------//
    
    public ObjectName getObjectName() {
        return oname;
    }

    public String getDomain() {
        return domain;
    }

    public ObjectName preRegister(MBeanServer server,
                                  ObjectName name) throws Exception {
        oname=name;
        mserver=server;
        domain=name.getDomain();
        return name;
    }

    public void postRegister(Boolean registrationDone) {
        // Do nothing
    }

    public void preDeregister() throws Exception {
        // Do nothing
    }

    public void postDeregister() {
        // Do nothing
    }  


    /**
     * Register JMX components.
     */
    private void registerComponents(){

        if( this.domain != null) {

            Registry reg = Registry.getRegistry();

            try {
                globalRequestProcessorName = new ObjectName(
                    domain + ":type=GlobalRequestProcessor,name=http" + port);
                reg.registerComponent(globalRequestProcessor,
                                      globalRequestProcessorName,
                                      null);
 
                keepAliveMbeanName = new ObjectName(
                    domain + ":type=PWCKeepAlive,name=http" + port);
                reg.registerComponent(keepAliveStats,
                                      keepAliveMbeanName,
                                      null);

                pwcConnectionQueueMbeanName = new ObjectName(
                    domain + ":type=PWCConnectionQueue,name=http" + port);
                reg.registerComponent(pipelineStat,
                                      pwcConnectionQueueMbeanName,
                                      null);
                
                pwcFileCacheMbeanName = new ObjectName(
                    domain + ":type=PWCFileCache,name=http" + port);
                reg.registerComponent(fileCacheFactory,
                                      pwcFileCacheMbeanName,
                                      null);                
            } catch (Exception ex) {
                logger.log(Level.WARNING,
                           "selectorThread.mbeanRegistrationException",
                           new Object[]{new Integer(port),ex});
            }
        }

    }
    
    
    /**
     * Unregister components.
     **/
    private void unregisterComponents(){

        if (this.domain != null) {

            Registry reg = Registry.getRegistry();

            try {
                if (globalRequestProcessorName != null) {
                    reg.unregisterComponent(globalRequestProcessorName);
                }
                if (keepAliveMbeanName != null) {
                    reg.unregisterComponent(keepAliveMbeanName);
                }
                if (pwcConnectionQueueMbeanName != null) {
                    reg.unregisterComponent(pwcConnectionQueueMbeanName);
                }
                if (pwcFileCacheMbeanName != null) {
                    reg.unregisterComponent(pwcFileCacheMbeanName);
                }                    
            } catch (Exception ex) {
                logger.log(Level.WARNING,
                           "mbeanDeregistrationException",
                           new Object[]{new Integer(port),ex});
            }
        }
    }

    
    /**
     * Enable gathering of monitoring datas.
     */
    public void enableMonitoring(){
        isMonitoringEnabled = true;
        enablePipelineStats();      
        
       fileCacheFactory.setIsMonitoringEnabled(isMonitoringEnabled);
    }
    
    
    /**
     * Disable gathering of monitoring datas. 
     */
    public void disableMonitoring(){
        disablePipelineStats();  
        
       fileCacheFactory.setIsMonitoringEnabled(isMonitoringEnabled);        
    }

    
    /**
     * Returns <code>true</code> if monitoring has been enabled, 
     * <code>false</code> otherwise.
     */
    public boolean isMonitoringEnabled() {
        return isMonitoringEnabled;
    }

    
    public RequestGroupInfo getRequestGroupInfo() {
        return globalRequestProcessor;
    }


    public KeepAliveStats getKeepAliveStats() {
        return keepAliveStats;
    }


    /*
     * Initializes the web container monitoring level from the domain.xml.
     */
    private void initMonitoringLevel() {
        pipelineStat = new PipelineStatistic(port);
        pipelineStat.setQueueSizeInBytes(maxQueueSizeInBytes);
    }
 
    
    public int getMaxHttpHeaderSize() {
        return maxHttpHeaderSize;
    }
    
    public void setMaxHttpHeaderSize(int maxHttpHeaderSize) {
        this.maxHttpHeaderSize = maxHttpHeaderSize;
    }
    
        
    /**
     * The minimun threads created at startup.
     */ 
    public void setMinThreads(int minWorkerThreads){
        this.minWorkerThreads = minWorkerThreads;
    }


    /**
     * Set the request input buffer size
     */
    public void setBufferSize(int requestBufferSize){
        this.requestBufferSize = requestBufferSize;
    }
    

    /**
     * Return the request input buffer size
     */
    public int getBufferSize(){
        return requestBufferSize;
    }
    
    
    public Selector getSelector(){
        return selector;
    }

    /************************* PWCThreadPool Stats *************************/

    public int getCountThreadsStats() {

        int ret = processorPipeline.getCurrentThreadCount();

        if (readPipeline != null
                && readPipeline != processorPipeline) {
            ret += readPipeline.getCurrentThreadCount();
        }

        return ret;
    }


    public int getCountThreadsIdleStats() {

        int ret = processorPipeline.getWaitingThread();

        if (readPipeline != null
                && readPipeline != processorPipeline) {
            ret += readPipeline.getWaitingThread();
        }

        return ret;
    }


    /************************* HTTPListener Stats *************************/

    public int getCurrentThreadCountStats() {

        int ret = processorPipeline.getCurrentThreadCount();

        if (readPipeline != null
                && readPipeline != processorPipeline) {
            ret += readPipeline.getCurrentThreadCount();
        }

        return ret;
    }


    public int getCurrentThreadsBusyStats() {

        int ret = processorPipeline.getCurrentThreadsBusy();

        if (readPipeline != null
                && readPipeline != processorPipeline) {
            ret += readPipeline.getCurrentThreadsBusy();
        }
 
        return ret;
    }

    public int getMaxSpareThreadsStats() {

        int ret = processorPipeline.getMaxSpareThreads();
 
        if (readPipeline != null
                && readPipeline != processorPipeline) {
            ret += readPipeline.getMaxSpareThreads();
        }

        return ret;
    }


    public int getMinSpareThreadsStats() {

        int ret = processorPipeline.getMinSpareThreads();

        if (readPipeline != null
                && readPipeline != processorPipeline) {
            ret += readPipeline.getMinSpareThreads();
        }

        return ret;
    }


    public int getMaxThreadsStats() {

        int ret = processorPipeline.getMaxThreads();

        if (readPipeline != null
                && readPipeline != processorPipeline) {
            ret += readPipeline.getMaxThreads();
        }
        
        return ret;
    }


    //------------------------------------------------- FileCache config -----/

   
    /**
     * The timeout in seconds before remove a <code>FileCacheEntry</code>
     * from the <code>fileCache</code>
     */
    public void setSecondsMaxAge(int sMaxAges){
        secondsMaxAge = sMaxAges;
    }
    
    
    /**
     * Set the maximum entries this cache can contains.
     */
    public void setMaxCacheEntries(int mEntries){
        maxCacheEntries = mEntries;
    }

    
    /**
     * Return the maximum entries this cache can contains.
     */    
    public int getMaxCacheEntries(){
        return maxCacheEntries;
    }
    
    
    /**
     * Set the maximum size a <code>FileCacheEntry</code> can have.
     */
    public void setMinEntrySize(long mSize){
        minEntrySize = mSize;
    }
    
    
    /**
     * Get the maximum size a <code>FileCacheEntry</code> can have.
     */
    public long getMinEntrySize(){
        return minEntrySize;
    }
     
    
    /**
     * Set the maximum size a <code>FileCacheEntry</code> can have.
     */
    public void setMaxEntrySize(long mEntrySize){
        maxEntrySize = mEntrySize;
    }
    
    
    /**
     * Get the maximum size a <code>FileCacheEntry</code> can have.
     */
    public long getMaxEntrySize(){
        return maxEntrySize;
    }
    
    
    /**
     * Set the maximum cache size
     */ 
    public void setMaxLargeCacheSize(long mCacheSize){
        maxLargeFileCacheSize = mCacheSize;
    }

    
    /**
     * Get the maximum cache size
     */ 
    public long getMaxLargeCacheSize(){
        return maxLargeFileCacheSize;
    }
    
    
    /**
     * Set the maximum cache size
     */ 
    public void setMaxSmallCacheSize(long mCacheSize){
        maxSmallFileCacheSize = mCacheSize;
    }
    
    
    /**
     * Get the maximum cache size
     */ 
    public long getMaxSmallCacheSize(){
        return maxSmallFileCacheSize;
    }    

    
    /**
     * Is the fileCache enabled.
     */
    public boolean isFileCacheEnabled(){
        return isFileCacheEnabled;
    }

    
    /**
     * Is the file caching mechanism enabled.
     */
    public void setFileCacheIsEnabled(boolean isFileCacheEnabled){
        this.isFileCacheEnabled = isFileCacheEnabled;
    }
   
    
    /**
     * Is the large file cache support enabled.
     */
    public void setLargeFileCacheEnabled(boolean isLargeEnabled){
        this.isLargeFileCacheEnabled = isLargeEnabled;
    }
   
    
    /**
     * Is the large file cache support enabled.
     */
    public boolean getLargeFileCacheEnabled(){
        return isLargeFileCacheEnabled;
    }    

    // --------------------------------------------------------------------//
    
    /**
     * Enable the <code>AsyncHandler</code> used when asynchronous
     */
    public void setEnableAsyncExecution(boolean asyncExecution){
        this.asyncExecution = asyncExecution;     
    }
    
       
    /**
     * Return true when asynchronous execution is 
     * enabled.
     */    
    public boolean getEnableAsyncExecution(){
        return asyncExecution;
    }
    
    
    /**
     * Set the <code>AsyncHandler</code> used when asynchronous execution is 
     * enabled.
     */
    public void setAsyncHandler(AsyncHandler asyncHandler){
        this.asyncHandler = asyncHandler;     
    }
    
       
    /**
     * Return the <code>AsyncHandler</code> used when asynchronous execution is 
     * enabled.
     */    
    public AsyncHandler getAsyncHandler(){
        return asyncHandler;
    }
    
    
    /**
     * Set the logger used by this instance.
     */
    public static void setLogger(Logger l){
        if ( l != null )
            logger = l;
    }

    
    /**
     * Return the logger used by the Grizzly classes.
     */
    public static Logger logger(){
        return logger;
    }
    
    
    /**
     * Set the document root folder
     */
    public static void setWebAppRootPath(String rf){
        rootFolder = rf;
    }
    
    
    /**
     * Return the folder's root where application are deployed.
     */
    public static String getWebAppRootPath(){
        return rootFolder;
    }
    
    
    /**
     * Display the Grizzly configuration parameters.
     */
    private void displayConfiguration(){
       if (displayConfiguration){
            logger.log(Level.INFO,
                    "\n Grizzly configuration for http-listener " 
                    + port 
                    + "\n\t useNioNonBlocking:" 
                    + useNioNonBlocking + "\n\t minReadQueueLength:" 
                    + minReadQueueLength + "\n\t minProcessorQueueLength:"  
                    + maxReadWorkerThreads+"\n\t maxProcessorWorkerThreads: " 
                    + maxProcessorWorkerThreads + "\n\t minWorkerThreads:" 
                    + minWorkerThreads + "\n\t selectorTimeout:" 
                    + selectorTimeout + "\n\t ByteBuffer size: " 
                    + Constants.CHANNEL_BYTE_SIZE 
                    + "\n\t maxHttpHeaderSize:" 
                    + maxHttpHeaderSize
                    + "\n\t maxKeepAliveRequests: "
                    + maxKeepAliveRequests
                    + "\n\t keepAliveTimeoutInSeconds: "
                    + keepAliveTimeoutInSeconds
                    + "\n\t useDirectByteBuffer: "
                    + useDirectByteBuffer       
                    + "\n\t socketSoTimeout: "
                    + Constants.DEFAULT_CONNECTION_TIMEOUT                  
                    + "\n\t useByteBufferView: "                        
                    + useByteBufferView
                    + "\n\t selectorReadThreadsCount: "                        
                    + selectorReadThreadsCount
                    + "\n\t recycleTasks: "                        
                    + recycleTasks);
        }
    }
}




