/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */
package com.sun.enterprise.web.connector.grizzly;

import java.util.LinkedList;
import java.util.NoSuchElementException;

/**
 * Internal FIFO used by the Worker Threads to pass information
 * between <code>Task</code> objects.
 *
 * @author Jean-Francois Arcand
 */
public class LinkedListPipeline extends LinkedList implements Pipeline{


    /**
     * The number of thread waiting for a <code>Task</code>
     */
    protected int waitingThreads = 0;
    
    
    /**
     * The maximum number of Thread
     */
    protected int maxThreads = 20;
    

    /**
     * The minimum numbers of <code>WorkerThread</code>
     */
    protected int minThreads = 5;
                                                                                
    /**
     * The minimum numbers of spare <code>WorkerThread</code>
     */
    protected int minSpareThreads = 2;


    /**
     * The port used.
     */
    protected int port = 8080;
    

    /**
     * The number of <code>??WorkerThread</code>
     */
    protected int threadCount = 0;
    

    /**
     * The name of this Pipeline
     */
    protected String name;
    
    
    /**
     * The Thread Priority
     */
    protected int priority = Thread.NORM_PRIORITY;
    
    
    /**
     * Has the pipeline already started
     */
    protected boolean isStarted = false; 
    

    /**
     * <code>WorkerThread</code> amanged by this pipeline.
     */
    protected WorkerThread[] workerThreads;
    
    
    /**
     * Maximum pending connection before refusing requests.
     */
    protected int maxQueueSizeInBytes = -1;
    
    
    /**
     * The increment number used when adding new thread.
     */
    protected int threadsIncrement = 1;
    
    
    /**
     * The request times out during transaction.
     */
    protected int threadsTimeout = Constants.DEFAULT_TIMEOUT;
        
    
    /**
     * The <code>PipelineStatistic</code> objects used when gathering statistics.
     */
    protected PipelineStatistic pipelineStat;
    
    // ------------------------------------------------------- Constructor -----/
    
    public LinkedListPipeline(){
        super();
    }
    
    public LinkedListPipeline(int maxThreads, int minThreads, String name, 
                              int port, int priority){
                        
        this.maxThreads = maxThreads;
        this.port = port;
        this.name = name;
        this.minThreads = minThreads;
        this.priority = priority;
        
        if ( minThreads < minSpareThreads )
            minSpareThreads = minThreads;
        
    }

    
    public LinkedListPipeline(int maxThreads, int minThreads, String name, 
                              int port){
                        
        this(maxThreads,minThreads,name,port,Thread.NORM_PRIORITY);
    }

   
    // ------------------------------------------------ Lifecycle ------------/
    
    /**
     * Init the <code>Pipeline</code> by initializing the required
     * <code>WorkerThread</code>. Default value is 10
     */
    public void initPipeline(){
        
        if (minThreads > maxThreads) {
            minThreads = maxThreads;
        }
        
        workerThreads = new WorkerThread[maxThreads];
        increaseWorkerThread(minThreads, false);
        
   }

    
    /**
     * Start the <code>Pipeline</code> and all associated 
     * <code>WorkerThread</code>
     */
    public void startPipeline(){
        if (!isStarted) {
            for (int i=0; i < minThreads; i++){
                workerThreads[i].start();
            }
            isStarted = true;
        }
    }
    

    /**
     * Stop the <code>Pipeline</code> and all associated
     * <code>WorkerThread</code>
     */
    public void stopPipeline(){
        if (isStarted) {
            for (int i=0; i < threadCount; i++){
                workerThreads[i].terminate();
            }
            isStarted = false;
        }
    }


    /**
     * Create new <code>WorkerThread</code>
     */
    protected void increaseWorkerThread(int increment, boolean startThread){        
        WorkerThread workerThread;
        int currentCount = threadCount;
        int increaseCount = threadCount + increment; 
        for (int i=currentCount; i < increaseCount; i++){
            workerThread = new WorkerThread(this, 
                    name + "WorkerThread-"  + port + "-" + i);
            workerThread.setPriority(priority);
            
            if (startThread)
                workerThread.start();
            
            workerThreads[i] = workerThread;
            threadCount++; 
        }
    }
    
    
    /**
     * Interrupt the <code>Thread</code> using it thread id
     */
    public boolean interruptThread(long threadID){
        ThreadGroup threadGroup = workerThreads[0].getThreadGroup();
        Thread[] threads = new Thread[threadGroup.activeCount()];
        threadGroup.enumerate(threads);
        
        for (Thread thread: threads){
            if ( thread != null && thread.getId() == threadID ){                
                if ( Thread.State.RUNNABLE != thread.getState()){
                    try{
                        thread.interrupt();
                        return true;
                    } catch (Throwable t){
                        ; // Swallow any exceptions.
                    }
                }
            }
        }
        return false;
    }
    
    
    // ---------------------------------------------------- Queue ------------//
  
    
    /**
     * Add an object to this pipeline
     */
    public synchronized void addTask(Task task) {
        
        boolean rejected = false;
        int queueSize =  size();
        if ( maxQueueSizeInBytes != -1 && maxQueueSizeInBytes < queueSize){
            task.cancelTask("Maximum Connections Reached: " 
                + maxQueueSizeInBytes + " -- Retry later", 
                    HtmlHelper.OK);
            task.getSelectorThread().returnTask(task);           
            return;
        }
        
        addLast(task);
        notify();

        // Create worker threads if we know we will run out of them
        if (threadCount < maxThreads && waitingThreads < queueSize ){
            int left = maxThreads - threadCount;
            if (threadsIncrement > left){
                threadsIncrement = left;
            }
            increaseWorkerThread(threadsIncrement,true);
        }
    }


    /**
     * Return a <code>Task</code> object available in the pipeline.
     * All Threads will synchronize on that method
     */
    public synchronized Task getTask() {
        if ( isEmpty() ) {
            
            try { 
                waitingThreads++; 
                wait();

            }  catch (InterruptedException e)  {
                Thread.interrupted();
            }
            waitingThreads--;
       
        }
        try {
            return (Task)removeFirst();
        } catch(NoSuchElementException e) {
            return null;
        } finally {
            if ( pipelineStat != null) {
                pipelineStat.gather(size());
            }       
        }
    }

    
    /**
     * Return <code>true</code> if the size of this <code>ArrayList</code>
     * minus the current waiting threads is lower than zero.
     */
    public boolean isEmpty() {
        return  (size() - waitingThreads <= 0);
    }
    
    // --------------------------------------------------Properties ----------//

    /**
     * Return the number of waiting threads.
     */
    public int getWaitingThread(){
        return waitingThreads;
    }
    
    
    /** 
     * Set the number of threads used by this pipeline.
     */
    public void setMaxThreads(int maxThreads){
        this.maxThreads = maxThreads;
    }    
    
    
    /** 
     * Return the number of threads used by this pipeline.
     */
    public int getMaxThreads(){
        return maxThreads;
    }
    
    
    public int getCurrentThreadCount() {
        return threadCount;
    }
      
      
    /**
     * Return the curent number of threads that are currently processing 
     * a task.
     */
    public int getCurrentThreadsBusy(){
        return (threadCount - waitingThreads);
    }
        

    /**
     * Return the maximum spare thread.
     */
    public int getMaxSpareThreads() {
        return maxThreads;
    }


    /**
     * Return the minimum spare thread.
     */
    public int getMinSpareThreads() {
        return minSpareThreads;
    }


    /**
     * Set the minimum space thread this <code>Pipeline</code> can handle.
     */
    public void setMinSpareThreads(int minSpareThreads) {
        this.minSpareThreads = minSpareThreads;
    }

    
    /**
     * Set the thread priority of the <code>Pipeline</code>
     */
    public void setPriority(int priority){
        this.priority = priority;
    }
    
    
    /**
     * Set the name of this <code>Pipeline</code>
     */
    public void setName(String name){
        this.name = name;
    }
    
    
    /**
     * Return the name of this <code>Pipeline</code>
     * @return the name of this <code>Pipeline</code>
     */
    public String getName(){
        return name+port;
    }    

    
    /**
     * Set the port used by this <code>Pipeline</code>
     * @param port the port used by this <code>Pipeline</code>
     */
    public void setPort(int port){
        this.port = port;
    }
    
    
    /**
     * Set the minimum thread this <code>Pipeline</code> will creates
     * when initializing.
     * @param minThreads the minimum number of threads.
     */
    public void setMinThreads(int minThreads){
        this.minThreads = minThreads;
    }
    
    
    public String toString(){
       return "name: " + name + " maxThreads: " + maxThreads 
                + " minThreads:" + minThreads;        
    }
    
    
    /**
     * Set the number the <code>Pipeline</code> will use when increasing the 
     * thread pool
     */    
    public void setThreadsIncrement(int threadsIncrement){
        this.threadsIncrement = threadsIncrement;
    }
    
    
    /**
     * Set the timeout value a thread will use to times out the request.
     */   
    public void setThreadsTimeout(int threadsTimeout){
        this.threadsTimeout = threadsTimeout;
    }
    
    
    /** 
     * The number of <code>Task</code> currently queued
     * @return number of queued connections
     */
    public int getTaskQueuedCount(){
       return size();
    }

    
    /**
     * Set the maximum pending connection this <code>Pipeline</code>
     * can handle.
     */
    public void setQueueSizeInBytes(int maxQueueSizeInBytesCount){
        this.maxQueueSizeInBytes = maxQueueSizeInBytesCount;
    }
    
    
    /**
     * Get the maximum pending connection this <code>Pipeline</code>
     * can handle.
     */
    public int getQueueSizeInBytes(){
        return maxQueueSizeInBytes;
    }   
  
    
    /**
     * Set the <code>PipelineStatistic</code> object used
     * to gather statistic;
     */
    public void setPipelineStatistic(PipelineStatistic pipelineStatistic){
        this.pipelineStat = pipelineStatistic;
    }
    
    
    /**
     * Return the <code>PipelineStatistic</code> object used
     * to gather statistic;
     */
    public PipelineStatistic getPipelineStatistic(){
        return pipelineStat;
    }
    
}
